<?php
require_once '../config/database.php';
require_once '../config/auth_enhanced.php';

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized access']);
    exit;
}

header('Content-Type: application/json');

try {
    $database = new Database();
    $pdo = $database->getConnection();
    
    $start_date = $_GET['start_date'] ?? date('Y-m-01');
    $end_date = $_GET['end_date'] ?? date('Y-m-t');
    
    // Get revenue data by day for the date range using new debit/credit model
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(transaction_date, '%Y-%m-%d') as date,
            SUM(CASE WHEN transaction_type = 'credit' THEN amount ELSE 0 END) as credit,
            SUM(CASE WHEN transaction_type = 'debit' THEN amount ELSE 0 END) as debit,
            SUM(CASE WHEN transaction_type = 'credit' THEN amount ELSE -amount END) as net_revenue
        FROM financial_transactions 
        WHERE transaction_date BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(transaction_date, '%Y-%m-%d')
        ORDER BY date ASC
    ");
    $stmt->execute([$start_date, $end_date]);
    $revenue_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format data for Chart.js
    $labels = [];
    $credit_data = [];
    $debit_data = [];
    $net_data = [];
    
    foreach ($revenue_data as $row) {
        $labels[] = date('M d', strtotime($row['date']));
        $credit_data[] = (float)$row['credit'];
        $debit_data[] = (float)$row['debit'];
        $net_data[] = (float)$row['net_revenue'];
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'labels' => $labels,
            'values' => $net_data,
            'credit_values' => $credit_data,
            'debit_values' => $debit_data
        ]
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error fetching revenue data: ' . $e->getMessage()
    ]);
}
?>
    
    echo json_encode([
        'success' => true,
        'data' => [
            'labels' => $labels,
            'values' => $net_data
        ],
        'chart_data' => [
            'labels' => $labels,
            'datasets' => [
                [
                    'label' => 'Income',
                    'data' => $income_data,
                    'backgroundColor' => 'rgba(40, 167, 69, 0.6)',
                    'borderColor' => 'rgba(40, 167, 69, 1)',
                    'borderWidth' => 2
                ],
                [
                    'label' => 'Expenses',
                    'data' => $expense_data,
                    'backgroundColor' => 'rgba(220, 53, 69, 0.6)',
                    'borderColor' => 'rgba(220, 53, 69, 1)',
                    'borderWidth' => 2
                ],
                [
                    'label' => 'Net Revenue',
                    'data' => $net_data,
                    'backgroundColor' => 'rgba(0, 123, 255, 0.6)',
                    'borderColor' => 'rgba(0, 123, 255, 1)',
                    'borderWidth' => 2
                ]
            ]
        ]
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}
?>
