<?php
require_once __DIR__ . '/../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    // Get the requested membership type (default to Individual)
    $membershipType = isset($_GET['type']) ? $_GET['type'] : 'Individual';
    
    // Get the current year
    $currentYear = date('Y');
    
    // Query to get the fee for the specified membership type
    $query = "SELECT * FROM membership_fee_settings 
              WHERE membership_type = ? 
              AND session_year = ? 
              AND is_active = 1";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$membershipType, $currentYear]);
    
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // If no fee found for current year, try with any year
    if (!$result) {
        $query = "SELECT * FROM membership_fee_settings 
                  WHERE membership_type = ? 
                  AND is_active = 1 
                  ORDER BY session_year DESC 
                  LIMIT 1";
        
        $stmt = $pdo->prepare($query);
        $stmt->execute([$membershipType]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    // If still no fee found, set default values
    if (!$result) {
        $result = [
            'membership_type' => $membershipType,
            'session_year' => $currentYear,
            'fee_amount' => $membershipType === 'Individual' ? 1000.00 : 1500.00,
            'late_fee_amount' => $membershipType === 'Individual' ? 200.00 : 300.00,
            'description' => 'Default fee structure'
        ];
    }
    
    // Get the duration options and other settings
    $settingsQuery = "SELECT setting_key, setting_value FROM settings 
                     WHERE setting_key IN (
                         'default_duration',
                         '{$membershipType}_2_years_fee', 
                         '{$membershipType}_3_years_fee', 
                         '{$membershipType}_5_years_fee',
                         'processing_fee'
                     )";
    
    $settingsStmt = $pdo->prepare($settingsQuery);
    $settingsStmt->execute();
    
    $settingsData = [];
    while ($row = $settingsStmt->fetch(PDO::FETCH_ASSOC)) {
        $settingsData[$row['setting_key']] = $row['setting_value'];
    }
    
    // Prepare duration options
    $durationOptions = [
        [
            'duration' => '1 Year',
            'amount' => $result['fee_amount']
        ]
    ];
    
    // Add multi-year options if available
    if (isset($settingsData["{$membershipType}_2_years_fee"])) {
        $durationOptions[] = [
            'duration' => '2 Years',
            'amount' => floatval($settingsData["{$membershipType}_2_years_fee"])
        ];
    }
    
    if (isset($settingsData["{$membershipType}_3_years_fee"])) {
        $durationOptions[] = [
            'duration' => '3 Years',
            'amount' => floatval($settingsData["{$membershipType}_3_years_fee"])
        ];
    }
    
    if (isset($settingsData["{$membershipType}_5_years_fee"])) {
        $durationOptions[] = [
            'duration' => '5 Years',
            'amount' => floatval($settingsData["{$membershipType}_5_years_fee"])
        ];
    }
    
    // Build response
    $responseData = [
        'base_fee' => $result['fee_amount'],
        'late_fee' => $result['late_fee_amount'],
        'processing_fee' => isset($settingsData['processing_fee']) ? floatval($settingsData['processing_fee']) : 50.00,
        'description' => $result['description'],
        'duration_options' => $durationOptions,
        'default_duration' => isset($settingsData['default_duration']) ? intval($settingsData['default_duration']) : 12
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $responseData,
        'message' => 'Membership fee details fetched successfully'
    ]);
    
} catch (PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
