<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        exit;
    }
    
    // Validate required fields
    $requiredFields = ['transaction_title', 'transaction_category', 'amount', 'transaction_date', 'payment_method', 'status'];
    foreach ($requiredFields as $field) {
        if (empty($_POST[$field])) {
            echo json_encode(['success' => false, 'message' => "Field '{$field}' is required"]);
            exit;
        }
    }
    
    // Validate amount
    if (!is_numeric($_POST['amount']) || $_POST['amount'] <= 0) {
        echo json_encode(['success' => false, 'message' => 'Amount must be a positive number']);
        exit;
    }
    
    // Validate category exists and is a debit type
    $categoryQuery = "SELECT * FROM transaction_categories WHERE id = ? AND type = 'debit'";
    $categoryStmt = $pdo->prepare($categoryQuery);
    $categoryStmt->execute([(int)$_POST['transaction_category']]);
    
    if (!$categoryStmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Invalid expense category']);
        exit;
    }
    
    // Handle file upload
    $receiptFile = null;
    if (!empty($_FILES['receipt_file']['name'])) {
        $targetDir = __DIR__ . '/../../uploads/receipts/';
        
        // Create directory if it doesn't exist
        if (!is_dir($targetDir)) {
            mkdir($targetDir, 0755, true);
        }
        
        // Generate unique filename
        $fileExtension = pathinfo($_FILES['receipt_file']['name'], PATHINFO_EXTENSION);
        $uniqueFilename = 'transaction_receipt_' . time() . '_' . uniqid() . '.' . $fileExtension;
        $targetFile = $targetDir . $uniqueFilename;
        
        // Check file size (limit to 2MB)
        if ($_FILES['receipt_file']['size'] > 2 * 1024 * 1024) {
            echo json_encode(['success' => false, 'message' => 'File size exceeds the 2MB limit']);
            exit;
        }
        
        // Check file type
        $allowedTypes = ['jpg', 'jpeg', 'png', 'pdf'];
        if (!in_array(strtolower($fileExtension), $allowedTypes)) {
            echo json_encode(['success' => false, 'message' => 'Only JPG, PNG, and PDF files are allowed']);
            exit;
        }
        
        // Upload file
        if (move_uploaded_file($_FILES['receipt_file']['tmp_name'], $targetFile)) {
            $receiptFile = 'uploads/receipts/' . $uniqueFilename;
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to upload file']);
            exit;
        }
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    // Insert into database
    $query = "INSERT INTO financial_transactions (
                transaction_title,
                transaction_category,
                amount,
                transaction_date,
                transaction_type,
                receipt_number,
                receipt_file,
                description,
                status,
                payment_method,
                created_by,
                approved_by,
                notes,
                created_at,
                updated_at
              ) VALUES (?, ?, ?, ?, 'debit', ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
    
    $adminId = isset($_SESSION['admin_id']) ? $_SESSION['admin_id'] : null;
    $approvedBy = ($_POST['status'] === 'approved' || $_POST['status'] === 'paid') ? $adminId : null;
    
    $stmt = $pdo->prepare($query);
    $result = $stmt->execute([
        $_POST['transaction_title'],
        (int)$_POST['transaction_category'],
        $_POST['amount'],
        $_POST['transaction_date'],
        $_POST['receipt_number'] ?? null,
        $receiptFile,
        $_POST['description'] ?? null,
        $_POST['status'],
        $_POST['payment_method'],
        $adminId,
        $approvedBy,
        $_POST['notes'] ?? null
    ]);
    
    if (!$result) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to add expense record']);
        exit;
    }
    
    $expenseId = $pdo->lastInsertId();
    
    // Log activity
    $activityQuery = "INSERT INTO activity_logs (
                        action, description, user_id, user_type, ip_address
                    ) VALUES (?, ?, ?, ?, ?)";
    
    $activityStmt = $pdo->prepare($activityQuery);
    $activityResult = $activityStmt->execute([
        'expense_added',
        'Expense record added: ' . $_POST['transaction_title'] . ' (₹' . $_POST['amount'] . ')',
        $adminId,
        'admin',
        $_SERVER['REMOTE_ADDR']
    ]);
    
    if (!$activityResult) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to log activity']);
        exit;
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Expense record added successfully',
        'transaction_id' => $expenseId
    ]);
    
} catch (PDOException $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
