<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        exit;
    }
    
    if (empty($_POST['category_id']) || !is_numeric($_POST['category_id'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid category ID']);
        exit;
    }
    
    // Check if category exists
    $checkQuery = "SELECT * FROM transaction_categories WHERE id = ?";
    $checkStmt = $pdo->prepare($checkQuery);
    $checkStmt->execute([(int)$_POST['category_id']]);
    $category = $checkStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$category) {
        echo json_encode(['success' => false, 'message' => 'Category not found']);
        exit;
    }
    
    // Check if category is in use
    $usageQuery = "SELECT COUNT(*) FROM financial_transactions WHERE transaction_category = ?";
    $usageStmt = $pdo->prepare($usageQuery);
    $usageStmt->execute([(int)$_POST['category_id']]);
    
    if ($usageStmt->fetchColumn() > 0) {
        echo json_encode([
            'success' => false, 
            'message' => 'Cannot delete category as it is being used by transactions. Reassign transactions to another category first.'
        ]);
        exit;
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    // Delete category
    $query = "DELETE FROM transaction_categories WHERE id = ?";
    $stmt = $pdo->prepare($query);
    $result = $stmt->execute([(int)$_POST['category_id']]);
    
    if (!$result) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to delete category']);
        exit;
    }
    
    // Log activity
    $adminId = isset($_SESSION['admin_id']) ? $_SESSION['admin_id'] : null;
    
    $activityQuery = "INSERT INTO activity_logs (
                        action, description, user_id, user_type, ip_address
                    ) VALUES (?, ?, ?, ?, ?)";
    
    $activityStmt = $pdo->prepare($activityQuery);
    $activityResult = $activityStmt->execute([
        'category_deleted',
        $category['type'] . ' category deleted: ' . $category['category_name'],
        $adminId,
        'admin',
        $_SERVER['REMOTE_ADDR']
    ]);
    
    if (!$activityResult) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to log activity']);
        exit;
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Category deleted successfully'
    ]);
    
} catch (PDOException $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
