<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        exit;
    }
    
    // Validate expense ID
    if (empty($_POST['transaction_id']) || !is_numeric($_POST['transaction_id'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid transaction ID']);
        exit;
    }
    
    $transactionId = (int)$_POST['transaction_id'];
    
    // Check if transaction exists and get receipt file if exists
    $checkQuery = "SELECT id, receipt_file FROM financial_transactions WHERE id = ?";
    $checkStmt = $pdo->prepare($checkQuery);
    $checkStmt->execute([$transactionId]);
    $transaction = $checkStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$transaction) {
        echo json_encode(['success' => false, 'message' => 'Transaction not found']);
        exit;
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    // Delete from database
    $deleteQuery = "DELETE FROM financial_transactions WHERE id = ?";
    $deleteStmt = $pdo->prepare($deleteQuery);
    $result = $deleteStmt->execute([$transactionId]);
    
    if (!$result) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to delete transaction']);
        exit;
    }
    
    // Delete receipt file if exists
    if (!empty($transaction['receipt_file'])) {
        $filePath = __DIR__ . '/../../' . $transaction['receipt_file'];
        if (file_exists($filePath)) {
            unlink($filePath);
        }
    }
    
    // Log activity
    $adminId = isset($_SESSION['admin_id']) ? $_SESSION['admin_id'] : null;
    $activityQuery = "INSERT INTO activity_logs (
                        action, description, user_id, user_type, ip_address
                    ) VALUES (?, ?, ?, ?, ?)";
    
    $activityStmt = $pdo->prepare($activityQuery);
    $activityResult = $activityStmt->execute([
        'transaction_deleted',
        'Expense transaction deleted (ID: ' . $transactionId . ')',
        $adminId,
        'admin',
        $_SERVER['REMOTE_ADDR']
    ]);
    
    if (!$activityResult) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to log activity']);
        exit;
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Transaction deleted successfully'
    ]);
    
} catch (PDOException $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
