<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        exit;
    }
    
    if (empty($_POST['transaction_id']) || !is_numeric($_POST['transaction_id'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid transaction ID']);
        exit;
    }
    
    // Get current transaction data to check if file exists
    $getQuery = "SELECT * FROM financial_transactions WHERE id = ?";
    $getStmt = $pdo->prepare($getQuery);
    $getStmt->execute([(int)$_POST['transaction_id']]);
    $transaction = $getStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$transaction) {
        echo json_encode(['success' => false, 'message' => 'Transaction not found']);
        exit;
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    // Delete transaction record
    $query = "DELETE FROM financial_transactions WHERE id = ?";
    $stmt = $pdo->prepare($query);
    $result = $stmt->execute([(int)$_POST['transaction_id']]);
    
    if (!$result) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to delete transaction']);
        exit;
    }
    
    // Delete receipt file if exists
    if ($transaction['receipt_file'] && file_exists(__DIR__ . '/../../' . $transaction['receipt_file'])) {
        unlink(__DIR__ . '/../../' . $transaction['receipt_file']);
    }
    
    // Log activity
    $adminId = isset($_SESSION['admin_id']) ? $_SESSION['admin_id'] : null;
    
    $activityQuery = "INSERT INTO activity_logs (
                        action, description, user_id, user_type, ip_address
                    ) VALUES (?, ?, ?, ?, ?)";
    
    $action = $transaction['transaction_type'] . '_deleted';
    $description = ucfirst($transaction['transaction_type']) . ' transaction deleted: ' . 
                  $transaction['transaction_title'] . ' (₹' . $transaction['amount'] . ')';
    
    $activityStmt = $pdo->prepare($activityQuery);
    $activityResult = $activityStmt->execute([
        $action,
        $description,
        $adminId,
        'admin',
        $_SERVER['REMOTE_ADDR']
    ]);
    
    if (!$activityResult) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to log activity']);
        exit;
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Transaction deleted successfully'
    ]);
    
} catch (PDOException $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
