<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();

    // Get total income (credit transactions)
    $incomeQuery = "SELECT COALESCE(SUM(amount), 0) as total_income 
                    FROM financial_transactions 
                    WHERE transaction_type = 'credit'";
    $incomeResult = $pdo->query($incomeQuery);
    $totalIncome = $incomeResult->fetchColumn();

    // Get total expenses (debit transactions)
    $expenseQuery = "SELECT COALESCE(SUM(amount), 0) as total_expenses 
                     FROM financial_transactions 
                     WHERE transaction_type = 'debit'";
    $expenseResult = $pdo->query($expenseQuery);
    $totalExpenses = $expenseResult->fetchColumn();

    // Calculate balance
    $balance = $totalIncome - $totalExpenses;

    // Get pending transactions count
    $pendingQuery = "SELECT COUNT(*) as pending_count 
                     FROM financial_transactions 
                     WHERE status = 'pending'";
    $pendingResult = $pdo->query($pendingQuery);
    $pendingCount = $pendingResult->fetchColumn();

    // Get transaction count by category (top 5)
    $categoryQuery = "SELECT c.category_name, c.type, COUNT(t.id) as transaction_count
                      FROM financial_transactions t
                      JOIN transaction_categories c ON t.transaction_category = c.id
                      GROUP BY t.transaction_category
                      ORDER BY transaction_count DESC
                      LIMIT 5";
    $categoryResult = $pdo->query($categoryQuery);
    $categoryStats = $categoryResult->fetchAll(PDO::FETCH_ASSOC);

    // Get monthly transaction totals for the last 6 months
    $monthlyQuery = "SELECT 
                        DATE_FORMAT(transaction_date, '%Y-%m') as month,
                        SUM(CASE WHEN transaction_type = 'credit' THEN amount ELSE 0 END) as income,
                        SUM(CASE WHEN transaction_type = 'debit' THEN amount ELSE 0 END) as expense
                     FROM financial_transactions
                     WHERE transaction_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
                     GROUP BY month
                     ORDER BY month ASC";
    $monthlyResult = $pdo->query($monthlyQuery);
    $monthlyStats = $monthlyResult->fetchAll(PDO::FETCH_ASSOC);

    // Format monthly data for charts
    $months = [];
    $incomeData = [];
    $expenseData = [];

    foreach ($monthlyStats as $month) {
        $months[] = date('M Y', strtotime($month['month'] . '-01'));
        $incomeData[] = round($month['income'], 2);
        $expenseData[] = round($month['expense'], 2);
    }

    echo json_encode([
        'success' => true, 
        'data' => [
            'total_income' => $totalIncome,
            'total_expenses' => $totalExpenses,
            'balance' => $balance,
            'pending_count' => $pendingCount,
            'category_stats' => $categoryStats,
            'monthly_stats' => [
                'months' => $months,
                'income' => $incomeData,
                'expense' => $expenseData
            ]
        ]
    ]);
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
