<?php
header('Content-Type: application/json');

try {
    // Include database connection
    require_once __DIR__ . '/../../config/database.php';
    $pdo = getDBConnection();

    // Set up pagination
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 25;
    $offset = ($page - 1) * $limit;
    
    // Set up filters
    $filters = [];
    $filterParams = [];
    
    // Transaction type filter (debit/credit)
    if (!empty($_GET['transaction_type'])) {
        $filters[] = "t.transaction_type = ?";
        $filterParams[] = $_GET['transaction_type'];
    }
    
    // Category filter
    if (!empty($_GET['category_id'])) {
        $filters[] = "t.transaction_category = ?";
        $filterParams[] = (int)$_GET['category_id'];
    }
    
    // Date range filter
    if (!empty($_GET['start_date'])) {
        $filters[] = "t.transaction_date >= ?";
        $filterParams[] = $_GET['start_date'];
    }
    
    if (!empty($_GET['end_date'])) {
        $filters[] = "t.transaction_date <= ?";
        $filterParams[] = $_GET['end_date'];
    }
    
    // Status filter
    if (!empty($_GET['status'])) {
        $filters[] = "t.status = ?";
        $filterParams[] = $_GET['status'];
    }
    
    // Payment method filter
    if (!empty($_GET['payment_method'])) {
        $filters[] = "t.payment_method = ?";
        $filterParams[] = $_GET['payment_method'];
    }
    
    // Search filter
    if (!empty($_GET['search'])) {
        $filters[] = "(t.transaction_title LIKE ? OR t.description LIKE ? OR t.receipt_number LIKE ?)";
        $searchTerm = "%" . $_GET['search'] . "%";
        $filterParams[] = $searchTerm;
        $filterParams[] = $searchTerm;
        $filterParams[] = $searchTerm;
    }
    
    // Build the WHERE clause
    $whereClause = !empty($filters) ? " WHERE " . implode(" AND ", $filters) : "";
    
    // Count total records for pagination
    $countQuery = "SELECT COUNT(*) FROM financial_transactions t" . $whereClause;
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->execute($filterParams);
    $totalRecords = $countStmt->fetchColumn();
    
    // Main query
    $query = "SELECT 
        t.id,
        t.transaction_title,
        t.transaction_type,
        t.amount,
        t.transaction_date,
        t.receipt_number,
        t.receipt_file,
        t.description,
        t.status,
        t.payment_method,
        t.created_at,
        t.updated_at,
        c.category_name,
        c.type as category_type,
        a1.name as created_by_name,
        a2.name as approved_by_name
    FROM 
        financial_transactions t
    LEFT JOIN 
        transaction_categories c ON t.transaction_category = c.id
    LEFT JOIN 
        admins a1 ON t.created_by = a1.id
    LEFT JOIN 
        admins a2 ON t.approved_by = a2.id
    " . $whereClause . "
    ORDER BY 
        t.transaction_date DESC, t.created_at DESC
    LIMIT ?, ?";
    
    // Add pagination parameters
    $filterParams[] = $offset;
    $filterParams[] = $limit;
    
    $stmt = $pdo->prepare($query);
    $stmt->execute($filterParams);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format data for the response
    foreach ($transactions as &$transaction) {
        // Format dates
        $transaction['transaction_date_formatted'] = date('d M Y', strtotime($transaction['transaction_date']));
        $transaction['created_at_formatted'] = date('d M Y H:i', strtotime($transaction['created_at']));
        
        // Format amount with currency symbol
        $transaction['amount_formatted'] = '₹' . number_format($transaction['amount'], 2);
        
        // Status badge class
        switch ($transaction['status']) {
            case 'pending':
                $transaction['status_badge'] = 'warning';
                break;
            case 'approved':
                $transaction['status_badge'] = 'success';
                break;
            case 'paid':
                $transaction['status_badge'] = 'success';
                break;
            case 'rejected':
                $transaction['status_badge'] = 'danger';
                break;
            default:
                $transaction['status_badge'] = 'secondary';
        }
    }
    
    echo json_encode([
        'success' => true,
        'data' => $transactions,
        'total' => $totalRecords,
        'page' => $page,
        'limit' => $limit,
        'pages' => ceil($totalRecords / $limit)
    ]);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
