<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    // Get time range parameters
    $period = isset($_GET['period']) ? $_GET['period'] : 'month';
    $startDate = null;
    $endDate = null;
    
    // Calculate date ranges based on the period
    switch ($period) {
        case 'week':
            $startDate = date('Y-m-d', strtotime('-1 week'));
            $endDate = date('Y-m-d');
            break;
        case 'month':
            $startDate = date('Y-m-01'); // First day of current month
            $endDate = date('Y-m-t');    // Last day of current month
            break;
        case 'quarter':
            $currentMonth = date('n');
            $currentQuarter = ceil($currentMonth / 3);
            $startMonth = (($currentQuarter - 1) * 3) + 1;
            $startDate = date('Y-' . str_pad($startMonth, 2, '0', STR_PAD_LEFT) . '-01');
            $endDate = date('Y-m-t', strtotime($startDate . ' +2 month'));
            break;
        case 'year':
            $startDate = date('Y-01-01'); // First day of current year
            $endDate = date('Y-12-31');   // Last day of current year
            break;
        case 'custom':
            $startDate = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
            $endDate = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
            break;
    }
    
    // Get total expenses
    $totalQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM expenses";
    $totalStmt = $pdo->query($totalQuery);
    $totalExpenses = (float)$totalStmt->fetchColumn();
    
    // Get expenses for the selected period
    $periodQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM expenses 
                    WHERE expense_date BETWEEN ? AND ?";
    $periodStmt = $pdo->prepare($periodQuery);
    $periodStmt->execute([$startDate, $endDate]);
    $periodExpenses = (float)$periodStmt->fetchColumn();
    
    // Get expenses by category for the selected period
    $categoryQuery = "SELECT 
                       c.id,
                       c.category_name,
                       COALESCE(SUM(e.amount), 0) as total,
                       c.budget_limit
                     FROM expense_categories c
                     LEFT JOIN expenses e ON c.id = e.expense_category AND e.expense_date BETWEEN ? AND ?
                     GROUP BY c.id, c.category_name, c.budget_limit
                     ORDER BY total DESC";
    $categoryStmt = $pdo->prepare($categoryQuery);
    $categoryStmt->execute([$startDate, $endDate]);
    $categoryExpenses = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate budget utilization percentages
    foreach ($categoryExpenses as &$category) {
        $category['total'] = (float)$category['total'];
        
        if (isset($category['budget_limit']) && $category['budget_limit'] > 0) {
            $category['budget_limit'] = (float)$category['budget_limit'];
            $category['utilization_percentage'] = ($category['total'] / $category['budget_limit']) * 100;
        } else {
            $category['utilization_percentage'] = null;
        }
    }
    
    // Get expenses by payment method
    $methodQuery = "SELECT 
                     payment_method,
                     COALESCE(SUM(amount), 0) as total
                   FROM expenses
                   WHERE expense_date BETWEEN ? AND ?
                   GROUP BY payment_method
                   ORDER BY total DESC";
    $methodStmt = $pdo->prepare($methodQuery);
    $methodStmt->execute([$startDate, $endDate]);
    $methodExpenses = $methodStmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($methodExpenses as &$method) {
        $method['total'] = (float)$method['total'];
    }
    
    // Get expenses by status
    $statusQuery = "SELECT 
                     status,
                     COALESCE(SUM(amount), 0) as total,
                     COUNT(*) as count
                   FROM expenses
                   WHERE expense_date BETWEEN ? AND ?
                   GROUP BY status";
    $statusStmt = $pdo->prepare($statusQuery);
    $statusStmt->execute([$startDate, $endDate]);
    $statusExpenses = $statusStmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($statusExpenses as &$status) {
        $status['total'] = (float)$status['total'];
        $status['count'] = (int)$status['count'];
    }
    
    // Get monthly trend data
    $trendQuery = "SELECT 
                    DATE_FORMAT(expense_date, '%Y-%m') as month,
                    COALESCE(SUM(amount), 0) as total
                  FROM expenses
                  WHERE expense_date >= DATE_SUB(CURRENT_DATE(), INTERVAL 6 MONTH)
                  GROUP BY DATE_FORMAT(expense_date, '%Y-%m')
                  ORDER BY month ASC";
    $trendStmt = $pdo->query($trendQuery);
    $trendData = $trendStmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($trendData as &$month) {
        $month['total'] = (float)$month['total'];
        // Format month for display (e.g., "2025-07" to "Jul 2025")
        $monthDate = new DateTime($month['month'] . '-01');
        $month['month_display'] = $monthDate->format('M Y');
    }
    
    echo json_encode([
        'success' => true,
        'total_expenses' => $totalExpenses,
        'period_expenses' => $periodExpenses,
        'period_start' => $startDate,
        'period_end' => $endDate,
        'categories' => $categoryExpenses,
        'payment_methods' => $methodExpenses,
        'status_breakdown' => $statusExpenses,
        'monthly_trend' => $trendData
    ]);
    
} catch (PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
