<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    // Get total credits (income)
    $creditQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM financial_transactions 
                    WHERE transaction_type = 'credit' AND (status = 'approved' OR status = 'paid')";
    $creditStmt = $pdo->query($creditQuery);
    $totalCredits = (float)$creditStmt->fetchColumn();
    
    // Get total debits (expenses)
    $debitQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM financial_transactions 
                   WHERE transaction_type = 'debit' AND (status = 'approved' OR status = 'paid')";
    $debitStmt = $pdo->query($debitQuery);
    $totalDebits = (float)$debitStmt->fetchColumn();
    
    // Calculate balance
    $balance = $totalCredits - $totalDebits;
    
    // Get current month's data
    $currentMonth = date('Y-m-01');
    $monthEndDate = date('Y-m-t');
    
    $monthCreditQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM financial_transactions 
                         WHERE transaction_type = 'credit' AND (status = 'approved' OR status = 'paid')
                         AND transaction_date BETWEEN ? AND ?";
    $monthCreditStmt = $pdo->prepare($monthCreditQuery);
    $monthCreditStmt->execute([$currentMonth, $monthEndDate]);
    $monthCredits = (float)$monthCreditStmt->fetchColumn();
    
    $monthDebitQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM financial_transactions 
                        WHERE transaction_type = 'debit' AND (status = 'approved' OR status = 'paid')
                        AND transaction_date BETWEEN ? AND ?";
    $monthDebitStmt = $pdo->prepare($monthDebitQuery);
    $monthDebitStmt->execute([$currentMonth, $monthEndDate]);
    $monthDebits = (float)$monthDebitStmt->fetchColumn();
    
    $monthlyNet = $monthCredits - $monthDebits;
    
    // Get monthly data for chart (last 6 months)
    $monthlyData = [];
    for ($i = 5; $i >= 0; $i--) {
        $month = date('Y-m-01', strtotime("-$i months"));
        $monthEnd = date('Y-m-t', strtotime("-$i months"));
        $monthName = date('M Y', strtotime("-$i months"));
        
        $monthCreditQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM financial_transactions 
                            WHERE transaction_type = 'credit' AND (status = 'approved' OR status = 'paid')
                            AND transaction_date BETWEEN ? AND ?";
        $monthCreditStmt = $pdo->prepare($monthCreditQuery);
        $monthCreditStmt->execute([$month, $monthEnd]);
        $credits = (float)$monthCreditStmt->fetchColumn();
        
        $monthDebitQuery = "SELECT COALESCE(SUM(amount), 0) as total FROM financial_transactions 
                           WHERE transaction_type = 'debit' AND (status = 'approved' OR status = 'paid')
                           AND transaction_date BETWEEN ? AND ?";
        $monthDebitStmt = $pdo->prepare($monthDebitQuery);
        $monthDebitStmt->execute([$month, $monthEnd]);
        $debits = (float)$monthDebitStmt->fetchColumn();
        
        $net = $credits - $debits;
        
        $monthlyData[] = [
            'month' => $monthName,
            'credits' => $credits,
            'debits' => $debits,
            'net' => $net
        ];
    }
    
    // Get category distribution data
    $categoryData = [
        'credit' => [],
        'debit' => []
    ];
    
    // Credit categories
    $creditCategoryQuery = "SELECT 
                            c.category_name as category,
                            COALESCE(SUM(t.amount), 0) as amount
                          FROM transaction_categories c
                          LEFT JOIN financial_transactions t ON c.id = t.transaction_category AND (t.status = 'approved' OR t.status = 'paid')
                          WHERE c.type = 'credit'
                          GROUP BY c.id, c.category_name
                          ORDER BY amount DESC";
    $creditCategoryStmt = $pdo->query($creditCategoryQuery);
    $categoryData['credit'] = $creditCategoryStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Debit categories
    $debitCategoryQuery = "SELECT 
                           c.category_name as category,
                           COALESCE(SUM(t.amount), 0) as amount
                         FROM transaction_categories c
                         LEFT JOIN financial_transactions t ON c.id = t.transaction_category AND (t.status = 'approved' OR t.status = 'paid')
                         WHERE c.type = 'debit'
                         GROUP BY c.id, c.category_name
                         ORDER BY amount DESC";
    $debitCategoryStmt = $pdo->query($debitCategoryQuery);
    $categoryData['debit'] = $debitCategoryStmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'totalCredits' => $totalCredits,
        'totalDebits' => $totalDebits,
        'balance' => $balance,
        'monthlyNet' => $monthlyNet,
        'monthlyData' => $monthlyData,
        'categoryData' => $categoryData
    ]);
    
} catch (PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
