<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        exit;
    }
    
    if (empty($_POST['transaction_id']) || !is_numeric($_POST['transaction_id'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid transaction ID']);
        exit;
    }
    
    // Validate required fields
    $requiredFields = ['transaction_title', 'transaction_category', 'amount', 'transaction_date', 'payment_method', 'status'];
    foreach ($requiredFields as $field) {
        if (empty($_POST[$field])) {
            echo json_encode(['success' => false, 'message' => "Field '{$field}' is required"]);
            exit;
        }
    }
    
    // Validate amount
    if (!is_numeric($_POST['amount']) || $_POST['amount'] <= 0) {
        echo json_encode(['success' => false, 'message' => 'Amount must be a positive number']);
        exit;
    }
    
    // Get current expense data to check if receipt file exists
    $getQuery = "SELECT receipt_file FROM financial_transactions WHERE id = ?";
    $getStmt = $pdo->prepare($getQuery);
    $getStmt->execute([(int)$_POST['transaction_id']]);
    $currentExpense = $getStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$currentExpense) {
        echo json_encode(['success' => false, 'message' => 'Transaction record not found']);
        exit;
    }
    
    // Handle file upload
    $receiptFile = $currentExpense['receipt_file'];
    if (!empty($_FILES['receipt_file']['name'])) {
        $targetDir = __DIR__ . '/../../uploads/receipts/';
        
        // Create directory if it doesn't exist
        if (!is_dir($targetDir)) {
            mkdir($targetDir, 0755, true);
        }
        
        // Generate unique filename
        $fileExtension = pathinfo($_FILES['receipt_file']['name'], PATHINFO_EXTENSION);
        $uniqueFilename = 'transaction_receipt_' . time() . '_' . uniqid() . '.' . $fileExtension;
        $targetFile = $targetDir . $uniqueFilename;
        
        // Check file size (limit to 2MB)
        if ($_FILES['receipt_file']['size'] > 2 * 1024 * 1024) {
            echo json_encode(['success' => false, 'message' => 'File size exceeds the 2MB limit']);
            exit;
        }
        
        // Check file type
        $allowedTypes = ['jpg', 'jpeg', 'png', 'pdf'];
        if (!in_array(strtolower($fileExtension), $allowedTypes)) {
            echo json_encode(['success' => false, 'message' => 'Only JPG, PNG, and PDF files are allowed']);
            exit;
        }
        
        // Upload file
        if (move_uploaded_file($_FILES['receipt_file']['tmp_name'], $targetFile)) {
            $receiptFile = 'uploads/receipts/' . $uniqueFilename;
            
            // Delete old file if exists
            if ($currentExpense['receipt_file'] && file_exists(__DIR__ . '/../../' . $currentExpense['receipt_file'])) {
                unlink(__DIR__ . '/../../' . $currentExpense['receipt_file']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to upload file']);
            exit;
        }
    }
    
    // Update database
    $query = "UPDATE financial_transactions SET
                transaction_title = ?,
                transaction_category = ?,
                amount = ?,
                transaction_date = ?,
                receipt_number = ?,
                receipt_file = ?,
                description = ?,
                status = ?,
                payment_method = ?,
                notes = ?,
                updated_at = NOW()
              WHERE id = ?";
    
    $adminId = isset($_SESSION['admin_id']) ? $_SESSION['admin_id'] : null;
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([
        $_POST['transaction_title'],
        $_POST['transaction_category'],
        $_POST['amount'],
        $_POST['transaction_date'],
        $_POST['receipt_number'] ?? null,
        $receiptFile,
        $_POST['description'] ?? null,
        $_POST['status'],
        $_POST['payment_method'],
        $_POST['notes'] ?? null,
        (int)$_POST['transaction_id']
    ]);
    
    // If status changed to approved, update approved_by
    if ($_POST['status'] === 'approved' || $_POST['status'] === 'paid') {
        $approveQuery = "UPDATE financial_transactions SET approved_by = ? WHERE id = ?";
        $approveStmt = $pdo->prepare($approveQuery);
        $approveStmt->execute([$adminId, (int)$_POST['transaction_id']]);
    }
    
    // Log activity
    $activityQuery = "INSERT INTO activity_logs (
                        action, description, user_id, user_type, ip_address
                    ) VALUES (?, ?, ?, ?, ?)";
    
    $activityStmt = $pdo->prepare($activityQuery);
    $activityStmt->execute([
        'transaction_updated',
        'Expense transaction updated: ' . $_POST['transaction_title'] . ' (₹' . $_POST['amount'] . ')',
        $adminId,
        'admin',
        $_SERVER['REMOTE_ADDR']
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Expense record updated successfully'
    ]);
    
} catch (PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
