<?php
require_once __DIR__ . '/../../config/database.php';
header('Content-Type: application/json');

try {
    $pdo = getDBConnection();
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        exit;
    }
    
    if (empty($_POST['transaction_id']) || !is_numeric($_POST['transaction_id'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid transaction ID']);
        exit;
    }
    
    // Validate required fields
    $requiredFields = [
        'transaction_title', 
        'transaction_category', 
        'amount', 
        'transaction_date', 
        'transaction_type',
        'payment_method',
        'status'
    ];
    
    foreach ($requiredFields as $field) {
        if (empty($_POST[$field])) {
            echo json_encode(['success' => false, 'message' => "Field '{$field}' is required"]);
            exit;
        }
    }
    
    // Validate transaction type
    if (!in_array($_POST['transaction_type'], ['credit', 'debit'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid transaction type']);
        exit;
    }
    
    // Validate amount
    if (!is_numeric($_POST['amount']) || $_POST['amount'] <= 0) {
        echo json_encode(['success' => false, 'message' => 'Amount must be a positive number']);
        exit;
    }
    
    // Validate category
    $categoryQuery = "SELECT * FROM transaction_categories WHERE id = ? AND type = ?";
    $categoryStmt = $pdo->prepare($categoryQuery);
    $categoryStmt->execute([
        (int)$_POST['transaction_category'],
        $_POST['transaction_type']
    ]);
    
    if (!$categoryStmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Invalid category for this transaction type']);
        exit;
    }
    
    // Get current transaction data to check if receipt file exists
    $getQuery = "SELECT * FROM financial_transactions WHERE id = ?";
    $getStmt = $pdo->prepare($getQuery);
    $getStmt->execute([(int)$_POST['transaction_id']]);
    $currentTransaction = $getStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$currentTransaction) {
        echo json_encode(['success' => false, 'message' => 'Transaction not found']);
        exit;
    }
    
    // Handle file upload
    $receiptFile = $currentTransaction['receipt_file'];
    if (!empty($_FILES['receipt_file']['name'])) {
        $targetDir = __DIR__ . '/../../uploads/receipts/';
        
        // Create directory if it doesn't exist
        if (!is_dir($targetDir)) {
            mkdir($targetDir, 0755, true);
        }
        
        // Generate unique filename
        $fileExtension = pathinfo($_FILES['receipt_file']['name'], PATHINFO_EXTENSION);
        $uniqueFilename = 'receipt_' . time() . '_' . uniqid() . '.' . $fileExtension;
        $targetFile = $targetDir . $uniqueFilename;
        
        // Check file size (limit to 2MB)
        if ($_FILES['receipt_file']['size'] > 2 * 1024 * 1024) {
            echo json_encode(['success' => false, 'message' => 'File size exceeds the 2MB limit']);
            exit;
        }
        
        // Check file type
        $allowedTypes = ['jpg', 'jpeg', 'png', 'pdf'];
        if (!in_array(strtolower($fileExtension), $allowedTypes)) {
            echo json_encode(['success' => false, 'message' => 'Only JPG, PNG, and PDF files are allowed']);
            exit;
        }
        
        // Upload file
        if (move_uploaded_file($_FILES['receipt_file']['tmp_name'], $targetFile)) {
            $receiptFile = 'uploads/receipts/' . $uniqueFilename;
            
            // Delete old file if exists
            if ($currentTransaction['receipt_file'] && file_exists(__DIR__ . '/../../' . $currentTransaction['receipt_file'])) {
                unlink(__DIR__ . '/../../' . $currentTransaction['receipt_file']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to upload file']);
            exit;
        }
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    // Update database
    $query = "UPDATE financial_transactions SET
                transaction_title = ?,
                transaction_category = ?,
                amount = ?,
                transaction_date = ?,
                transaction_type = ?,
                receipt_number = ?,
                receipt_file = ?,
                description = ?,
                status = ?,
                payment_method = ?,
                notes = ?,
                updated_at = NOW()
              WHERE id = ?";
    
    $stmt = $pdo->prepare($query);
    $result = $stmt->execute([
        $_POST['transaction_title'],
        (int)$_POST['transaction_category'],
        $_POST['amount'],
        $_POST['transaction_date'],
        $_POST['transaction_type'],
        $_POST['receipt_number'] ?? null,
        $receiptFile,
        $_POST['description'] ?? null,
        $_POST['status'],
        $_POST['payment_method'],
        $_POST['notes'] ?? null,
        (int)$_POST['transaction_id']
    ]);
    
    if (!$result) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to update transaction']);
        exit;
    }
    
    // If status changed to approved/paid, update approved_by
    $adminId = isset($_SESSION['admin_id']) ? $_SESSION['admin_id'] : null;
    
    if (($_POST['status'] === 'approved' || $_POST['status'] === 'paid') && 
        ($currentTransaction['status'] !== 'approved' && $currentTransaction['status'] !== 'paid')) {
        $approveQuery = "UPDATE financial_transactions SET approved_by = ? WHERE id = ?";
        $approveStmt = $pdo->prepare($approveQuery);
        $approveResult = $approveStmt->execute([$adminId, (int)$_POST['transaction_id']]);
        
        if (!$approveResult) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Failed to update approval status']);
            exit;
        }
    }
    
    // Log activity
    $activityQuery = "INSERT INTO activity_logs (
                        action, description, user_id, user_type, ip_address
                    ) VALUES (?, ?, ?, ?, ?)";
    
    $action = $_POST['transaction_type'] . '_updated';
    $description = ucfirst($_POST['transaction_type']) . ' transaction updated: ' . 
                  $_POST['transaction_title'] . ' (₹' . $_POST['amount'] . ')';
    
    $activityStmt = $pdo->prepare($activityQuery);
    $activityResult = $activityStmt->execute([
        $action,
        $description,
        $adminId,
        'admin',
        $_SERVER['REMOTE_ADDR']
    ]);
    
    if (!$activityResult) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to log activity']);
        exit;
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Transaction updated successfully'
    ]);
    
} catch (PDOException $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
