<?php
require_once '../../config/database.php';
require_once '../../auth/auth_functions.php';

if (!isLoggedIn() || !isAdmin()) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized access']);
    exit;
}

// Include TCPDF library for PDF generation
require_once '../../vendor/autoload.php';

// Include PhpSpreadsheet for Excel export
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $type = $_GET['type'] ?? 'members';
    $start_date = $_GET['start_date'] ?? date('Y-m-01');
    $end_date = $_GET['end_date'] ?? date('Y-m-t');
    $format = $_GET['format'] ?? 'csv';
    $includeCharts = isset($_GET['includeCharts']) && $_GET['includeCharts'] == '1';
    $includeSummary = isset($_GET['includeSummary']) && $_GET['includeSummary'] == '1';
    $includeDetails = isset($_GET['includeDetails']) && $_GET['includeDetails'] == '1';
    
    // Standardize date format for filename
    $start_date_formatted = date('Y-m-d', strtotime($start_date));
    $end_date_formatted = date('Y-m-d', strtotime($end_date));
    $filename = "report_{$type}_{$start_date_formatted}_to_{$end_date_formatted}";
    
    // Fetch data based on report type
    $data = [];
    $headers = [];
    
    switch ($type) {
        case 'members':
            $headers = ['ID', 'Name', 'Email', 'Membership Type', 'Status', 'Registration Date'];
            $stmt = $pdo->prepare("
                SELECT id, name, email, membership_type, status, created_at
                FROM members 
                WHERE created_at BETWEEN ? AND ?
                ORDER BY created_at DESC
            ");
            $stmt->execute([$start_date, $end_date]);
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
            
        case 'events':
            $headers = ['ID', 'Title', 'Date', 'Location', 'Registrations', 'Revenue', 'Status'];
            $stmt = $pdo->prepare("
                SELECT 
                    e.id, e.title, e.date, e.location, e.status,
                    COUNT(er.id) as registrations,
                    COALESCE(SUM(CASE WHEN er.payment_status = 'paid' THEN er.amount_paid ELSE 0 END), 0) as revenue
                FROM events e
                LEFT JOIN event_registrations er ON e.id = er.event_id
                WHERE e.date BETWEEN ? AND ?
                GROUP BY e.id
                ORDER BY e.date DESC
            ");
            $stmt->execute([$start_date, $end_date]);
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
            
        case 'detailed_finance':
            $headers = ['ID', 'Date', 'Title', 'Category', 'Type', 'Amount', 'Status', 'Payment Method'];
            $stmt = $pdo->prepare("
                SELECT 
                    ft.id, 
                    ft.transaction_date, 
                    ft.transaction_title,
                    tc.name as category_name,
                    ft.transaction_type,
                    ft.amount,
                    ft.status,
                    ft.payment_method
                FROM financial_transactions ft
                LEFT JOIN transaction_categories tc ON ft.category_id = tc.id
                WHERE ft.transaction_date BETWEEN ? AND ?
                ORDER BY ft.transaction_date DESC
            ");
            $stmt->execute([$start_date, $end_date]);
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
            
        case 'financial':
            $headers = ['Category', 'Type', 'Total Amount', 'Count'];
            $stmt = $pdo->prepare("
                SELECT 
                    tc.name as category_name,
                    ft.transaction_type,
                    SUM(ft.amount) as total_amount,
                    COUNT(*) as count
                FROM financial_transactions ft
                LEFT JOIN transaction_categories tc ON ft.category_id = tc.id
                WHERE ft.transaction_date BETWEEN ? AND ?
                GROUP BY tc.name, ft.transaction_type
                ORDER BY tc.name, ft.transaction_type
            ");
            $stmt->execute([$start_date, $end_date]);
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Invalid report type specified']);
            exit;
    }
    
    // Export based on format
    switch ($format) {
        case 'csv':
            exportAsCsv($filename, $headers, $data);
            break;
            
        case 'excel':
            exportAsExcel($filename, $headers, $data, $type);
            break;
            
        case 'pdf':
            exportAsPdf($filename, $headers, $data, $type, $start_date_formatted, $end_date_formatted, $includeCharts);
            break;
            
        case 'json':
            exportAsJson($filename, $data);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Invalid export format specified']);
            exit;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    if ($format === 'json') {
        echo json_encode(['error' => 'Error generating report: ' . $e->getMessage()]);
    } else {
        header('Content-Type: text/plain');
        echo "Error generating report: " . $e->getMessage();
    }
}

/**
 * Export data as CSV
 */
function exportAsCsv($filename, $headers, $data) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '.csv"');
    header('Cache-Control: no-cache, must-revalidate');
    header('Pragma: public');
    
    $output = fopen('php://output', 'w');
    
    // Add headers
    fputcsv($output, $headers);
    
    // Add data rows
    foreach ($data as $row) {
        // Format financial data
        if (isset($row['transaction_type']) && isset($row['amount'])) {
            $row['amount'] = formatAmount($row['amount'], $row['transaction_type']);
        } elseif (isset($row['revenue'])) {
            $row['revenue'] = '₹' . number_format($row['revenue'], 2);
        }
        
        // Format dates
        if (isset($row['created_at'])) {
            $row['created_at'] = date('Y-m-d H:i:s', strtotime($row['created_at']));
        }
        if (isset($row['transaction_date'])) {
            $row['transaction_date'] = date('Y-m-d', strtotime($row['transaction_date']));
        }
        
        fputcsv($output, $row);
    }
    
    fclose($output);
}

/**
 * Export data as Excel
 */
function exportAsExcel($filename, $headers, $data, $type) {
    // Create new Spreadsheet object
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    
    // Set sheet name
    $reportTitles = [
        'members' => 'Member Report',
        'events' => 'Event Report',
        'detailed_finance' => 'Detailed Financial Report',
        'financial' => 'Financial Summary'
    ];
    $sheet->setTitle(isset($reportTitles[$type]) ? $reportTitles[$type] : 'Report');
    
    // Set headers
    $columnIndex = 1;
    foreach ($headers as $header) {
        $sheet->setCellValueByColumnAndRow($columnIndex++, 1, $header);
    }
    
    // Add styling to headers
    $headerStyle = [
        'font' => [
            'bold' => true,
            'color' => ['rgb' => 'FFFFFF'],
        ],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => '4472C4'],
        ],
        'alignment' => [
            'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
        ]
    ];
    $sheet->getStyle('A1:' . \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(count($headers)) . '1')->applyFromArray($headerStyle);
    
    // Set data
    $rowIndex = 2;
    foreach ($data as $row) {
        $columnIndex = 1;
        foreach ($row as $key => $value) {
            // Format amounts
            if ($key === 'amount' && isset($row['transaction_type'])) {
                $value = formatAmount($value, $row['transaction_type'], false);
                
                // Apply cell styling based on transaction type
                $cellCoord = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($columnIndex) . $rowIndex;
                if ($row['transaction_type'] === 'credit') {
                    $sheet->getStyle($cellCoord)->getFont()->getColor()->setRGB('008800'); // Green color
                } else {
                    $sheet->getStyle($cellCoord)->getFont()->getColor()->setRGB('CC0000'); // Red color
                }
            } elseif ($key === 'revenue') {
                $value = number_format($value, 2);
            }
            
            // Format dates
            if ($key === 'created_at' || $key === 'transaction_date') {
                $value = date('Y-m-d', strtotime($value));
            }
            
            $sheet->setCellValueByColumnAndRow($columnIndex++, $rowIndex, $value);
        }
        $rowIndex++;
    }
    
    // Auto-size columns
    foreach (range(1, count($headers)) as $col) {
        $sheet->getColumnDimensionByColumn($col)->setAutoSize(true);
    }
    
    // Create writer and output file
    $writer = new Xlsx($spreadsheet);
    
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="' . $filename . '.xlsx"');
    header('Cache-Control: max-age=0');
    
    $writer->save('php://output');
}

/**
 * Export data as PDF
 */
function exportAsPdf($filename, $headers, $data, $type, $start_date, $end_date, $includeCharts = false) {
    // Create new PDF document
    $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
    
    // Set document information
    $pdf->SetCreator('KSO Chandigarh');
    $pdf->SetAuthor('KSO Chandigarh');
    
    $reportTitles = [
        'members' => 'Member Report',
        'events' => 'Event Report',
        'detailed_finance' => 'Detailed Financial Report',
        'financial' => 'Financial Summary'
    ];
    $title = isset($reportTitles[$type]) ? $reportTitles[$type] : 'Report';
    $pdf->SetTitle($title);
    
    // Set default header and footer data
    $pdf->setHeaderData(PDF_HEADER_LOGO, PDF_HEADER_LOGO_WIDTH, 'KSO Chandigarh', $title . ' | ' . $start_date . ' to ' . $end_date);
    $pdf->setFooterData(array(0,64,0), array(0,64,128));
    
    // Set header and footer fonts
    $pdf->setHeaderFont(Array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));
    $pdf->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
    
    // Set default monospaced font
    $pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);
    
    // Set margins
    $pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
    $pdf->SetHeaderMargin(PDF_MARGIN_HEADER);
    $pdf->SetFooterMargin(PDF_MARGIN_FOOTER);
    
    // Set auto page breaks
    $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);
    
    // Set image scale factor
    $pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);
    
    // Add a page
    $pdf->AddPage();
    
    // Set font
    $pdf->SetFont('helvetica', '', 10);
    
    // Create table header
    $html = '<table border="1" cellpadding="5">';
    $html .= '<tr style="background-color: #4472C4; color: white; font-weight: bold;">';
    foreach ($headers as $header) {
        $html .= '<th>' . $header . '</th>';
    }
    $html .= '</tr>';
    
    // Add table data
    foreach ($data as $row) {
        $html .= '<tr>';
        foreach ($row as $key => $value) {
            $style = '';
            
            // Format financial data with colors
            if ($key === 'amount' && isset($row['transaction_type'])) {
                $value = formatAmount($value, $row['transaction_type'], false);
                $style = $row['transaction_type'] === 'credit' ? 'color: green;' : 'color: red;';
            } elseif ($key === 'revenue') {
                $value = '₹' . number_format($value, 2);
            }
            
            // Format dates
            if ($key === 'created_at' || $key === 'transaction_date') {
                $value = date('Y-m-d', strtotime($value));
            }
            
            $html .= '<td style="' . $style . '">' . $value . '</td>';
        }
        $html .= '</tr>';
    }
    
    $html .= '</table>';
    
    // Output HTML content
    $pdf->writeHTML($html, true, false, true, false, '');
    
    // Close and output PDF
    $pdf->Output($filename . '.pdf', 'D');
}

/**
 * Export data as JSON
 */
function exportAsJson($filename, $data) {
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="' . $filename . '.json"');
    header('Cache-Control: no-cache, must-revalidate');
    header('Pragma: public');
    
    // Format data for better readability
    foreach ($data as &$row) {
        // Format dates
        if (isset($row['created_at'])) {
            $row['created_at_formatted'] = date('Y-m-d H:i:s', strtotime($row['created_at']));
        }
        if (isset($row['transaction_date'])) {
            $row['transaction_date_formatted'] = date('Y-m-d', strtotime($row['transaction_date']));
        }
        
        // Format amounts
        if (isset($row['transaction_type']) && isset($row['amount'])) {
            $row['amount_formatted'] = formatAmount($row['amount'], $row['transaction_type'], false);
            $row['amount_raw'] = floatval($row['amount']);
        }
        if (isset($row['revenue'])) {
            $row['revenue_formatted'] = '₹' . number_format($row['revenue'], 2);
            $row['revenue_raw'] = floatval($row['revenue']);
        }
    }
    
    echo json_encode([
        'success' => true,
        'data' => $data,
        'count' => count($data),
        'generated_at' => date('Y-m-d H:i:s')
    ], JSON_PRETTY_PRINT);
}

/**
 * Format amount based on transaction type
 */
function formatAmount($amount, $type, $includeHtml = true) {
    $amount = floatval($amount);
    
    if ($type === 'debit') {
        if ($includeHtml) {
            return '<span class="text-danger">-₹' . number_format($amount, 2) . '</span>';
        } else {
            return '-₹' . number_format($amount, 2);
        }
    } else {
        if ($includeHtml) {
            return '<span class="text-success">+₹' . number_format($amount, 2) . '</span>';
        } else {
            return '+₹' . number_format($amount, 2);
        }
    }
}
