<?php
require_once '../config/database.php';
require_once '../config/auth_enhanced.php';

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized access']);
    exit;
}

header('Content-Type: application/json');

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $start_date = $_GET['start_date'] ?? date('Y-m-01');
    $end_date = $_GET['end_date'] ?? date('Y-m-t');
    
    // Fetch event reports data
    $stmt = $pdo->prepare("
        SELECT 
            e.id,
            e.title as event_name,
            e.date as event_date,
            e.location,
            e.registration_fee,
            e.capacity,
            e.status,
            COUNT(er.id) as registrations,
            SUM(CASE WHEN er.attended = 1 THEN 1 ELSE 0 END) as attendance,
            COALESCE(SUM(CASE WHEN er.payment_status = 'paid' THEN er.amount_paid ELSE 0 END), 0) as revenue,
            ROUND(COUNT(er.id) / NULLIF(e.capacity, 0) * 100, 2) as capacity_percentage
        FROM events e
        LEFT JOIN event_registrations er ON e.id = er.event_id
        WHERE e.date BETWEEN ? AND ?
        GROUP BY e.id
        ORDER BY e.date DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate summary statistics
    $totalEvents = count($events);
    $totalRegistrations = array_sum(array_column($events, 'registrations'));
    $totalAttendance = array_sum(array_column($events, 'attendance'));
    $totalRevenue = array_sum(array_column($events, 'revenue'));
    
    // Format data for DataTables
    $data = [];
    foreach ($events as $event) {
        $attendance_percentage = $event['registrations'] > 0 ? round(($event['attendance'] / $event['registrations']) * 100, 1) : 0;
        
        $data[] = [
            'id' => $event['id'],
            'event_name' => $event['event_name'],
            'event_date' => date('M d, Y', strtotime($event['event_date'])),
            'location' => $event['location'],
            'registrations' => $event['registrations'],
            'attendance' => $event['attendance'],
            'revenue' => $event['revenue'],
            'status' => ucfirst($event['status']),
            'performance' => $attendance_percentage
        ];
    }
    
    echo json_encode([
        'data' => $data,
        'summary' => [
            'total_events' => $totalEvents,
            'total_registrations' => $totalRegistrations,
            'total_attendance' => $totalAttendance,
            'total_revenue' => $totalRevenue
        ]
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}
?>
