# Assignment System Migration Guide

## Overview

This document provides instructions for migrating existing assignments from the old system (using `member_assignments` table) to the new system (using `admin_assignments` table).

## Background

The assignment system has been refactored to align with the database schema, transitioning from the previous implementation that used:

- Table: `member_assignments`
- Fields: `assigned_to_id`, `assigned_to_type`, `is_active`, etc.

To the new schema-compliant implementation that uses:

- Table: `admin_assignments`
- Fields: `assigned_to`, `assignment_type`, `priority`, `status`, etc.

## Migration Steps

### 1. Database Migration

Execute the following SQL script to migrate existing assignments:

```sql
-- Create a temporary backup of existing assignments
CREATE TABLE IF NOT EXISTS member_assignments_backup AS 
SELECT * FROM member_assignments;

-- Migrate active assignments to admin_assignments
INSERT INTO admin_assignments (
    member_id,
    assigned_to,
    assigned_by,
    assignment_type,
    priority,
    status,
    notes,
    created_at
)
SELECT 
    member_id,
    assigned_to_id,
    assigned_by_id,
    'review',                           -- Default assignment_type
    'normal',                           -- Default priority
    CASE 
        WHEN is_active = 1 THEN 'active'
        ELSE 'cancelled'
    END,                                -- Map is_active to status
    notes,
    created_at
FROM 
    member_assignments
WHERE 
    assigned_to_type = 'admin';         -- Only migrate admin assignments

-- For inactive assignments, set completed info
UPDATE admin_assignments aa
JOIN member_assignments ma ON aa.member_id = ma.member_id 
                          AND aa.assigned_to = ma.assigned_to_id
                          AND aa.status = 'cancelled'
                          AND ma.is_active = 0
SET 
    aa.completed_at = ma.unassigned_at,
    aa.completed_by = ma.unassigned_by_id,
    aa.notes = CONCAT(IFNULL(aa.notes, ''), '\nMigrated from old system. Reason: ', IFNULL(ma.unassignment_reason, 'Unknown'));
```

### 2. API Endpoint Updates

All API endpoints have been updated to work with the new schema:

1. `assign.php`: Now accepts `assigned_to` (admin ID) instead of `assigned_to_id` and `assigned_to_type`
2. `unassign.php`: Now uses `status` for state changes instead of `is_active`
3. `list-assigned.php`: Now accepts `assigned_to` and `status` parameters
4. All other endpoints: Updated to use the correct field names

### 3. Frontend Updates

The JavaScript module `assignment-system.js` has been updated to:

1. Use the correct field names when sending requests to API endpoints
2. Handle new fields like `assignment_type`, `priority`, and `status`
3. Display assignments with proper status and priority indicators
4. Support due dates and assignment types

### 4. Testing Migrated Data

After migration, perform the following tests:

1. **Admin Dashboard**: Verify all assigned members appear correctly
2. **Member Profile**: Verify all assignments for each member appear correctly
3. **Assignment Operations**: Test creating new assignments, cancelling, and completing
4. **Statistics**: Verify assignment statistics are calculated correctly

## Handling Group Assignments

The new system only supports admin assignments. For any group assignments in the old system:

### Option 1: Manual Reassignment

1. Generate a report of all group assignments
2. Manually reassign these to appropriate admins

### Option 2: Group Admin Assignment

1. For each group, identify a primary admin
2. Run the following SQL to migrate group assignments to the primary admin:

```sql
-- Create mapping table for groups to primary admin
CREATE TEMPORARY TABLE group_admin_mapping (
    group_id INT,
    admin_id INT
);

-- Populate with your group->admin mappings
INSERT INTO group_admin_mapping VALUES
(1, 5),    -- Group 1 -> Admin 5
(2, 8),    -- Group 2 -> Admin 8
(3, 12);   -- Group 3 -> Admin 12

-- Migrate group assignments to the mapped admin
INSERT INTO admin_assignments (
    member_id,
    assigned_to,
    assigned_by,
    assignment_type,
    priority,
    status,
    notes,
    created_at
)
SELECT 
    ma.member_id,
    gam.admin_id,           -- Map to admin ID
    ma.assigned_by_id,
    'review',
    'normal',
    CASE 
        WHEN ma.is_active = 1 THEN 'active'
        ELSE 'cancelled'
    END,
    CONCAT(IFNULL(ma.notes, ''), '\nMigrated from group assignment. Group ID: ', ma.assigned_to_id),
    ma.created_at
FROM 
    member_assignments ma
JOIN 
    group_admin_mapping gam ON ma.assigned_to_id = gam.group_id
WHERE 
    ma.assigned_to_type = 'group';
```

## Rollback Plan

If issues are encountered during migration, use the following rollback plan:

1. Restore the previous service files from backup
2. Restore the previous API endpoints from backup
3. Restore the previous JavaScript files from backup
4. Continue using the `member_assignments` table

## Post-Migration Tasks

1. **Update Documentation**: Ensure all documentation references the new fields and structure
2. **User Training**: Brief admins on the new assignment types and priorities
3. **Monitoring**: Monitor system for any issues during the first week after migration
4. **Performance Check**: Verify database indexes are working efficiently with the new query patterns

## Timeline

1. **Backup**: 30 minutes
2. **Database Migration**: 1 hour
3. **Service & API Updates**: 2 hours
4. **Frontend Updates**: 2 hours
5. **Testing**: 3 hours
6. **Total Estimated Time**: 8 hours

## Support Contacts

If issues are encountered during migration, contact:

- Database Support: database-admin@kso.org
- Backend Support: backend-dev@kso.org
- Frontend Support: frontend-dev@kso.org

---

*Migration Guide Created: July 15, 2025*
