<?php
require_once '../config/database.php';
require_once '../config/auth_enhanced.php';

if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized access']);
    exit;
}

header('Content-Type: application/json');

try {
    $database = new Database();
    $pdo = $database->getConnection();
    
    $start_date = $_GET['start_date'] ?? date('Y-m-01');
    $end_date = $_GET['end_date'] ?? date('Y-m-t');
    
    // Get event data by month
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(date, '%Y-%m') as month,
            COUNT(*) as total_events,
            COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_events,
            COUNT(CASE WHEN status = 'upcoming' THEN 1 END) as upcoming_events,
            COUNT(CASE WHEN status = 'cancelled' THEN 1 END) as cancelled_events
        FROM events 
        WHERE date BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(date, '%Y-%m')
        ORDER BY month ASC
    ");
    $stmt->execute([$start_date, $end_date]);
    $event_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format data for Chart.js
    $labels = [];
    $total_data = [];
    $completed_data = [];
    $upcoming_data = [];
    $cancelled_data = [];
    
    foreach ($event_data as $row) {
        $labels[] = date('M Y', strtotime($row['month'] . '-01'));
        $total_data[] = (int)$row['total_events'];
        $completed_data[] = (int)$row['completed_events'];
        $upcoming_data[] = (int)$row['upcoming_events'];
        $cancelled_data[] = (int)$row['cancelled_events'];
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'labels' => $labels,
            'values' => $total_data
        ],
        'chart_data' => [
            'labels' => $labels,
            'datasets' => [
                [
                    'label' => 'Total Events',
                    'data' => $total_data,
                    'backgroundColor' => 'rgba(0, 123, 255, 0.6)',
                    'borderColor' => 'rgba(0, 123, 255, 1)',
                    'borderWidth' => 2
                ],
                [
                    'label' => 'Completed',
                    'data' => $completed_data,
                    'backgroundColor' => 'rgba(40, 167, 69, 0.6)',
                    'borderColor' => 'rgba(40, 167, 69, 1)',
                    'borderWidth' => 2
                ],
                [
                    'label' => 'Upcoming',
                    'data' => $upcoming_data,
                    'backgroundColor' => 'rgba(255, 193, 7, 0.6)',
                    'borderColor' => 'rgba(255, 193, 7, 1)',
                    'borderWidth' => 2
                ],
                [
                    'label' => 'Cancelled',
                    'data' => $cancelled_data,
                    'backgroundColor' => 'rgba(220, 53, 69, 0.6)',
                    'borderColor' => 'rgba(220, 53, 69, 1)',
                    'borderWidth' => 2
                ]
            ]
        ]
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}
?>
