# Notification System Setup Guide

This guide will help you set up the real-time notification system for the KSO Management System.

## Prerequisites

1. PHP 7.4 or higher
2. Composer (for installing dependencies)
3. MySQL database
4. WebSocket server (using Ratchet library)

## Installation Steps

### 1. Install Required Dependencies

The notification system relies on the Ratchet WebSocket library. Install it using Composer:

```bash
cd /path/to/adminlte
composer require cboden/ratchet
```

### 2. Database Setup

The notification system uses several database tables:

- `notifications` - Stores all notification data
- `notification_recipients` - Maps notifications to recipients
- `notification_settings` - Stores user notification preferences
- `admin_assignments` - Stores member assignments to admin users
- `member_approval_history` - Tracks assignment history and status changes

These tables should already be set up in your database. If not, you can find the schema in the `database/notifications.sql` file.

### 3. WebSocket Server Setup

The real-time notification system uses a WebSocket server to deliver notifications instantly. Follow these steps to set it up:

#### Windows:

1. Navigate to the WebSocket server directory:
   ```
   cd C:\Users\samue\OneDrive\Documents\public_html\adminlte\services\websocket
   ```

2. Run the WebSocket server by double-clicking the `start_websocket.bat` file or run it from the command line:
   ```
   start_websocket.bat
   ```

3. Alternatively, you can run the PHP script directly:
   ```
   php -q websocket_server.php
   ```

#### Linux/macOS:

1. Navigate to the WebSocket server directory:
   ```
   cd /path/to/adminlte/services/websocket
   ```

2. Make the manager script executable:
   ```
   chmod +x websocket_manager.php
   ```

3. Start the WebSocket server:
   ```
   ./websocket_manager.php start
   ```

4. You can also check the status, stop, or restart the server:
   ```
   ./websocket_manager.php status
   ./websocket_manager.php stop
   ./websocket_manager.php restart
   ```

### 4. Configure the Client

The notification client is automatically initialized in the header of the admin dashboard. It connects to the WebSocket server and handles incoming notifications.

## Usage

### Creating Notifications

You can create notifications in several ways:

1. **Through the UI**: Use the "Create Notification" button on the Notifications page.

2. **Programmatically**: Use the NotificationService class:

```php
// Include the NotificationService
require_once 'services/NotificationService.php';

// Create a new notification
$notificationService = new NotificationService($conn);
$notificationService->createNotification(
    'Title',
    'Message',
    'info', // Type: info, success, warning, error, etc.
    'normal', // Priority: low, normal, high, urgent
    'admins', // Target audience: all, admins, members, or specific
    $created_by_id, // Admin ID who created the notification
    null // Specific recipients (array of user IDs, optional)
);
```

### Member Assignments

You can assign members to admin users for review:

1. **Through the UI**: Use the "Create Assignment" button on the Assignments tab.

2. **Programmatically**: Use the AssignmentService class:

```php
// Include the AssignmentService
require_once 'services/AssignmentService.php';

// Create a new assignment
$assignmentService = new AssignmentService($conn);
$assignmentService->assignMember(
    $member_id,
    $admin_id, // Admin to assign to
    $assigned_by_id, // Admin creating the assignment
    'review', // Type: review, approval, verification, follow_up
    'normal', // Priority: low, normal, high, urgent
    $due_date, // Optional due date
    $notes // Optional notes
);
```

## Troubleshooting

### WebSocket Connection Issues

If you're having trouble with the WebSocket connection:

1. Ensure the WebSocket server is running. Check with:
   ```
   ./websocket_manager.php status
   ```

2. Verify that the WebSocket port (8080 by default) is not blocked by a firewall.

3. Check the WebSocket server logs for errors:
   ```
   cat services/websocket/websocket_server.log
   ```

### Notification Not Appearing

If notifications aren't appearing:

1. Check browser console for any JavaScript errors.

2. Verify that the user has permission to receive the notification.

3. Check the user's notification settings to ensure they haven't disabled the notification type.

4. Verify the WebSocket connection is established.

## Security Considerations

The WebSocket server includes basic authentication to ensure only authenticated users can connect. However, for production environments, consider the following additional security measures:

1. Use SSL/TLS for WebSocket connections (WSS protocol).

2. Configure a reverse proxy (like Nginx) to handle WebSocket connections.

3. Implement rate limiting for notification creation to prevent abuse.

4. Regularly audit the notification logs for unusual patterns.

## Maintenance

To keep the notification system running smoothly:

1. Periodically clean up old notifications to prevent database bloat.

2. Monitor WebSocket server logs for errors.

3. Update the Ratchet library periodically for security updates.

4. Restart the WebSocket server after PHP updates or server reboots.
