<?php
/**
 * PDF Receipt and Invoice Generator for KSO Chandigarh
 * Uses TCPDF library for professional PDF generation
 */

// Ensure CURL constants are defined if extension is not loaded
require_once 'curl_mock.php';

require_once 'TCPDF-main/tcpdf.php';

class ReceiptPDFGenerator extends TCPDF {
    
    protected $angle = 0;
    
    private $organizationInfo = [
        'name' => 'Kashmiri Students Organization',
        'location' => 'Chandigarh',
        'address' => 'Chandigarh, India',
        'phone' => '+91-XXXXX-XXXXX',
        'email' => 'info@ksochandigarh.com',
        'website' => 'https://ksochandigarh.com'
    ];
    
    private $receiptData = [];
    private $documentType = 'receipt'; // receipt or invoice
    
    public function __construct($orientation = 'P', $unit = 'mm', $format = 'A4') {
        parent::__construct($orientation, $unit, $format, true, 'UTF-8', false);
        
        // Set document information
        $this->SetCreator('KSO Chandigarh Portal');
        $this->SetAuthor('KSO Chandigarh');
        $this->SetTitle('Receipt/Invoice');
        $this->SetSubject('Payment Receipt/Invoice');
        $this->SetKeywords('KSO, Chandigarh, Receipt, Invoice, Payment');
        
        // Set default header/footer data
        $this->SetHeaderData('', 0, '', '', array(0,64,255), array(0,64,128));
        $this->setFooterData(array(0,64,0), array(0,64,128));
        
        // Set header and footer fonts
        $this->setHeaderFont(Array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));
        $this->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
        
        // Set default monospaced font
        $this->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);
        
        // Set margins
        $this->SetMargins(20, 30, 20);
        $this->SetHeaderMargin(10);
        $this->SetFooterMargin(10);
        
        // Set auto page breaks
        $this->SetAutoPageBreak(TRUE, 25);
        
        // Set image scale factor
        $this->setImageScale(PDF_IMAGE_SCALE_RATIO);
        
        // Set font
        $this->SetFont('helvetica', '', 10);
    }
    
    public function setReceiptData($data) {
        $this->receiptData = $data;
    }
    
    public function setDocumentType($type) {
        $this->documentType = $type;
    }
    
    public function Header() {
        // Organization logo and info
        $this->SetFont('helvetica', 'B', 16);
        $this->SetTextColor(0, 64, 128);
        $this->Cell(0, 10, $this->organizationInfo['name'], 0, 1, 'C');
        
        $this->SetFont('helvetica', '', 12);
        $this->SetTextColor(0, 0, 0);
        $this->Cell(0, 6, $this->organizationInfo['location'], 0, 1, 'C');
        
        $this->SetFont('helvetica', '', 10);
        $this->SetTextColor(100, 100, 100);
        $this->Cell(0, 5, $this->organizationInfo['address'], 0, 1, 'C');
        $this->Cell(0, 5, 'Phone: ' . $this->organizationInfo['phone'] . ' | Email: ' . $this->organizationInfo['email'], 0, 1, 'C');
        
        // Add some space
        $this->Ln(5);
        
        // Document type title
        $this->SetFont('helvetica', 'B', 14);
        $this->SetTextColor(0, 0, 0);
        $docTitle = ucfirst($this->documentType);
        $this->Cell(0, 10, $docTitle, 0, 1, 'C');
        
        // Add line
        $this->SetDrawColor(0, 64, 128);
        $this->Line(20, $this->GetY(), 190, $this->GetY());
        $this->Ln(5);
    }
    
    public function Footer() {
        $this->SetY(-15);
        $this->SetFont('helvetica', 'I', 8);
        $this->SetTextColor(100, 100, 100);
        
        // Footer text
        $footerText = 'Thank you for your payment. This is a computer-generated ' . $this->documentType . '.';
        $this->Cell(0, 10, $footerText, 0, 0, 'C');
        
        // Page number
        $this->Cell(0, 10, 'Page ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, 0, 'R');
    }
    
    public function generateReceipt() {
        $this->AddPage();
        
        // Receipt/Invoice details
        $this->SetFont('helvetica', 'B', 12);
        $this->SetTextColor(0, 0, 0);
        
        // Receipt/Invoice number and date
        $docNumber = $this->documentType === 'receipt' ? 'Receipt No.' : 'Invoice No.';
        $number = $this->receiptData['number'] ?? 'N/A';
        $date = $this->receiptData['date'] ?? date('Y-m-d');
        
        $this->Cell(85, 8, $docNumber . ': ' . $number, 1, 0, 'L');
        $this->Cell(85, 8, 'Date: ' . date('F j, Y', strtotime($date)), 1, 1, 'L');
        
        // Member information
        $this->Ln(5);
        $this->SetFont('helvetica', 'B', 11);
        $this->Cell(0, 8, 'Member Information', 0, 1, 'L');
        
        $this->SetFont('helvetica', '', 10);
        $memberInfo = [
            'Name: ' . ($this->receiptData['member_name'] ?? 'N/A'),
            'KSO ID: ' . ($this->receiptData['kso_id'] ?? 'N/A'),
            'Email: ' . ($this->receiptData['email'] ?? 'N/A'),
            'Phone: ' . ($this->receiptData['phone'] ?? 'N/A')
        ];
        
        foreach ($memberInfo as $info) {
            $this->Cell(0, 6, $info, 0, 1, 'L');
        }
        
        // Payment details
        $this->Ln(5);
        $this->SetFont('helvetica', 'B', 11);
        $this->Cell(0, 8, 'Payment Details', 0, 1, 'L');
        
        // Table header
        $this->SetFont('helvetica', 'B', 10);
        $this->SetFillColor(240, 240, 240);
        $this->Cell(80, 8, 'Description', 1, 0, 'L', true);
        $this->Cell(30, 8, 'Type', 1, 0, 'C', true);
        $this->Cell(30, 8, 'Method', 1, 0, 'C', true);
        $this->Cell(30, 8, 'Amount', 1, 1, 'R', true);
        
        // Table content
        $this->SetFont('helvetica', '', 10);
        $this->SetFillColor(255, 255, 255);
        
        if ($this->documentType === 'invoice' && isset($this->receiptData['line_items'])) {
            $total = 0;
            foreach ($this->receiptData['line_items'] as $item) {
                $this->Cell(80, 8, $item['description'], 1, 0, 'L');
                $this->Cell(30, 8, $item['type'] ?? 'Service', 1, 0, 'C');
                $this->Cell(30, 8, '-', 1, 0, 'C');
                $this->Cell(30, 8, '₹' . number_format($item['amount'], 2), 1, 1, 'R');
                $total += $item['amount'];
            }
            
            // Total
            $this->SetFont('helvetica', 'B', 10);
            $this->Cell(140, 8, 'Total Amount', 1, 0, 'R');
            $this->Cell(30, 8, '₹' . number_format($total, 2), 1, 1, 'R');
        } else {
            // Single payment entry
            $description = $this->receiptData['description'] ?? 'Payment';
            $type = $this->receiptData['transaction_type'] ?? 'Payment';
            $method = $this->receiptData['payment_method'] ?? 'Cash';
            $amount = $this->receiptData['amount'] ?? 0;
            
            $this->Cell(80, 8, $description, 1, 0, 'L');
            $this->Cell(30, 8, ucfirst(str_replace('_', ' ', $type)), 1, 0, 'C');
            $this->Cell(30, 8, ucfirst($method), 1, 0, 'C');
            $this->Cell(30, 8, '₹' . number_format($amount, 2), 1, 1, 'R');
            
            // Total
            $this->SetFont('helvetica', 'B', 10);
            $this->Cell(140, 8, 'Total Amount', 1, 0, 'R');
            $this->Cell(30, 8, '₹' . number_format($amount, 2), 1, 1, 'R');
        }
        
        // Amount in words
        $this->Ln(5);
        $amount = $this->receiptData['amount'] ?? 0;
        $amountInWords = $this->numberToWords($amount);
        $this->SetFont('helvetica', '', 10);
        $this->Cell(0, 8, 'Amount in Words: ' . $amountInWords, 0, 1, 'L');
        
        // Additional information
        if ($this->documentType === 'invoice') {
            $this->Ln(5);
            $this->SetFont('helvetica', 'B', 11);
            $this->Cell(0, 8, 'Payment Information', 0, 1, 'L');
            
            $this->SetFont('helvetica', '', 10);
            $dueDate = $this->receiptData['due_date'] ?? date('Y-m-d', strtotime('+30 days'));
            $this->Cell(0, 6, 'Due Date: ' . date('F j, Y', strtotime($dueDate)), 0, 1, 'L');
            $this->Cell(0, 6, 'Payment Terms: ' . ($this->receiptData['payment_terms'] ?? 'Net 30'), 0, 1, 'L');
        }
        
        // Notes
        if (!empty($this->receiptData['notes'])) {
            $this->Ln(5);
            $this->SetFont('helvetica', 'B', 11);
            $this->Cell(0, 8, 'Notes', 0, 1, 'L');
            
            $this->SetFont('helvetica', '', 10);
            $this->MultiCell(0, 6, $this->receiptData['notes'], 0, 'L');
        }
        
        // QR Code (if enabled)
        if (isset($this->receiptData['include_qr']) && $this->receiptData['include_qr']) {
            $this->addQRCode();
        }
        
        // Digital signature placeholder
        if (isset($this->receiptData['digital_signature']) && $this->receiptData['digital_signature']) {
            $this->addSignaturePlaceholder();
        }
        
        // Watermark for paid receipts
        if ($this->documentType === 'receipt' && isset($this->receiptData['status']) && $this->receiptData['status'] === 'paid') {
            $this->addPaidWatermark();
        }
    }
    
    private function addQRCode() {
        $this->Ln(10);
        $qrData = json_encode([
            'type' => $this->documentType,
            'number' => $this->receiptData['number'] ?? '',
            'amount' => $this->receiptData['amount'] ?? 0,
            'date' => $this->receiptData['date'] ?? date('Y-m-d'),
            'member_id' => $this->receiptData['kso_id'] ?? ''
        ]);
        
        // Generate QR code using built-in TCPDF method
        $this->write2DBarcode($qrData, 'QRCODE,L', 150, $this->GetY(), 30, 30, [], 'N');
        
        $this->SetXY(20, $this->GetY() + 5);
        $this->SetFont('helvetica', '', 8);
        $this->Cell(0, 5, 'Scan QR code to verify', 0, 1, 'L');
    }
    
    private function addSignaturePlaceholder() {
        $this->Ln(15);
        $this->SetDrawColor(0, 0, 0);
        $this->Line(130, $this->GetY(), 190, $this->GetY());
        $this->SetXY(130, $this->GetY() + 2);
        $this->SetFont('helvetica', '', 9);
        $this->Cell(60, 5, 'Authorized Signature', 0, 1, 'C');
    }
    
    private function addPaidWatermark() {
        $this->SetAlpha(0.3);
        $this->SetFont('helvetica', 'B', 60);
        $this->SetTextColor(0, 128, 0);
        $this->RotatedText(105, 150, 'PAID', 45);
        $this->SetAlpha(1);
        $this->SetTextColor(0, 0, 0);
    }
    
    private function numberToWords($number) {
        $ones = [
            '', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine',
            'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen',
            'Seventeen', 'Eighteen', 'Nineteen'
        ];
        
        $tens = [
            '', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'
        ];
        
        if ($number < 20) {
            return $ones[$number];
        } elseif ($number < 100) {
            return $tens[intval($number / 10)] . ' ' . $ones[$number % 10];
        } elseif ($number < 1000) {
            return $ones[intval($number / 100)] . ' Hundred ' . $this->numberToWords($number % 100);
        } elseif ($number < 100000) {
            return $this->numberToWords(intval($number / 1000)) . ' Thousand ' . $this->numberToWords($number % 1000);
        } elseif ($number < 10000000) {
            return $this->numberToWords(intval($number / 100000)) . ' Lakh ' . $this->numberToWords($number % 100000);
        } else {
            return $this->numberToWords(intval($number / 10000000)) . ' Crore ' . $this->numberToWords($number % 10000000);
        }
    }
    
    public function RotatedText($x, $y, $txt, $angle) {
        // Text rotation
        $this->Rotate($angle, $x, $y);
        $this->Text($x, $y, $txt);
        $this->Rotate(0);
    }
    
    public function Rotate($angle, $x = -1, $y = -1) {
        if ($x == -1) {
            $x = $this->x;
        }
        if ($y == -1) {
            $y = $this->y;
        }
        if ($this->angle != 0) {
            $this->_out('Q');
        }
        $this->angle = $angle;
        if ($angle != 0) {
            $angle *= M_PI / 180;
            $c = cos($angle);
            $s = sin($angle);
            $cx = $x * $this->k;
            $cy = ($this->h - $y) * $this->k;
            $this->_out(sprintf('q %.5F %.5F %.5F %.5F %.2F %.2F cm 1 0 0 1 %.2F %.2F cm', $c, $s, -$s, $c, $cx, $cy, -$cx, -$cy));
        }
    }
    
    public function _endpage() {
        if ($this->angle != 0) {
            $this->angle = 0;
            $this->_out('Q');
        }
        parent::_endpage();
    }
}

// Invoice PDF Generator Class
class InvoicePDFGenerator extends TCPDF {
    
    protected $angle = 0;
    
    public function __construct() {
        parent::__construct(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        
        // Document settings
        $this->SetCreator('KSO System');
        $this->SetAuthor('KSO Administration');
        $this->SetTitle('Invoice');
        $this->SetSubject('Invoice Document');
        $this->SetKeywords('invoice, kso, payment, billing');
        
        // Page settings
        $this->SetMargins(15, 25, 15);
        $this->SetHeaderMargin(10);
        $this->SetFooterMargin(10);
        $this->SetAutoPageBreak(true, 25);
        $this->setImageScale(PDF_IMAGE_SCALE_RATIO);
        
        // Default font
        $this->SetFont('helvetica', '', 10);
    }
    
    public function Header() {
        // Logo placeholder
        $this->SetXY(15, 15);
        $this->SetFont('helvetica', 'B', 16);
        $this->SetTextColor(0, 100, 150);
        $this->Cell(0, 10, 'KSO Organization', 0, 1, 'L');
        
        // Header line
        $this->SetDrawColor(0, 100, 150);
        $this->SetLineWidth(1);
        $this->Line(15, 28, 195, 28);
        
        // Company info
        $this->SetFont('helvetica', '', 9);
        $this->SetTextColor(80, 80, 80);
        $this->SetXY(15, 30);
        $this->Cell(0, 4, 'Address: KSO Headquarters, City, State 12345', 0, 1, 'L');
        $this->Cell(0, 4, 'Phone: (555) 123-4567 | Email: info@kso.com', 0, 1, 'L');
        
        // Invoice title
        $this->SetFont('helvetica', 'B', 20);
        $this->SetTextColor(0, 0, 0);
        $this->SetXY(140, 30);
        $this->Cell(50, 10, 'INVOICE', 0, 1, 'R');
    }
    
    public function Footer() {
        $this->SetY(-15);
        $this->SetFont('helvetica', 'I', 8);
        $this->SetTextColor(128, 128, 128);
        $this->Cell(0, 10, 'Page ' . $this->getAliasNumPage() . ' of ' . $this->getAliasNbPages(), 0, 0, 'C');
    }
    
    public function generateInvoice($data) {
        $this->AddPage();
        
        // Invoice details box
        $this->SetFont('helvetica', 'B', 11);
        $this->SetXY(140, 45);
        $this->SetFillColor(240, 240, 240);
        $this->Cell(50, 6, 'Invoice #: ' . $data['number'], 1, 1, 'L', true);
        
        $this->SetFont('helvetica', '', 10);
        $this->SetX(140);
        $this->Cell(50, 6, 'Date: ' . date('Y-m-d', strtotime($data['date'])), 1, 1, 'L', true);
        $this->SetX(140);
        $this->Cell(50, 6, 'Due Date: ' . date('Y-m-d', strtotime($data['due_date'])), 1, 1, 'L', true);
        $this->SetX(140);
        $this->Cell(50, 6, 'Status: ' . ucfirst($data['status']), 1, 1, 'L', true);
        
        // Bill to section
        $this->SetFont('helvetica', 'B', 12);
        $this->SetXY(15, 45);
        $this->Cell(0, 6, 'Bill To:', 0, 1, 'L');
        
        $this->SetFont('helvetica', '', 10);
        $this->SetX(15);
        $this->Cell(0, 5, $data['member_name'], 0, 1, 'L');
        $this->SetX(15);
        $this->Cell(0, 5, 'KSO ID: ' . $data['kso_id'], 0, 1, 'L');
        $this->SetX(15);
        $this->Cell(0, 5, 'Email: ' . $data['email'], 0, 1, 'L');
        $this->SetX(15);
        $this->Cell(0, 5, 'Phone: ' . $data['phone'], 0, 1, 'L');
        if (!empty($data['address'])) {
            $this->SetX(15);
            $this->Cell(0, 5, 'Address: ' . $data['address'], 0, 1, 'L');
        }
        
        // Items table
        $this->SetY(90);
        $this->SetFont('helvetica', 'B', 10);
        $this->SetFillColor(220, 220, 220);
        $this->SetTextColor(0, 0, 0);
        
        // Table headers
        $this->Cell(80, 8, 'Description', 1, 0, 'C', true);
        $this->Cell(25, 8, 'Quantity', 1, 0, 'C', true);
        $this->Cell(30, 8, 'Unit Price', 1, 0, 'C', true);
        $this->Cell(45, 8, 'Total', 1, 1, 'C', true);
        
        // Items
        $this->SetFont('helvetica', '', 10);
        $this->SetFillColor(255, 255, 255);
        
        foreach ($data['items'] as $item) {
            $this->Cell(80, 8, $item['description'], 1, 0, 'L', true);
            $this->Cell(25, 8, $item['quantity'], 1, 0, 'C', true);
            $this->Cell(30, 8, '$' . number_format($item['unit_price'], 2), 1, 0, 'R', true);
            $this->Cell(45, 8, '$' . number_format($item['total'], 2), 1, 1, 'R', true);
        }
        
        // Totals
        $this->SetFont('helvetica', 'B', 10);
        $this->SetFillColor(240, 240, 240);
        
        $this->Cell(135, 8, 'Subtotal:', 1, 0, 'R', true);
        $this->Cell(45, 8, '$' . number_format($data['subtotal'], 2), 1, 1, 'R', true);
        
        if ($data['tax_amount'] > 0) {
            $this->Cell(135, 8, 'Tax:', 1, 0, 'R', true);
            $this->Cell(45, 8, '$' . number_format($data['tax_amount'], 2), 1, 1, 'R', true);
        }
        
        $this->SetFont('helvetica', 'B', 12);
        $this->SetFillColor(200, 200, 200);
        $this->Cell(135, 10, 'Total Amount:', 1, 0, 'R', true);
        $this->Cell(45, 10, '$' . number_format($data['total_amount'], 2), 1, 1, 'R', true);
        
        // Amount in words
        $this->SetFont('helvetica', 'I', 10);
        $this->SetTextColor(80, 80, 80);
        $this->Ln(5);
        $this->Cell(0, 6, 'Amount in words: ' . $this->numberToWords($data['total_amount']) . ' dollars', 0, 1, 'L');
        
        // Notes
        if (!empty($data['notes'])) {
            $this->Ln(5);
            $this->SetFont('helvetica', 'B', 10);
            $this->SetTextColor(0, 0, 0);
            $this->Cell(0, 6, 'Notes:', 0, 1, 'L');
            $this->SetFont('helvetica', '', 10);
            $this->MultiCell(0, 6, $data['notes'], 0, 'L');
        }
        
        // Payment terms
        $this->Ln(10);
        $this->SetFont('helvetica', 'B', 10);
        $this->Cell(0, 6, 'Payment Terms & Conditions:', 0, 1, 'L');
        $this->SetFont('helvetica', '', 9);
        $this->MultiCell(0, 4, 
            "• Payment is due within 30 days of invoice date\n" .
            "• Late payments may incur additional charges\n" .
            "• Please reference invoice number when making payment\n" .
            "• For questions, contact our billing department", 0, 'L');
        
        // QR Code
        if ($data['include_qr']) {
            $qrData = "Invoice: " . $data['number'] . "\n" .
                     "Amount: $" . number_format($data['total_amount'], 2) . "\n" .
                     "Due: " . date('Y-m-d', strtotime($data['due_date'])) . "\n" .
                     "Member: " . $data['member_name'];
            
            $this->write2DBarcode($qrData, 'QRCODE,L', 15, 250, 25, 25, [], 'N');
        }
        
        // Digital signature placeholder
        if ($data['digital_signature']) {
            $this->SetXY(140, 250);
            $this->SetFont('helvetica', '', 9);
            $this->Cell(50, 6, 'Authorized Signature:', 0, 1, 'L');
            $this->SetX(140);
            $this->Cell(50, 15, '', 'B', 1, 'L');
            $this->SetX(140);
            $this->Cell(50, 6, 'Date: ' . date('Y-m-d'), 0, 1, 'L');
        }
        
        // Watermark for unpaid invoices
        if ($data['status'] !== 'paid') {
            $this->SetFont('helvetica', 'B', 50);
            $this->SetTextColor(255, 0, 0);
            $this->SetAlpha(0.3);
            $this->RotatedText(105, 150, strtoupper($data['status']), 45);
        }
    }
    
    // Helper method for number to words conversion
    public function numberToWords($number) {
        $ones = array(
            '', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine',
            'ten', 'eleven', 'twelve', 'thirteen', 'fourteen', 'fifteen', 'sixteen',
            'seventeen', 'eighteen', 'nineteen'
        );
        
        $tens = array(
            '', '', 'twenty', 'thirty', 'forty', 'fifty', 'sixty', 'seventy', 'eighty', 'ninety'
        );
        
        $hundreds = array('', 'thousand', 'million', 'billion', 'trillion');
        
        $number = number_format($number, 2, '.', '');
        list($integer, $fraction) = explode('.', $number);
        
        $output = '';
        
        if ($integer == '0') {
            $output = 'zero';
        } else {
            $integer = str_pad($integer, 15, '0', STR_PAD_LEFT);
            
            for ($i = 0; $i < 5; $i++) {
                $triplet = substr($integer, $i * 3, 3);
                $h = $triplet[0];
                $t = $triplet[1];
                $o = $triplet[2];
                
                if ($triplet != '000') {
                    $group = '';
                    
                    if ($h != '0') {
                        $group .= $ones[$h] . ' hundred ';
                    }
                    
                    if ($t == '1') {
                        $group .= $ones[$t . $o] . ' ';
                    } else {
                        $group .= $tens[$t] . ' ' . $ones[$o] . ' ';
                    }
                    
                    $group .= $hundreds[4 - $i] . ' ';
                    $output .= $group;
                }
            }
        }
        
        return trim($output);
    }
    
    // Helper method for rotated text
    public function RotatedText($x, $y, $txt, $angle) {
        $this->Rotate($angle, $x, $y);
        $this->Text($x, $y, $txt);
        $this->Rotate(0);
    }
    
    // Helper method for rotation
    public function Rotate($angle, $x = -1, $y = -1) {
        if ($x == -1) $x = $this->x;
        if ($y == -1) $y = $this->y;
        
        if ($this->angle != 0) {
            $this->_out('Q');
        }
        
        $this->angle = $angle;
        
        if ($angle != 0) {
            $angle *= M_PI / 180;
            $c = cos($angle);
            $s = sin($angle);
            $cx = $x * $this->k;
            $cy = ($this->h - $y) * $this->k;
            $this->_out(sprintf('q %.5F %.5F %.5F %.5F %.2F %.2F cm 1 0 0 1 %.2F %.2F cm', $c, $s, -$s, $c, $cx, $cy, -$cx, -$cy));
        }
    }
}

// Factory class for easier PDF generation
class PDFFactory {
    public static function outputForView($type, $data, $filename = null) {
        if ($type === 'receipt') {
            $pdf = new ReceiptPDFGenerator();
            $pdf->generateReceipt($data);
        } else {
            $pdf = new InvoicePDFGenerator();
            $pdf->generateInvoice($data);
        }
        
        $filename = $filename ?: $type . '_' . $data['number'] . '.pdf';
        $pdf->Output($filename, 'I');
    }
    
    public static function outputForDownload($type, $data, $filename = null) {
        if ($type === 'receipt') {
            $pdf = new ReceiptPDFGenerator();
            $pdf->generateReceipt($data);
        } else {
            $pdf = new InvoicePDFGenerator();
            $pdf->generateInvoice($data);
        }
        
        $filename = $filename ?: $type . '_' . $data['number'] . '.pdf';
        $pdf->Output($filename, 'D');
    }
    
    public static function outputForEmail($type, $data) {
        if ($type === 'receipt') {
            $pdf = new ReceiptPDFGenerator();
            $pdf->generateReceipt($data);
        } else {
            $pdf = new InvoicePDFGenerator();
            $pdf->generateInvoice($data);
        }
        
        return $pdf->Output('', 'S');
    }
}
?>
