/**
 * KSO Chandigarh Print System
 * Enhanced printing functionality for reports, member data, and documents
 */

class KSOPrintSystem {
    constructor() {
        this.printStyles = '';
        this.init();
    }

    init() {
        this.addPrintStyles();
        this.createPrintButtons();
    }

    addPrintStyles() {
        const style = document.createElement('style');
        style.textContent = `
            /* Print Styles */
            @media print {
                * {
                    -webkit-print-color-adjust: exact !important;
                    color-adjust: exact !important;
                }

                body {
                    margin: 0;
                    padding: 0;
                    font-family: Arial, sans-serif;
                    font-size: 12px;
                    line-height: 1.4;
                }

                .no-print {
                    display: none !important;
                }

                .print-only {
                    display: block !important;
                }

                .main-header,
                .main-sidebar,
                .main-footer,
                .content-header,
                .breadcrumb,
                .btn,
                .form-control,
                .pagination,
                .modal,
                .toast-container,
                .preloader {
                    display: none !important;
                }

                .content-wrapper {
                    margin-left: 0 !important;
                    padding: 0 !important;
                }

                .kso-print-header {
                    text-align: center;
                    margin-bottom: 30px;
                    padding-bottom: 20px;
                    border-bottom: 2px solid #333;
                }

                .kso-print-logo {
                    max-width: 100px;
                    height: auto;
                    margin-bottom: 10px;
                }

                .kso-print-title {
                    font-size: 24px;
                    font-weight: bold;
                    color: #333;
                    margin: 10px 0;
                }

                .kso-print-subtitle {
                    font-size: 16px;
                    color: #666;
                    margin: 5px 0;
                }

                .kso-print-date {
                    font-size: 12px;
                    color: #999;
                    margin-top: 10px;
                }

                .kso-print-footer {
                    position: fixed;
                    bottom: 0;
                    left: 0;
                    right: 0;
                    text-align: center;
                    font-size: 10px;
                    color: #666;
                    padding: 10px;
                    border-top: 1px solid #ddd;
                }

                .kso-print-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-bottom: 20px;
                }

                .kso-print-table th,
                .kso-print-table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                    font-size: 11px;
                }

                .kso-print-table th {
                    background-color: #f8f9fa;
                    font-weight: bold;
                }

                .kso-print-section {
                    margin-bottom: 25px;
                    page-break-inside: avoid;
                }

                .kso-print-section-title {
                    font-size: 16px;
                    font-weight: bold;
                    margin-bottom: 10px;
                    color: #333;
                    border-bottom: 1px solid #ddd;
                    padding-bottom: 5px;
                }

                .kso-print-member-card {
                    border: 1px solid #ddd;
                    padding: 15px;
                    margin-bottom: 15px;
                    page-break-inside: avoid;
                }

                .kso-print-member-photo {
                    float: right;
                    width: 80px;
                    height: 80px;
                    border: 1px solid #ddd;
                    margin-left: 15px;
                }

                .kso-print-qr-code {
                    text-align: center;
                    margin-top: 15px;
                }

                .kso-print-qr-code img {
                    max-width: 100px;
                    height: auto;
                }

                .page-break {
                    page-break-after: always;
                }

                .kso-print-watermark {
                    position: fixed;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%) rotate(-45deg);
                    font-size: 72px;
                    color: rgba(0, 0, 0, 0.05);
                    z-index: -1;
                    font-weight: bold;
                }
            }

            /* Print Button Styles */
            .kso-print-btn {
                background: #007bff;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
                cursor: pointer;
                font-size: 14px;
                margin: 5px;
                display: inline-flex;
                align-items: center;
                gap: 5px;
                transition: background-color 0.2s ease;
            }

            .kso-print-btn:hover {
                background: #0056b3;
            }

            .kso-print-btn i {
                font-size: 16px;
            }

            .kso-print-options {
                position: fixed;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                background: white;
                border: 1px solid #ddd;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                padding: 20px;
                z-index: 9999;
                min-width: 300px;
                display: none;
            }

            .kso-print-options.show {
                display: block;
            }

            .kso-print-options-overlay {
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.5);
                z-index: 9998;
                display: none;
            }

            .kso-print-options-overlay.show {
                display: block;
            }

            .kso-print-options-title {
                font-size: 18px;
                font-weight: bold;
                margin-bottom: 15px;
                color: #333;
            }

            .kso-print-options-form {
                display: flex;
                flex-direction: column;
                gap: 15px;
            }

            .kso-print-options-group {
                display: flex;
                flex-direction: column;
                gap: 5px;
            }

            .kso-print-options-label {
                font-size: 14px;
                font-weight: 500;
                color: #333;
            }

            .kso-print-options-input {
                padding: 8px;
                border: 1px solid #ddd;
                border-radius: 4px;
                font-size: 14px;
            }

            .kso-print-options-checkbox {
                display: flex;
                align-items: center;
                gap: 8px;
            }

            .kso-print-options-actions {
                display: flex;
                gap: 10px;
                justify-content: flex-end;
                margin-top: 15px;
            }

            .kso-print-options-btn {
                padding: 8px 16px;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                font-size: 14px;
                transition: background-color 0.2s ease;
            }

            .kso-print-options-btn.primary {
                background: #007bff;
                color: white;
            }

            .kso-print-options-btn.primary:hover {
                background: #0056b3;
            }

            .kso-print-options-btn.secondary {
                background: #6c757d;
                color: white;
            }

            .kso-print-options-btn.secondary:hover {
                background: #545b62;
            }
        `;
        document.head.appendChild(style);
    }

    createPrintButtons() {
        // Add print buttons to existing tables and sections
        const tables = document.querySelectorAll('table');
        tables.forEach(table => {
            const parent = table.closest('.card, .kso-widget');
            if (parent && !parent.querySelector('.kso-print-btn')) {
                const header = parent.querySelector('.card-header, .kso-widget-header');
                if (header) {
                    const printBtn = document.createElement('button');
                    printBtn.className = 'kso-print-btn no-print';
                    printBtn.innerHTML = '<i class="fas fa-print"></i> Print';
                    printBtn.onclick = () => this.showPrintOptions(table);
                    
                    const tools = header.querySelector('.card-tools, .kso-widget-actions');
                    if (tools) {
                        tools.appendChild(printBtn);
                    } else {
                        header.appendChild(printBtn);
                    }
                }
            }
        });
    }

    showPrintOptions(element) {
        const overlay = document.createElement('div');
        overlay.className = 'kso-print-options-overlay';
        overlay.onclick = () => this.closePrintOptions();

        const modal = document.createElement('div');
        modal.className = 'kso-print-options';
        modal.innerHTML = `
            <div class="kso-print-options-title">Print Options</div>
            <div class="kso-print-options-form">
                <div class="kso-print-options-group">
                    <label class="kso-print-options-label">Document Title</label>
                    <input type="text" class="kso-print-options-input" id="print-title" value="KSO Chandigarh Report">
                </div>
                <div class="kso-print-options-group">
                    <label class="kso-print-options-label">Report Type</label>
                    <select class="kso-print-options-input" id="print-type">
                        <option value="table">Table Report</option>
                        <option value="member-cards">Member Cards</option>
                        <option value="financial">Financial Report</option>
                        <option value="activity">Activity Report</option>
                    </select>
                </div>
                <div class="kso-print-options-group">
                    <div class="kso-print-options-checkbox">
                        <input type="checkbox" id="print-header" checked>
                        <label for="print-header">Include Header</label>
                    </div>
                </div>
                <div class="kso-print-options-group">
                    <div class="kso-print-options-checkbox">
                        <input type="checkbox" id="print-footer" checked>
                        <label for="print-footer">Include Footer</label>
                    </div>
                </div>
                <div class="kso-print-options-group">
                    <div class="kso-print-options-checkbox">
                        <input type="checkbox" id="print-watermark">
                        <label for="print-watermark">Add Watermark</label>
                    </div>
                </div>
                <div class="kso-print-options-group">
                    <div class="kso-print-options-checkbox">
                        <input type="checkbox" id="print-qr" checked>
                        <label for="print-qr">Include QR Codes (Member Cards)</label>
                    </div>
                </div>
            </div>
            <div class="kso-print-options-actions">
                <button class="kso-print-options-btn secondary" onclick="window.ksoPrint.closePrintOptions()">Cancel</button>
                <button class="kso-print-options-btn primary" onclick="window.ksoPrint.executePrint()">Print</button>
            </div>
        `;

        document.body.appendChild(overlay);
        document.body.appendChild(modal);
        
        overlay.classList.add('show');
        modal.classList.add('show');
        
        this.currentElement = element;
    }

    closePrintOptions() {
        const overlay = document.querySelector('.kso-print-options-overlay');
        const modal = document.querySelector('.kso-print-options');
        
        if (overlay) overlay.remove();
        if (modal) modal.remove();
    }

    executePrint() {
        const title = document.getElementById('print-title').value;
        const type = document.getElementById('print-type').value;
        const includeHeader = document.getElementById('print-header').checked;
        const includeFooter = document.getElementById('print-footer').checked;
        const includeWatermark = document.getElementById('print-watermark').checked;
        const includeQR = document.getElementById('print-qr').checked;

        this.closePrintOptions();

        switch (type) {
            case 'table':
                this.printTable(this.currentElement, { title, includeHeader, includeFooter, includeWatermark });
                break;
            case 'member-cards':
                this.printMemberCards({ title, includeHeader, includeFooter, includeWatermark, includeQR });
                break;
            case 'financial':
                this.printFinancialReport({ title, includeHeader, includeFooter, includeWatermark });
                break;
            case 'activity':
                this.printActivityReport({ title, includeHeader, includeFooter, includeWatermark });
                break;
        }
    }

    printTable(table, options = {}) {
        const printWindow = window.open('', '_blank');
        const tableClone = table.cloneNode(true);
        
        // Clean up table for printing
        tableClone.className = 'kso-print-table';
        const actionColumns = tableClone.querySelectorAll('th:last-child, td:last-child');
        actionColumns.forEach(col => {
            if (col.textContent.includes('Actions') || col.querySelector('button')) {
                col.style.display = 'none';
            }
        });

        const printContent = this.generatePrintDocument(tableClone.outerHTML, options);
        
        printWindow.document.write(printContent);
        printWindow.document.close();
        printWindow.print();
    }

    printMemberCards(options = {}) {
        // This would typically fetch member data from your API
        const memberCards = this.generateMemberCards(options);
        const printContent = this.generatePrintDocument(memberCards, options);
        
        const printWindow = window.open('', '_blank');
        printWindow.document.write(printContent);
        printWindow.document.close();
        printWindow.print();
    }

    printFinancialReport(options = {}) {
        const financialData = this.generateFinancialReport(options);
        const printContent = this.generatePrintDocument(financialData, options);
        
        const printWindow = window.open('', '_blank');
        printWindow.document.write(printContent);
        printWindow.document.close();
        printWindow.print();
    }

    printActivityReport(options = {}) {
        const activityData = this.generateActivityReport(options);
        const printContent = this.generatePrintDocument(activityData, options);
        
        const printWindow = window.open('', '_blank');
        printWindow.document.write(printContent);
        printWindow.document.close();
        printWindow.print();
    }

    generatePrintDocument(content, options = {}) {
        const header = options.includeHeader ? `
            <div class="kso-print-header">
                <img src="dist/img/kso-logo.png" alt="KSO Logo" class="kso-print-logo">
                <div class="kso-print-title">Kashmiri Student Organization</div>
                <div class="kso-print-subtitle">Chandigarh</div>
                <div class="kso-print-subtitle">${options.title || 'Report'}</div>
                <div class="kso-print-date">Generated on: ${new Date().toLocaleDateString()}</div>
            </div>
        ` : '';

        const footer = options.includeFooter ? `
            <div class="kso-print-footer">
                <div>© ${new Date().getFullYear()} KSO Chandigarh. All rights reserved.</div>
                <div>This document is computer generated and does not require signature.</div>
            </div>
        ` : '';

        const watermark = options.includeWatermark ? `
            <div class="kso-print-watermark">KSO CHANDIGARH</div>
        ` : '';

        return `
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset="utf-8">
                <title>${options.title || 'KSO Chandigarh Report'}</title>
                <style>
                    ${this.getPrintStyles()}
                </style>
            </head>
            <body>
                ${watermark}
                ${header}
                <div class="kso-print-content">
                    ${content}
                </div>
                ${footer}
            </body>
            </html>
        `;
    }

    generateMemberCards(options = {}) {
        // This is a placeholder - you would fetch real member data
        const members = [
            {
                id: 'KSO001',
                name: 'John Doe',
                email: 'john@example.com',
                phone: '+91 98765 43210',
                college: 'DAV College',
                course: 'B.Tech CSE',
                membershipType: 'Individual',
                photo: 'dist/img/user-placeholder.png'
            }
            // Add more members...
        ];

        return members.map(member => `
            <div class="kso-print-member-card">
                <img src="${member.photo}" alt="${member.name}" class="kso-print-member-photo">
                <div>
                    <strong>Member ID:</strong> ${member.id}<br>
                    <strong>Name:</strong> ${member.name}<br>
                    <strong>Email:</strong> ${member.email}<br>
                    <strong>Phone:</strong> ${member.phone}<br>
                    <strong>College:</strong> ${member.college}<br>
                    <strong>Course:</strong> ${member.course}<br>
                    <strong>Membership:</strong> ${member.membershipType}
                </div>
                ${options.includeQR ? `
                    <div class="kso-print-qr-code">
                        <img src="https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=${member.id}" alt="QR Code">
                    </div>
                ` : ''}
            </div>
        `).join('');
    }

    generateFinancialReport(options = {}) {
        return `
            <div class="kso-print-section">
                <div class="kso-print-section-title">Financial Summary</div>
                <table class="kso-print-table">
                    <tr>
                        <th>Category</th>
                        <th>Amount (₹)</th>
                        <th>Status</th>
                    </tr>
                    <tr>
                        <td>Membership Fees</td>
                        <td>₹50,000</td>
                        <td>Collected</td>
                    </tr>
                    <tr>
                        <td>Event Revenue</td>
                        <td>₹25,000</td>
                        <td>Collected</td>
                    </tr>
                    <tr>
                        <td>Operational Expenses</td>
                        <td>₹15,000</td>
                        <td>Paid</td>
                    </tr>
                    <tr>
                        <td>Event Expenses</td>
                        <td>₹20,000</td>
                        <td>Paid</td>
                    </tr>
                    <tr>
                        <td><strong>Net Balance</strong></td>
                        <td><strong>₹40,000</strong></td>
                        <td><strong>Available</strong></td>
                    </tr>
                </table>
            </div>
        `;
    }

    generateActivityReport(options = {}) {
        return `
            <div class="kso-print-section">
                <div class="kso-print-section-title">Recent Activities</div>
                <div class="kso-print-activity-item">
                    <strong>2024-01-15 10:30 AM</strong> - New member registration: John Doe
                </div>
                <div class="kso-print-activity-item">
                    <strong>2024-01-14 03:45 PM</strong> - Event "Cultural Night" created
                </div>
                <div class="kso-print-activity-item">
                    <strong>2024-01-14 11:20 AM</strong> - Membership fee payment received: ₹1,000
                </div>
                <div class="kso-print-activity-item">
                    <strong>2024-01-13 09:15 AM</strong> - System backup completed successfully
                </div>
            </div>
        `;
    }

    getPrintStyles() {
        return `
            @media print {
                * { -webkit-print-color-adjust: exact !important; color-adjust: exact !important; }
                body { margin: 0; padding: 20px; font-family: Arial, sans-serif; font-size: 12px; line-height: 1.4; }
                .kso-print-header { text-align: center; margin-bottom: 30px; padding-bottom: 20px; border-bottom: 2px solid #333; }
                .kso-print-logo { max-width: 100px; height: auto; margin-bottom: 10px; }
                .kso-print-title { font-size: 24px; font-weight: bold; color: #333; margin: 10px 0; }
                .kso-print-subtitle { font-size: 16px; color: #666; margin: 5px 0; }
                .kso-print-date { font-size: 12px; color: #999; margin-top: 10px; }
                .kso-print-footer { position: fixed; bottom: 20px; left: 0; right: 0; text-align: center; font-size: 10px; color: #666; padding: 10px; border-top: 1px solid #ddd; }
                .kso-print-table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
                .kso-print-table th, .kso-print-table td { border: 1px solid #ddd; padding: 8px; text-align: left; font-size: 11px; }
                .kso-print-table th { background-color: #f8f9fa; font-weight: bold; }
                .kso-print-section { margin-bottom: 25px; page-break-inside: avoid; }
                .kso-print-section-title { font-size: 16px; font-weight: bold; margin-bottom: 10px; color: #333; border-bottom: 1px solid #ddd; padding-bottom: 5px; }
                .kso-print-member-card { border: 1px solid #ddd; padding: 15px; margin-bottom: 15px; page-break-inside: avoid; }
                .kso-print-member-photo { float: right; width: 80px; height: 80px; border: 1px solid #ddd; margin-left: 15px; }
                .kso-print-qr-code { text-align: center; margin-top: 15px; }
                .kso-print-qr-code img { max-width: 100px; height: auto; }
                .kso-print-watermark { position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%) rotate(-45deg); font-size: 72px; color: rgba(0, 0, 0, 0.05); z-index: -1; font-weight: bold; }
                .kso-print-activity-item { margin-bottom: 10px; padding: 5px 0; border-bottom: 1px solid #eee; }
            }
        `;
    }

    // Quick print methods
    quickPrint(selector) {
        const element = document.querySelector(selector);
        if (element) {
            this.printTable(element, { title: 'Quick Print', includeHeader: true, includeFooter: true });
        }
    }

    printPage() {
        window.print();
    }

    exportToPDF(element, filename = 'kso-report.pdf') {
        // This would require a PDF library like jsPDF or html2pdf
        console.log('PDF export functionality would go here');
    }
}

// Initialize print system
document.addEventListener('DOMContentLoaded', () => {
    window.ksoPrint = new KSOPrintSystem();
});
