# 🔄 KSO Chandigarh - Migration Guide for File References

## 📝 **CRITICAL PATH UPDATES NEEDED**

### **1. API Endpoint References (JavaScript Files)**

#### **Admin Frontend JS Files** (`admin/frontend/js/`)
Update these files to use new API paths:

```javascript
// OLD REFERENCES:
const response = await fetch('/api/members/index.php');
const response = await fetch('/api/auth/login.php');
const response = await fetch('/api/dashboard.php');

// NEW REFERENCES:
const response = await fetch('/admin/api/members/index.php');
const response = await fetch('/admin/api/auth/login.php');
const response = await fetch('/admin/api/dashboard/dashboard.php');
```

#### **Files to Update:**
- `admin/frontend/js/dashboard.js`
- `admin/frontend/js/members.js`
- `admin/frontend/js/events.js`
- `admin/frontend/js/finance.js`
- `admin/frontend/js/reports.js`

### **2. Include Path Updates (PHP Files)**

#### **Admin API Files** (`admin/api/*/`)
```php
// OLD INCLUDES:
require_once('../../config.php');
require_once('../services/EmailService.php');

// NEW INCLUDES:
require_once('../../../shared/api/config.php');
require_once('../../../shared/services/EmailService.php');
```

#### **Member API Files** (`member/api/*/`)
```php
// OLD INCLUDES:
require_once('../../config.php');

// NEW INCLUDES:
require_once('../../../shared/api/config.php');
```

### **3. Asset Path Updates (HTML Files)**

#### **Admin Pages** (`admin/frontend/pages/`)
```html
<!-- OLD ASSET PATHS: -->
<link href="../../frontend/assets/css/adminlte.min.css" rel="stylesheet">
<script src="../../frontend/js/dashboard.js"></script>

<!-- NEW ASSET PATHS: -->
<link href="../css/admin.css" rel="stylesheet">
<script src="../js/dashboard.js"></script>
```

#### **Member Pages** (`member/frontend/pages/`)
```html
<!-- OLD ASSET PATHS: -->
<link href="../frontend/assets/css/member.css" rel="stylesheet">
<script src="../frontend/js/member.js"></script>

<!-- NEW ASSET PATHS: -->
<link href="../css/member.css" rel="stylesheet">
<script src="../js/member.js"></script>
```

---

## 🛠️ **AUTOMATION SCRIPT**

### **PowerShell Script for Bulk Updates**

```powershell
# Navigate to the access directory
cd "c:\Users\samue\OneDrive\Documents\public_html\access"

# Update API references in admin JS files
$adminJSFiles = Get-ChildItem "admin\frontend\js\*.js" -Recurse
foreach ($file in $adminJSFiles) {
    $content = Get-Content $file.FullName -Raw
    $content = $content -replace "/api/members/", "/admin/api/members/"
    $content = $content -replace "/api/auth/", "/admin/api/auth/"
    $content = $content -replace "/api/dashboard\.php", "/admin/api/dashboard/dashboard.php"
    $content = $content -replace "/api/events/", "/admin/api/events/"
    $content = $content -replace "/api/finance/", "/admin/api/finance/"
    $content = $content -replace "/api/reports/", "/admin/api/reports/"
    $content = $content -replace "/api/notifications/", "/admin/api/notifications/"
    Set-Content $file.FullName $content
    Write-Host "Updated: $($file.Name)"
}

# Update includes in admin API files
$adminAPIFiles = Get-ChildItem "admin\api\*.php" -Recurse
foreach ($file in $adminAPIFiles) {
    $content = Get-Content $file.FullName -Raw
    $content = $content -replace "require_once\('\.\.\/\.\.\/config\.php'\);", "require_once('../../../shared/api/config.php');"
    $content = $content -replace "require_once\('\.\.\/services\/", "require_once('../../../shared/services/"
    Set-Content $file.FullName $content
    Write-Host "Updated: $($file.Name)"
}

Write-Host "Migration script completed!"
```

---

## 📋 **MANUAL VERIFICATION CHECKLIST**

### **1. Test Admin Dashboard**
- [ ] Dashboard loads without errors
- [ ] Member statistics display correctly
- [ ] Recent activities show properly
- [ ] Navigation links work

### **2. Test Member Management**
- [ ] Member list loads
- [ ] Member approval functions work
- [ ] Member search functions
- [ ] Member statistics accurate

### **3. Test Member Portal**
- [ ] Member login works
- [ ] Member dashboard loads
- [ ] Member profile accessible
- [ ] Member notifications work

### **4. Test API Endpoints**
- [ ] Admin APIs respond correctly
- [ ] Member APIs function properly
- [ ] Shared services accessible
- [ ] Authentication working

### **5. Test File Operations**
- [ ] File uploads work
- [ ] Document downloads function
- [ ] Photo uploads successful
- [ ] Storage paths correct

---

## 🚨 **TROUBLESHOOTING COMMON ISSUES**

### **Issue 1: 404 Errors on API Calls**
```
Problem: JavaScript making calls to old API paths
Solution: Update API endpoints in JS files using the script above
```

### **Issue 2: Include Path Errors**
```
Problem: PHP files can't find included files
Solution: Update require_once paths to use new directory structure
```

### **Issue 3: Asset Not Loading**
```
Problem: CSS/JS files not loading in HTML pages
Solution: Update asset paths in HTML files to match new structure
```

### **Issue 4: Database Connection Issues**
```
Problem: Database config not found
Solution: Ensure shared/api/config.php has correct database settings
```

---

## 🎯 **IMPLEMENTATION PRIORITY**

### **Phase 1 (Critical - Do First)**
1. Update API endpoints in admin JavaScript files
2. Test admin dashboard functionality
3. Update PHP include paths in admin APIs
4. Verify member portal login

### **Phase 2 (Important - Do Second)**
1. Update asset paths in HTML files
2. Test all admin management functions
3. Verify member portal functionality
4. Update member API references

### **Phase 3 (Optimization - Do Last)**
1. Clean up legacy file references
2. Optimize asset loading
3. Create new service classes
4. Document new structure

---

## ✅ **SUCCESS INDICATORS**

You'll know the migration is successful when:
1. **No 404 errors** in browser console
2. **Admin dashboard** loads and functions properly
3. **Member portal** accessible and working
4. **All API calls** return expected data
5. **File operations** work without errors

This migration guide ensures a smooth transition to the new, organized structure while maintaining full functionality.
