<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

// Require admin authentication for dashboard
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            getDashboardStats();
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    error_log("Error in dashboard.php: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

function getDashboardStats() {
    global $pdo;
    
    try {
        $stats = [];
        
        // Members Statistics
        $stmt = $pdo->query("
            SELECT 
                COUNT(*) as total_members,
                SUM(CASE WHEN approval_status = 'approved' THEN 1 ELSE 0 END) as approved_members,
                SUM(CASE WHEN approval_status = 'pending' THEN 1 ELSE 0 END) as pending_approvals,
                SUM(CASE WHEN membership_type = 'Family' THEN 1 ELSE 0 END) as family_memberships,
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_members,
                SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as approved_this_month
            FROM members
        ");
        $memberStats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Events Statistics
        $stmt = $pdo->query("
            SELECT 
                COUNT(*) as total_events,
                SUM(CASE WHEN status IN ('upcoming', 'ongoing') AND event_date >= CURDATE() THEN 1 ELSE 0 END) as active_events,
                SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as events_this_month
            FROM events
        ");
        $eventStats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Financial Statistics (Income from finance table)
        $stmt = $pdo->query("
            SELECT 
                COALESCE(SUM(amount), 0) as total_income,
                COALESCE(SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN amount ELSE 0 END), 0) as income_this_month
            FROM finance
            WHERE status = 'completed'
        ");
        $incomeStats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Expenses Statistics
        $stmt = $pdo->query("
            SELECT 
                COALESCE(SUM(amount), 0) as total_expenses,
                COALESCE(SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN amount ELSE 0 END), 0) as expenses_this_month
            FROM expenses
            WHERE status IN ('approved', 'paid')
        ");
        $expenseStats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Membership Fees Statistics (from finance table)
        $stmt = $pdo->query("
            SELECT 
                COALESCE(SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END), 0) as total_fees_collected,
                COALESCE(SUM(CASE WHEN status = 'pending' THEN amount ELSE 0 END), 0) as pending_fees,
                COALESCE(SUM(CASE WHEN status = 'failed' THEN amount ELSE 0 END), 0) as overdue_fees
            FROM finance
            WHERE transaction_type = 'membership_fee'
        ");
        $feeStats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Recent Activities (last 10)
        $stmt = $pdo->prepare("
            SELECT 
                'member' as type,
                CONCAT(first_name, ' ', last_name) as title,
                'New member registration' as description,
                created_at as date
            FROM members 
            WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
            UNION ALL
            SELECT 
                'event' as type,
                event_title as title,
                'New event created' as description,
                created_at as date
            FROM events 
            WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
            UNION ALL
            SELECT 
                'finance' as type,
                description as title,
                CONCAT(transaction_type, ' transaction') as description,
                created_at as date
            FROM finance 
            WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
            UNION ALL
            SELECT 
                'expense' as type,
                expense_title as title,
                'New expense recorded' as description,
                created_at as date
            FROM expenses 
            WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
            ORDER BY date DESC 
            LIMIT 10
        ");
        $stmt->execute();
        $recentActivities = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Recent Members (last 8)
        $stmt = $pdo->prepare("
            SELECT 
                id,
                kso_id,
                first_name,
                last_name,
                email,
                phone,
                approval_status,
                status,
                created_at,
                photo_url
            FROM members 
            ORDER BY created_at DESC 
            LIMIT 8
        ");
        $stmt->execute();
        $recentMembers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Monthly data for charts (last 12 months)
        $monthlyData = [];
        for ($i = 11; $i >= 0; $i--) {
            $month = date('Y-m', strtotime("-$i months"));
            $monthName = date('M Y', strtotime("-$i months"));
            
            // Members joined this month
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as count 
                FROM members 
                WHERE DATE_FORMAT(created_at, '%Y-%m') = ?
            ");
            $stmt->execute([$month]);
            $memberCount = $stmt->fetchColumn();
            
            // Revenue this month (from finance table)
            $stmt = $pdo->prepare("
                SELECT COALESCE(SUM(amount), 0) as total 
                FROM finance 
                WHERE status = 'completed' AND DATE_FORMAT(created_at, '%Y-%m') = ?
            ");
            $stmt->execute([$month]);
            $revenue = $stmt->fetchColumn();
            
            $monthlyData[] = [
                'month' => $monthName,
                'members' => (int)$memberCount,
                'revenue' => (float)$revenue
            ];
        }
        
        // Compile all statistics
        $stats = [
            'members' => [
                'total' => (int)$memberStats['total_members'],
                'approved' => (int)$memberStats['approved_members'],
                'pending_approvals' => (int)$memberStats['pending_approvals'],
                'family_memberships' => (int)$memberStats['family_memberships'],
                'active' => (int)$memberStats['active_members'],
                'approved_this_month' => (int)$memberStats['approved_this_month']
            ],
            'events' => [
                'total' => (int)$eventStats['total_events'],
                'active' => (int)$eventStats['active_events'],
                'this_month' => (int)$eventStats['events_this_month']
            ],
            'finance' => [
                'total_income' => (float)$incomeStats['total_income'],
                'total_expenses' => (float)$expenseStats['total_expenses'],
                'total_revenue' => (float)$incomeStats['total_income'] - (float)$expenseStats['total_expenses'],
                'income_this_month' => (float)$incomeStats['income_this_month'],
                'expenses_this_month' => (float)$expenseStats['expenses_this_month']
            ],
            'fees' => [
                'total_collected' => (float)$feeStats['total_fees_collected'],
                'pending' => (float)$feeStats['pending_fees'],
                'overdue' => (float)$feeStats['overdue_fees']
            ],
            'recent_activities' => $recentActivities,
            'recent_members' => $recentMembers,
            'monthly_data' => $monthlyData
        ];
        
        sendResponse(200, true, 'Dashboard statistics retrieved successfully', $stats);
        
    } catch (Exception $e) {
        error_log("Error getting dashboard stats: " . $e->getMessage());
        sendResponse(500, false, 'Failed to retrieve dashboard statistics');
    }
}
?>


