<?php
/**
 * Dashboard Statistics API
 * Returns dashboard data for AdminLTE frontend
 */

define('API_ACCESS', true);
require_once '../../../shared/api/config.php';

// Require admin authentication
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'GET') {
    sendResponse(405, false, 'Method not allowed');
}

try {
    $stats = [];
    
    // Total Members
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM members WHERE status = 'active'");
    $stmt->execute();
    $stats['total_members'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    // Total Events
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM events WHERE status != 'cancelled'");
    $stmt->execute();
    $stats['total_events'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    // Pending Approvals
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM members WHERE approval_status = 'pending'");
    $stmt->execute();
    $stats['pending_approvals'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    // Recent Revenue (this month)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(amount), 0) as total 
        FROM finance_transactions 
        WHERE type = 'income' 
        AND DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(NOW(), '%Y-%m')
    ");
    $stmt->execute();
    $stats['recent_revenue'] = (float)$stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Active Members (last 30 days)
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM members 
        WHERE status = 'active' 
        AND last_login >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    ");
    $stmt->execute();
    $stats['active_members'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    // Recent Members (last 7 days)
    $stmt = $pdo->prepare("
        SELECT 
            id,
            first_name, 
            last_name, 
            kso_id,
            DATE_FORMAT(created_at, '%M %d, %Y') as formatted_date,
            DATE_FORMAT(created_at, '%Y-%m-%d') as date_only
        FROM members 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) 
        ORDER BY created_at DESC 
        LIMIT 5
    ");
    $stmt->execute();
    $stats['recent_members'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Upcoming Events (next 30 days)
    $stmt = $pdo->prepare("
        SELECT 
            id,
            title, 
            description,
            event_date,
            DATE_FORMAT(event_date, '%M %d, %Y') as formatted_date,
            status
        FROM events 
        WHERE event_date >= CURDATE() 
        AND event_date <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)
        AND status = 'active'
        ORDER BY event_date ASC 
        LIMIT 5
    ");
    $stmt->execute();
    $stats['upcoming_events'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Monthly Growth Stats
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(NOW(), '%Y-%m') THEN 1 END) as this_month,
            COUNT(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(NOW(), INTERVAL 1 MONTH), '%Y-%m') THEN 1 END) as last_month
        FROM members 
        WHERE status = 'active'
    ");
    $stmt->execute();
    $growth_data = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $growth_rate = 0;
    if ($growth_data['last_month'] > 0) {
        $growth_rate = round((($growth_data['this_month'] - $growth_data['last_month']) / $growth_data['last_month']) * 100, 1);
    }
    
    $stats['member_growth'] = [
        'this_month' => (int)$growth_data['this_month'],
        'last_month' => (int)$growth_data['last_month'],
        'growth_rate' => $growth_rate
    ];
    
    sendResponse(200, true, 'Dashboard statistics retrieved successfully', $stats);
    
} catch (Exception $e) {
    error_log("Dashboard stats API error: " . $e->getMessage());
    sendResponse(500, false, 'Failed to fetch dashboard statistics');
}
?>
