<?php
/**
 * Event Single Record API
 * GET /api/events/event.php?id=123 - Get single event
 * PUT /api/events/event.php?id=123 - Update event
 * DELETE /api/events/event.php?id=123 - Delete event
 */

require_once '../../../shared/api/config.php';

// Require admin authentication
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$event_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$event_id) {
    sendResponse(400, false, 'Event ID is required');
}

try {
    switch ($method) {
        case 'GET':
            getEvent($event_id);
            break;
        case 'PUT':
            updateEvent($event_id);
            break;
        case 'DELETE':
            deleteEvent($event_id);
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    error_log("Error in events/event.php: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

function getEvent($event_id) {
    global $pdo;
    
    // Get event with detailed information
    $sql = "SELECT 
                e.*,
                a.first_name as creator_first_name,
                a.last_name as creator_last_name,
                a.role as creator_role,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id) as total_registrations,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id AND er.payment_status = 'paid') as paid_registrations,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id AND er.payment_status = 'pending') as pending_payments,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id AND er.attendance_status = 'attended') as attended_count,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id AND er.attendance_status = 'no_show') as no_show_count
            FROM events e
            LEFT JOIN admins a ON e.created_by = a.id
            WHERE e.id = ?";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$event_id]);
    $event = $stmt->fetch();
    
    if (!$event) {
        sendResponse(404, false, 'Event not found');
    }
    
    // Format event data
    $event['creator_name'] = trim($event['creator_first_name'] . ' ' . $event['creator_last_name']);
    $event['event_date_formatted'] = date('M j, Y', strtotime($event['event_date']));
    $event['event_time_formatted'] = $event['event_time'] ? date('g:i A', strtotime($event['event_time'])) : null;
    $event['spots_remaining'] = $event['max_participants'] ? 
        max(0, $event['max_participants'] - $event['total_registrations']) : null;
    $event['registration_rate'] = $event['max_participants'] > 0 ? 
        round(($event['total_registrations'] / $event['max_participants']) * 100, 1) : 0;
    
    // Get recent registrations
    $reg_sql = "SELECT 
                    er.*,
                    m.first_name,
                    m.last_name,
                    m.kso_id,
                    m.email,
                    m.phone
                FROM event_registrations er
                LEFT JOIN members m ON er.member_id = m.id
                WHERE er.event_id = ?
                ORDER BY er.registration_date DESC
                LIMIT 10";
    
    $reg_stmt = $pdo->prepare($reg_sql);
    $reg_stmt->execute([$event_id]);
    $registrations = $reg_stmt->fetchAll();
    
    // Format registrations
    foreach ($registrations as &$registration) {
        $registration['member_name'] = trim($registration['first_name'] . ' ' . $registration['last_name']);
        $registration['registration_date_formatted'] = date('M j, Y g:i A', strtotime($registration['registration_date']));
    }
    
    $event['recent_registrations'] = $registrations;
    
    // Remove sensitive data
    unset($event['creator_first_name'], $event['creator_last_name']);
    
    sendResponse(200, true, 'Event retrieved successfully', ['event' => $event]);
}

function updateEvent($event_id) {
    global $pdo;
    
    // Check if event exists
    $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $stmt->execute([$event_id]);
    $existing_event = $stmt->fetch();
    
    if (!$existing_event) {
        sendResponse(404, false, 'Event not found');
    }
    
    $input = getJsonInput();
    
    // Validate event date if provided
    if (isset($input['event_date']) && !strtotime($input['event_date'])) {
        sendResponse(400, false, 'Invalid event date format');
    }
    
    // Validate event type if provided
    if (isset($input['event_type'])) {
        $valid_types = ['meeting', 'cultural', 'sports', 'educational', 'social', 'other'];
        if (!in_array($input['event_type'], $valid_types)) {
            sendResponse(400, false, 'Invalid event type');
        }
    }
    
    // Validate status if provided
    if (isset($input['status'])) {
        $valid_statuses = ['upcoming', 'ongoing', 'completed', 'cancelled'];
        if (!in_array($input['status'], $valid_statuses)) {
            sendResponse(400, false, 'Invalid status');
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Build update query dynamically
        $update_fields = [];
        $params = [];
        
        $updatable_fields = [
            'event_title', 'event_description', 'event_date', 'event_time',
            'event_location', 'event_type', 'max_participants', 'registration_fee', 'status'
        ];
        
        foreach ($updatable_fields as $field) {
            if (isset($input[$field])) {
                $update_fields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($update_fields)) {
            sendResponse(400, false, 'No valid fields to update');
        }
        
        // Add updated_at
        $update_fields[] = "updated_at = NOW()";
        $params[] = $event_id;
        
        $sql = "UPDATE events SET " . implode(', ', $update_fields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Log activity
        $admin = getCurrentAdmin();
        $changes = array_intersect_key($input, array_flip($updatable_fields));
        
        logActivity($admin['id'], 'events', 'update', 'Updated event: ' . $existing_event['event_title'], [
            'event_id' => $event_id,
            'changes' => $changes
        ]);
        
        // Get updated event
        $updated_sql = "SELECT e.*, 
                           a.first_name as creator_first_name, 
                           a.last_name as creator_last_name
                       FROM events e 
                       LEFT JOIN admins a ON e.created_by = a.id 
                       WHERE e.id = ?";
        $updated_stmt = $pdo->prepare($updated_sql);
        $updated_stmt->execute([$event_id]);
        $event = $updated_stmt->fetch();
        
        $event['creator_name'] = trim($event['creator_first_name'] . ' ' . $event['creator_last_name']);
        unset($event['creator_first_name'], $event['creator_last_name']);
        
        $pdo->commit();
        
        sendResponse(200, true, 'Event updated successfully', ['event' => $event]);
        
    } catch (Exception $e) {
        $pdo->rollback();
        error_log("Error updating event: " . $e->getMessage());
        sendResponse(500, false, 'Failed to update event');
    }
}

function deleteEvent($event_id) {
    global $pdo;
    
    // Check if event exists
    $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $stmt->execute([$event_id]);
    $event = $stmt->fetch();
    
    if (!$event) {
        sendResponse(404, false, 'Event not found');
    }
    
    // Check if event has registrations
    $reg_stmt = $pdo->prepare("SELECT COUNT(*) as count FROM event_registrations WHERE event_id = ?");
    $reg_stmt->execute([$event_id]);
    $registration_count = $reg_stmt->fetch()['count'];
    
    if ($registration_count > 0) {
        sendResponse(400, false, 'Cannot delete event with existing registrations. Cancel the event instead.');
    }
    
    try {
        $pdo->beginTransaction();
        
        // Delete the event
        $delete_stmt = $pdo->prepare("DELETE FROM events WHERE id = ?");
        $delete_stmt->execute([$event_id]);
        
        // Log activity
        $admin = getCurrentAdmin();
        logActivity($admin['id'], 'events', 'delete', 'Deleted event: ' . $event['event_title'], [
            'event_id' => $event_id,
            'event_type' => $event['event_type'],
            'event_date' => $event['event_date']
        ]);
        
        $pdo->commit();
        
        sendResponse(200, true, 'Event deleted successfully');
        
    } catch (Exception $e) {
        $pdo->rollback();
        error_log("Error deleting event: " . $e->getMessage());
        sendResponse(500, false, 'Failed to delete event');
    }
}
?>


