<?php
/**
 * Events API - Main Endpoint
 * GET /api/events/index.php - List events with filters
 * POST /api/events/index.php - Create new event
 */

// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

// Require admin authentication
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            getEvents();
            break;
        case 'POST':
            createEvent();
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    error_log("Error in events/index.php: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

function getEvents() {
    global $pdo;
    
    // Get query parameters
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 25;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    $status = isset($_GET['status']) ? trim($_GET['status']) : '';
    $event_type = isset($_GET['event_type']) ? trim($_GET['event_type']) : '';
    $date_from = isset($_GET['date_from']) ? trim($_GET['date_from']) : '';
    $date_to = isset($_GET['date_to']) ? trim($_GET['date_to']) : '';
    
    // Build query
    $where_conditions = [];
    $params = [];
    
    if (!empty($search)) {
        $where_conditions[] = "(e.event_title LIKE ? OR e.event_description LIKE ? OR e.event_location LIKE ?)";
        $search_param = "%$search%";
        $params[] = $search_param;
        $params[] = $search_param;
        $params[] = $search_param;
    }
    
    if (!empty($status)) {
        $where_conditions[] = "e.status = ?";
        $params[] = $status;
    }
    
    if (!empty($event_type)) {
        $where_conditions[] = "e.event_type = ?";
        $params[] = $event_type;
    }
    
    if (!empty($date_from)) {
        $where_conditions[] = "e.event_date >= ?";
        $params[] = $date_from;
    }
    
    if (!empty($date_to)) {
        $where_conditions[] = "e.event_date <= ?";
        $params[] = $date_to;
    }
    
    $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM events e $where_clause";
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total = $count_stmt->fetch()['total'];
    
    // Get events with registration counts
    $sql = "SELECT 
                e.*,
                a.first_name as creator_first_name,
                a.last_name as creator_last_name,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id) as total_registrations,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id AND er.payment_status = 'paid') as paid_registrations,
                (SELECT COUNT(*) FROM event_registrations er WHERE er.event_id = e.id AND er.attendance_status = 'attended') as attended_count
            FROM events e
            LEFT JOIN admins a ON e.created_by = a.id
            $where_clause
            ORDER BY e.event_date DESC, e.created_at DESC
            LIMIT ? OFFSET ?";
    
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $events = $stmt->fetchAll();
    
    // Format events
    foreach ($events as &$event) {
        $event['creator_name'] = trim($event['creator_first_name'] . ' ' . $event['creator_last_name']);
        $event['event_date_formatted'] = date('M j, Y', strtotime($event['event_date']));
        $event['event_time_formatted'] = $event['event_time'] ? date('g:i A', strtotime($event['event_time'])) : null;
        $event['registration_rate'] = $event['max_participants'] > 0 ? 
            round(($event['total_registrations'] / $event['max_participants']) * 100, 1) : 0;
        
        // Remove sensitive data
        unset($event['creator_first_name'], $event['creator_last_name']);
    }
    
    sendResponse(200, true, 'Events retrieved successfully', [
        'events' => $events,
        'pagination' => [
            'total' => $total,
            'limit' => $limit,
            'offset' => $offset,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function createEvent() {
    global $pdo;
    
    $input = getJsonInput();
    
    // Validate required fields
    $required = ['event_title', 'event_date', 'event_type'];
    foreach ($required as $field) {
        if (!isset($input[$field]) || empty(trim($input[$field]))) {
            sendResponse(400, false, "Field '$field' is required");
        }
    }
    
    // Validate event date
    if (!strtotime($input['event_date'])) {
        sendResponse(400, false, 'Invalid event date format');
    }
    
    // Validate event type
    $valid_types = ['meeting', 'cultural', 'sports', 'educational', 'social', 'other'];
    if (!in_array($input['event_type'], $valid_types)) {
        sendResponse(400, false, 'Invalid event type');
    }
    
    // Validate event time if provided
    if (!empty($input['event_time']) && !strtotime($input['event_time'])) {
        sendResponse(400, false, 'Invalid event time format');
    }
    
    // Validate max participants
    if (isset($input['max_participants']) && (!is_numeric($input['max_participants']) || $input['max_participants'] < 0)) {
        sendResponse(400, false, 'Max participants must be a positive number');
    }
    
    // Validate registration fee
    if (isset($input['registration_fee']) && (!is_numeric($input['registration_fee']) || $input['registration_fee'] < 0)) {
        sendResponse(400, false, 'Registration fee must be a positive number');
    }
    
    try {
        $pdo->beginTransaction();
        
        // Get current admin
        $admin = getCurrentAdmin();
        
        // Insert event
        $sql = "INSERT INTO events (
                    event_title, event_description, event_date, event_time, 
                    event_location, event_type, max_participants, registration_fee, 
                    status, created_by, created_at, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'upcoming', ?, NOW(), NOW())";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            trim($input['event_title']),
            trim($input['event_description'] ?? ''),
            $input['event_date'],
            $input['event_time'] ?? null,
            trim($input['event_location'] ?? ''),
            $input['event_type'],
            $input['max_participants'] ?? null,
            $input['registration_fee'] ?? 0.00,
            $admin['id']
        ]);
        
        $event_id = $pdo->lastInsertId();
        
        // Log activity
        logActivity($admin['id'], 'events', 'create', 'Created new event: ' . $input['event_title'], [
            'event_id' => $event_id,
            'event_type' => $input['event_type'],
            'event_date' => $input['event_date']
        ]);
        
        // Get the created event
        $event_sql = "SELECT e.*, 
                        a.first_name as creator_first_name, 
                        a.last_name as creator_last_name
                     FROM events e 
                     LEFT JOIN admins a ON e.created_by = a.id 
                     WHERE e.id = ?";
        $event_stmt = $pdo->prepare($event_sql);
        $event_stmt->execute([$event_id]);
        $event = $event_stmt->fetch();
        
        $event['creator_name'] = trim($event['creator_first_name'] . ' ' . $event['creator_last_name']);
        unset($event['creator_first_name'], $event['creator_last_name']);
        
        $pdo->commit();
        
        sendResponse(201, true, 'Event created successfully', ['event' => $event]);
        
    } catch (Exception $e) {
        $pdo->rollback();
        error_log("Error creating event: " . $e->getMessage());
        sendResponse(500, false, 'Failed to create event');
    }
}
?>


