<?php
/**
 * Event Registrations API
 * GET /api/events/registrations.php?event_id=123 - Get event registrations
 * POST /api/events/registrations.php - Register member for event
 * PUT /api/events/registrations.php?id=123 - Update registration (payment/attendance)
 * DELETE /api/events/registrations.php?id=123 - Cancel registration
 */

require_once '../../../shared/api/config.php';

// Require admin authentication
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            getRegistrations();
            break;
        case 'POST':
            createRegistration();
            break;
        case 'PUT':
            updateRegistration();
            break;
        case 'DELETE':
            deleteRegistration();
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    error_log("Error in events/registrations.php: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

function getRegistrations() {
    global $pdo;
    
    $event_id = isset($_GET['event_id']) ? (int)$_GET['event_id'] : 0;
    
    if (!$event_id) {
        sendResponse(400, false, 'Event ID is required');
    }
    
    // Check if event exists
    $event_stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $event_stmt->execute([$event_id]);
    $event = $event_stmt->fetch();
    
    if (!$event) {
        sendResponse(404, false, 'Event not found');
    }
    
    // Get filters
    $payment_status = isset($_GET['payment_status']) ? trim($_GET['payment_status']) : '';
    $attendance_status = isset($_GET['attendance_status']) ? trim($_GET['attendance_status']) : '';
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    
    // Build query
    $where_conditions = ["er.event_id = ?"];
    $params = [$event_id];
    
    if (!empty($payment_status)) {
        $where_conditions[] = "er.payment_status = ?";
        $params[] = $payment_status;
    }
    
    if (!empty($attendance_status)) {
        $where_conditions[] = "er.attendance_status = ?";
        $params[] = $attendance_status;
    }
    
    if (!empty($search)) {
        $where_conditions[] = "(m.first_name LIKE ? OR m.last_name LIKE ? OR m.kso_id LIKE ? OR m.email LIKE ?)";
        $search_param = "%$search%";
        $params[] = $search_param;
        $params[] = $search_param;
        $params[] = $search_param;
        $params[] = $search_param;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    $sql = "SELECT 
                er.*,
                m.kso_id,
                m.first_name,
                m.last_name,
                m.email,
                m.phone,
                m.college,
                m.membership_type
            FROM event_registrations er
            LEFT JOIN members m ON er.member_id = m.id
            WHERE $where_clause
            ORDER BY er.registration_date DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $registrations = $stmt->fetchAll();
    
    // Format registrations
    foreach ($registrations as &$registration) {
        $registration['member_name'] = trim($registration['first_name'] . ' ' . $registration['last_name']);
        $registration['registration_date_formatted'] = date('M j, Y g:i A', strtotime($registration['registration_date']));
    }
    
    // Get summary statistics
    $stats_sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN payment_status = 'paid' THEN 1 ELSE 0 END) as paid,
                    SUM(CASE WHEN payment_status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN payment_status = 'waived' THEN 1 ELSE 0 END) as waived,
                    SUM(CASE WHEN attendance_status = 'attended' THEN 1 ELSE 0 END) as attended,
                    SUM(CASE WHEN attendance_status = 'no_show' THEN 1 ELSE 0 END) as no_show
                  FROM event_registrations 
                  WHERE event_id = ?";
    
    $stats_stmt = $pdo->prepare($stats_sql);
    $stats_stmt->execute([$event_id]);
    $stats = $stats_stmt->fetch();
    
    sendResponse(200, true, 'Registrations retrieved successfully', [
        'event' => $event,
        'registrations' => $registrations,
        'statistics' => $stats
    ]);
}

function createRegistration() {
    global $pdo;
    
    $input = getJsonInput();
    
    // Validate required fields
    if (!isset($input['event_id']) || !isset($input['member_id'])) {
        sendResponse(400, false, 'Event ID and Member ID are required');
    }
    
    $event_id = (int)$input['event_id'];
    $member_id = (int)$input['member_id'];
    
    // Check if event exists
    $event_stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $event_stmt->execute([$event_id]);
    $event = $event_stmt->fetch();
    
    if (!$event) {
        sendResponse(404, false, 'Event not found');
    }
    
    // Check if member exists
    $member_stmt = $pdo->prepare("SELECT * FROM members WHERE id = ?");
    $member_stmt->execute([$member_id]);
    $member = $member_stmt->fetch();
    
    if (!$member) {
        sendResponse(404, false, 'Member not found');
    }
    
    // Check if already registered
    $existing_stmt = $pdo->prepare("SELECT id FROM event_registrations WHERE event_id = ? AND member_id = ?");
    $existing_stmt->execute([$event_id, $member_id]);
    if ($existing_stmt->fetch()) {
        sendResponse(400, false, 'Member is already registered for this event');
    }
    
    // Check capacity
    if ($event['max_participants']) {
        $count_stmt = $pdo->prepare("SELECT COUNT(*) as count FROM event_registrations WHERE event_id = ?");
        $count_stmt->execute([$event_id]);
        $current_count = $count_stmt->fetch()['count'];
        
        if ($current_count >= $event['max_participants']) {
            sendResponse(400, false, 'Event has reached maximum capacity');
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Create registration
        $payment_status = $input['payment_status'] ?? 'pending';
        $notes = trim($input['notes'] ?? '');
        
        $sql = "INSERT INTO event_registrations (
                    event_id, member_id, registration_date, payment_status, 
                    attendance_status, notes
                ) VALUES (?, ?, NOW(), ?, 'registered', ?)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$event_id, $member_id, $payment_status, $notes]);
        
        $registration_id = $pdo->lastInsertId();
        
        // Create finance record if payment made
        if ($payment_status === 'paid' && $event['registration_fee'] > 0) {
            $finance_sql = "INSERT INTO finance (
                                member_id, transaction_type, amount, payment_method,
                                payment_date, description, status, 
                                processed_by, created_at, updated_at
                            ) VALUES (?, 'event_fee', ?, ?, NOW(), ?, 'completed', ?, NOW(), NOW())";
            
            $admin = getCurrentAdmin();
            $payment_method = $input['payment_method'] ?? 'cash';
            $description = "Registration fee for: " . $event['event_title'];
            
            $finance_stmt = $pdo->prepare($finance_sql);
            $finance_stmt->execute([
                $member_id,
                $event['registration_fee'],
                $payment_method,
                $description,
                $admin['id']
            ]);
        }
        
        // Log activity
        $admin = getCurrentAdmin();
        logActivity($admin['id'], 'events', 'register_member', 
            "Registered member {$member['kso_id']} for event: {$event['event_title']}", [
            'event_id' => $event_id,
            'member_id' => $member_id,
            'payment_status' => $payment_status
        ]);
        
        // Get the created registration
        $reg_sql = "SELECT er.*, m.first_name, m.last_name, m.kso_id 
                    FROM event_registrations er
                    LEFT JOIN members m ON er.member_id = m.id
                    WHERE er.id = ?";
        $reg_stmt = $pdo->prepare($reg_sql);
        $reg_stmt->execute([$registration_id]);
        $registration = $reg_stmt->fetch();
        
        $registration['member_name'] = trim($registration['first_name'] . ' ' . $registration['last_name']);
        
        $pdo->commit();
        
        sendResponse(201, true, 'Member registered successfully', ['registration' => $registration]);
        
    } catch (Exception $e) {
        $pdo->rollback();
        error_log("Error creating registration: " . $e->getMessage());
        sendResponse(500, false, 'Failed to register member');
    }
}

function updateRegistration() {
    global $pdo;
    
    $registration_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    
    if (!$registration_id) {
        sendResponse(400, false, 'Registration ID is required');
    }
    
    // Check if registration exists
    $stmt = $pdo->prepare("SELECT er.*, e.event_title, e.registration_fee 
                          FROM event_registrations er 
                          LEFT JOIN events e ON er.event_id = e.id 
                          WHERE er.id = ?");
    $stmt->execute([$registration_id]);
    $registration = $stmt->fetch();
    
    if (!$registration) {
        sendResponse(404, false, 'Registration not found');
    }
    
    $input = getJsonInput();
    
    try {
        $pdo->beginTransaction();
        
        // Build update query
        $update_fields = [];
        $params = [];
        
        if (isset($input['payment_status'])) {
            $valid_payment_statuses = ['pending', 'paid', 'waived'];
            if (!in_array($input['payment_status'], $valid_payment_statuses)) {
                sendResponse(400, false, 'Invalid payment status');
            }
            $update_fields[] = "payment_status = ?";
            $params[] = $input['payment_status'];
        }
        
        if (isset($input['attendance_status'])) {
            $valid_attendance_statuses = ['registered', 'attended', 'no_show', 'cancelled'];
            if (!in_array($input['attendance_status'], $valid_attendance_statuses)) {
                sendResponse(400, false, 'Invalid attendance status');
            }
            $update_fields[] = "attendance_status = ?";
            $params[] = $input['attendance_status'];
        }
        
        if (isset($input['notes'])) {
            $update_fields[] = "notes = ?";
            $params[] = trim($input['notes']);
        }
        
        if (empty($update_fields)) {
            sendResponse(400, false, 'No valid fields to update');
        }
        
        $params[] = $registration_id;
        
        $sql = "UPDATE event_registrations SET " . implode(', ', $update_fields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Handle payment status change
        if (isset($input['payment_status']) && $input['payment_status'] === 'paid' && 
            $registration['payment_status'] !== 'paid' && $registration['registration_fee'] > 0) {
            
            // Create finance record
            $finance_sql = "INSERT INTO finance (
                                member_id, transaction_type, amount, payment_method,
                                payment_date, description, status, 
                                processed_by, created_at, updated_at
                            ) VALUES (?, 'event_fee', ?, ?, NOW(), ?, 'completed', ?, NOW(), NOW())";
            
            $admin = getCurrentAdmin();
            $payment_method = $input['payment_method'] ?? 'cash';
            $description = "Registration fee for: " . $registration['event_title'];
            
            $finance_stmt = $pdo->prepare($finance_sql);
            $finance_stmt->execute([
                $registration['member_id'],
                $registration['registration_fee'],
                $payment_method,
                $description,
                $admin['id']
            ]);
        }
        
        // Log activity
        $admin = getCurrentAdmin();
        logActivity($admin['id'], 'events', 'update_registration', 
            "Updated registration for event: {$registration['event_title']}", [
            'registration_id' => $registration_id,
            'changes' => array_intersect_key($input, array_flip(['payment_status', 'attendance_status', 'notes']))
        ]);
        
        $pdo->commit();
        
        sendResponse(200, true, 'Registration updated successfully');
        
    } catch (Exception $e) {
        $pdo->rollback();
        error_log("Error updating registration: " . $e->getMessage());
        sendResponse(500, false, 'Failed to update registration');
    }
}

function deleteRegistration() {
    global $pdo;
    
    $registration_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    
    if (!$registration_id) {
        sendResponse(400, false, 'Registration ID is required');
    }
    
    // Check if registration exists
    $stmt = $pdo->prepare("SELECT er.*, e.event_title 
                          FROM event_registrations er 
                          LEFT JOIN events e ON er.event_id = e.id 
                          WHERE er.id = ?");
    $stmt->execute([$registration_id]);
    $registration = $stmt->fetch();
    
    if (!$registration) {
        sendResponse(404, false, 'Registration not found');
    }
    
    try {
        $pdo->beginTransaction();
        
        // Delete registration
        $delete_stmt = $pdo->prepare("DELETE FROM event_registrations WHERE id = ?");
        $delete_stmt->execute([$registration_id]);
        
        // Log activity
        $admin = getCurrentAdmin();
        logActivity($admin['id'], 'events', 'cancel_registration', 
            "Cancelled registration for event: {$registration['event_title']}", [
            'registration_id' => $registration_id,
            'event_id' => $registration['event_id'],
            'member_id' => $registration['member_id']
        ]);
        
        $pdo->commit();
        
        sendResponse(200, true, 'Registration cancelled successfully');
        
    } catch (Exception $e) {
        $pdo->rollback();
        error_log("Error deleting registration: " . $e->getMessage());
        sendResponse(500, false, 'Failed to cancel registration');
    }
}
?>


