<?php
/**
 * Events Statistics API
 * GET /api/events/stats.php - Get event statistics
 */

require_once '../../../shared/api/config.php';

// Require admin authentication
requireAuth();

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendResponse(405, false, 'Method not allowed');
}

try {
    global $pdo;
    
    // Get basic event statistics
    $stats = [];
    
    // Total events
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM events");
    $stats['total_events'] = $stmt->fetch()['count'];
    
    // Events by status
    $stmt = $pdo->query("SELECT status, COUNT(*) as count FROM events GROUP BY status");
    $status_counts = $stmt->fetchAll();
    
    $stats['upcoming'] = 0;
    $stats['ongoing'] = 0;
    $stats['completed'] = 0;
    $stats['cancelled'] = 0;
    
    foreach ($status_counts as $status) {
        $stats[$status['status']] = $status['count'];
    }
    
    // Events by type
    $stmt = $pdo->query("SELECT event_type, COUNT(*) as count FROM events GROUP BY event_type");
    $type_counts = $stmt->fetchAll();
    $stats['by_type'] = [];
    foreach ($type_counts as $type) {
        $stats['by_type'][$type['event_type']] = $type['count'];
    }
    
    // Total registrations
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM event_registrations");
    $stats['total_registrations'] = $stmt->fetch()['count'];
    
    // Registration statistics
    $stmt = $pdo->query("
        SELECT 
            payment_status,
            COUNT(*) as count 
        FROM event_registrations 
        GROUP BY payment_status
    ");
    $payment_counts = $stmt->fetchAll();
    
    $stats['paid_registrations'] = 0;
    $stats['pending_payments'] = 0;
    $stats['waived_payments'] = 0;
    
    foreach ($payment_counts as $payment) {
        switch ($payment['payment_status']) {
            case 'paid':
                $stats['paid_registrations'] = $payment['count'];
                break;
            case 'pending':
                $stats['pending_payments'] = $payment['count'];
                break;
            case 'waived':
                $stats['waived_payments'] = $payment['count'];
                break;
        }
    }
    
    // Attendance statistics
    $stmt = $pdo->query("
        SELECT 
            attendance_status,
            COUNT(*) as count 
        FROM event_registrations 
        GROUP BY attendance_status
    ");
    $attendance_counts = $stmt->fetchAll();
    
    $stats['attended'] = 0;
    $stats['registered'] = 0;
    $stats['no_show'] = 0;
    $stats['cancelled_registrations'] = 0;
    
    foreach ($attendance_counts as $attendance) {
        switch ($attendance['attendance_status']) {
            case 'attended':
                $stats['attended'] = $attendance['count'];
                break;
            case 'registered':
                $stats['registered'] = $attendance['count'];
                break;
            case 'no_show':
                $stats['no_show'] = $attendance['count'];
                break;
            case 'cancelled':
                $stats['cancelled_registrations'] = $attendance['count'];
                break;
        }
    }
    
    // Revenue statistics
    $stmt = $pdo->query("
        SELECT 
            SUM(e.registration_fee) as total_potential_revenue,
            SUM(CASE WHEN er.payment_status = 'paid' THEN e.registration_fee ELSE 0 END) as collected_revenue
        FROM events e
        LEFT JOIN event_registrations er ON e.id = er.event_id
    ");
    $revenue = $stmt->fetch();
    $stats['total_potential_revenue'] = $revenue['total_potential_revenue'] ?? 0;
    $stats['collected_revenue'] = $revenue['collected_revenue'] ?? 0;
    
    // Recent events (last 5)
    $stmt = $pdo->query("
        SELECT 
            e.id,
            e.event_title,
            e.event_date,
            e.event_type,
            e.status,
            COUNT(er.id) as registration_count
        FROM events e
        LEFT JOIN event_registrations er ON e.id = er.event_id
        GROUP BY e.id
        ORDER BY e.created_at DESC
        LIMIT 5
    ");
    $stats['recent_events'] = $stmt->fetchAll();
    
    // Upcoming events (next 5)
    $stmt = $pdo->query("
        SELECT 
            e.id,
            e.event_title,
            e.event_date,
            e.event_time,
            e.event_type,
            e.max_participants,
            COUNT(er.id) as registration_count
        FROM events e
        LEFT JOIN event_registrations er ON e.id = er.event_id
        WHERE e.event_date >= CURDATE() AND e.status = 'upcoming'
        GROUP BY e.id
        ORDER BY e.event_date ASC
        LIMIT 5
    ");
    $upcoming = $stmt->fetchAll();
    
    // Format upcoming events
    foreach ($upcoming as &$event) {
        $event['event_date_formatted'] = date('M j, Y', strtotime($event['event_date']));
        $event['event_time_formatted'] = $event['event_time'] ? date('g:i A', strtotime($event['event_time'])) : null;
        $event['spots_remaining'] = $event['max_participants'] ? 
            max(0, $event['max_participants'] - $event['registration_count']) : null;
    }
    
    $stats['upcoming_events'] = $upcoming;
    
    // Monthly event trend (last 12 months)
    $stmt = $pdo->query("
        SELECT 
            DATE_FORMAT(event_date, '%Y-%m') as month,
            COUNT(*) as event_count,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_count
        FROM events 
        WHERE event_date >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(event_date, '%Y-%m')
        ORDER BY month ASC
    ");
    $stats['monthly_trend'] = $stmt->fetchAll();
    
    sendResponse(200, true, 'Event statistics retrieved successfully', $stats);
    
} catch (Exception $e) {
    error_log("Error in events/stats.php: " . $e->getMessage());
    sendResponse(500, false, 'Failed to retrieve event statistics');
}
?>


