<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                getTransaction($_GET['id']);
            } else if (isset($_GET['export'])) {
                exportTransactions($_GET['export']);
            } else {
                getTransactions();
            }
            break;
        case 'POST':
            if (isset($_GET['recurring'])) {
                createRecurringTransaction();
            } else if (isset($_GET['attachment'])) {
                uploadAttachment();
            } else if (isset($_GET['archive_session'])) {
                archiveSessionData($_GET['archive_session']);
            } else if (isset($_GET['archive_membership_session'])) {
                archiveMembershipSessionData($_GET['archive_membership_session']);
            } else if (isset($_GET['archive_expenses_session'])) {
                archiveExpensesSessionData($_GET['archive_expenses_session']);
            } else {
                createTransaction();
            }
            break;
        case 'PUT':
            if (isset($_GET['id'])) {
                updateTransaction($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Transaction ID required for update']);
            }
            break;
        case 'DELETE':
            if (isset($_GET['id'])) {
                deleteTransaction($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Transaction ID required for delete']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getTransactions() {
    global $pdo;
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
    $offset = ($page - 1) * $limit;
    $search = isset($_GET['search']) ? $_GET['search'] : '';
    $type = isset($_GET['type']) ? $_GET['type'] : '';
    $dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : '';
    $dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : '';
    $category = isset($_GET['category']) ? $_GET['category'] : '';
    $amountMin = isset($_GET['amount_min']) ? $_GET['amount_min'] : '';
    $amountMax = isset($_GET['amount_max']) ? $_GET['amount_max'] : '';
    $memberId = isset($_GET['member_id']) ? $_GET['member_id'] : '';
    $sessionYear = isset($_GET['session_year']) ? $_GET['session_year'] : '';
    $table = isset($_GET['table']) && $_GET['table'] === 'expenses' ? 'expenses' : 'finance';
    $where = [];
    $params = [];
    if (!empty($search)) {
        $where[] = "(description LIKE ? OR notes LIKE ? OR receipt_number LIKE ? OR expense_title LIKE ? OR expense_category LIKE ? OR category LIKE ? )";
        for ($i = 0; $i < 6; $i++) $params[] = "%$search%";
    }
    if (!empty($type)) {
        if ($table === 'finance') {
            $where[] = "transaction_type = ?";
        } else {
            $where[] = "expense_category = ?";
        }
        $params[] = $type;
    }
    if (!empty($category)) {
        if ($table === 'finance') {
            $where[] = "category = ?";
        } else {
            $where[] = "expense_category = ?";
        }
        $params[] = $category;
    }
    if (!empty($dateFrom)) {
        $where[] = ($table === 'finance' ? "payment_date >= ?" : "expense_date >= ?");
        $params[] = $dateFrom;
    }
    if (!empty($dateTo)) {
        $where[] = ($table === 'finance' ? "payment_date <= ?" : "expense_date <= ?");
        $params[] = $dateTo;
    }
    if (!empty($amountMin)) {
        $where[] = "amount >= ?";
        $params[] = $amountMin;
    }
    if (!empty($amountMax)) {
        $where[] = "amount <= ?";
        $params[] = $amountMax;
    }
    if (!empty($memberId) && $table === 'finance') {
        $where[] = "member_id = ?";
        $params[] = $memberId;
    }
    if (!empty($sessionYear) && $table === 'finance') {
        $where[] = "session_year = ?";
        $params[] = $sessionYear;
    }
    $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";
    $countSql = "SELECT COUNT(*) as total FROM $table $whereClause";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute($params);
    $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    $sql = "SELECT * FROM $table $whereClause ORDER BY ".($table==='finance'?'payment_date':'expense_date')." DESC, created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    echo json_encode([
        'success' => true,
        'data' => $transactions,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function getTransaction($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM finance WHERE id = ?");
        $stmt->execute([$id]);
        $transaction = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$transaction) {
            http_response_code(404);
            echo json_encode(['error' => 'Transaction not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'data' => $transaction
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function createTransaction() {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Validate required fields
        $required = ['transaction_type', 'amount', 'description', 'transaction_date'];
        foreach ($required as $field) {
            if (!isset($input[$field]) || empty($input[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Field '$field' is required"]);
                return;
            }
        }
        
        // Validate transaction type
        $validTypes = ['income', 'expense'];
        if (!in_array($input['transaction_type'], $validTypes)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid transaction type']);
            return;
        }
        
        // Generate reference number if not provided
        $referenceNumber = $input['reference_number'] ?? 'TXN-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        $sql = "INSERT INTO finance (transaction_type, amount, description, payment_date, receipt_number, notes, processed_by, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['transaction_type'],
            $input['amount'],
            $input['description'],
            $input['transaction_date'],
            $referenceNumber,
            $input['category'] ?? null,
            $input['notes'] ?? null,
            $_SESSION['admin_id']
        ]);
        
        $transactionId = $pdo->lastInsertId();
        
        // Get the created transaction
        $stmt = $pdo->prepare("SELECT * FROM finance_transactions WHERE id = ?");
        $stmt->execute([$transactionId]);
        $transaction = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Transaction created successfully',
            'data' => $transaction
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateTransaction($id) {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Check if transaction exists
        $stmt = $pdo->prepare("SELECT id FROM finance WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Transaction not found']);
            return;
        }
        
        $updateFields = [];
        $params = [];
        
        $allowedFields = ['transaction_type', 'amount', 'description', 'transaction_date', 'reference_number', 'category', 'notes'];
        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateFields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($updateFields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        $updateFields[] = "updated_at = NOW()";
        $params[] = $id;
        
        $sql = "UPDATE finance SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Get updated transaction
        $stmt = $pdo->prepare("SELECT * FROM finance WHERE id = ?");
        $stmt->execute([$id]);
        $transaction = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Transaction updated successfully',
            'data' => $transaction
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteTransaction($id) {
    global $pdo;
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        $reason = isset($input['reason']) ? trim($input['reason']) : '';
        if (strlen($reason) < 5) {
            http_response_code(400);
            echo json_encode(['error' => 'A valid reason (at least 5 characters) is required for deletion.']);
            return;
        }
        // Check if transaction exists and fetch details
        $stmt = $pdo->prepare("SELECT * FROM finance WHERE id = ?");
        $stmt->execute([$id]);
        $transaction = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$transaction) {
            http_response_code(404);
            echo json_encode(['error' => 'Transaction not found']);
            return;
        }
        // Delete transaction
        $stmt = $pdo->prepare("DELETE FROM finance WHERE id = ?");
        $stmt->execute([$id]);
        // Log to activity_logs
        $logStmt = $pdo->prepare("INSERT INTO activity_logs (user_id, action, details, ip_address, entity_type, entity_id, severity, user_agent, session_id, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
        $userId = $_SESSION['admin_id'] ?? null;
        $action = 'delete_transaction';
        $details = json_encode([
            'reason' => $reason,
            'transaction' => $transaction
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? null;
        $entityType = 'finance';
        $entityId = $id;
        $severity = 'WARNING';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
        $sessionId = session_id();
        $logStmt->execute([$userId, $action, $details, $ip, $entityType, $entityId, $severity, $userAgent, $sessionId]);
        echo json_encode([
            'success' => true,
            'message' => 'Transaction deleted and action logged.'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

// Scaffolded: exportTransactions, createRecurringTransaction, uploadAttachment, budgeting, auditTrail, approvalWorkflow, notifications, roleBasedAccess
function exportTransactions($format) {
    global $pdo;
    $table = isset($_GET['table']) && $_GET['table'] === 'expenses' ? 'expenses' : 'finance';
    $sql = "SELECT * FROM $table";
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if ($format === 'excel') {
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment; filename="transactions_export.xls"');
        $heading = false;
        foreach($rows as $row) {
            if(!$heading) {
                echo implode("\t", array_keys($row)) . "\n";
                $heading = true;
            }
            echo implode("\t", array_values($row)) . "\n";
        }
        exit();
    } elseif ($format === 'pdf') {
        // Simple PDF export using FPDF (if available)
        if (!class_exists('FPDF')) {
            echo json_encode(['success'=>false,'message'=>'PDF export requires FPDF library']);
            return;
        }
        $pdf = new FPDF();
        $pdf->AddPage();
        $pdf->SetFont('Arial','B',12);
        foreach($rows as $i => $row) {
            if ($i === 0) {
                foreach(array_keys($row) as $col) $pdf->Cell(40,10,$col,1);
                $pdf->Ln();
            }
            foreach(array_values($row) as $val) $pdf->Cell(40,10,$val,1);
            $pdf->Ln();
        }
        $pdf->Output('D', 'transactions_export.pdf');
        exit();
    } else {
        echo json_encode(['success'=>false,'message'=>'Invalid export format']);
    }
}
function createRecurringTransaction() {
    // Placeholder for recurring transaction logic
    echo json_encode(['success'=>false,'message'=>'Recurring transaction logic not yet implemented.']);
}
function uploadAttachment() {
    // Placeholder for attachment upload logic
    echo json_encode(['success'=>false,'message'=>'Attachment upload logic not yet implemented.']);
}
function auditTrail() {
    // Placeholder for audit trail logic
    echo json_encode(['success'=>false,'message'=>'Audit trail logic not yet implemented.']);
}
function budgeting() {
    // Placeholder for budgeting logic
    echo json_encode(['success'=>false,'message'=>'Budgeting logic not yet implemented.']);
}
function approvalWorkflow() {
    // Placeholder for approval workflow logic
    echo json_encode(['success'=>false,'message'=>'Approval workflow logic not yet implemented.']);
}
function notifications() {
    // Placeholder for notifications logic
    echo json_encode(['success'=>false,'message'=>'Notifications logic not yet implemented.']);
}
function roleBasedAccess() {
    // Placeholder for role-based access logic
    echo json_encode(['success'=>false,'message'=>'Role-based access logic not yet implemented.']);
}
function analytics() {
    // Placeholder for analytics/chart logic
    echo json_encode(['success'=>false,'message'=>'Analytics/chart logic not yet implemented.']);
}
function archiveMembershipSessionData($sessionYear) {
    global $pdo;
    $archiveTable = 'members_archive';
    $pdo->exec("CREATE TABLE IF NOT EXISTS $archiveTable LIKE members");
    $stmt = $pdo->prepare("INSERT INTO $archiveTable SELECT * FROM members WHERE session_from_year = ?");
    $stmt->execute([$sessionYear]);
    $stmt = $pdo->prepare("DELETE FROM members WHERE session_from_year = ?");
    $stmt->execute([$sessionYear]);
    echo json_encode(['success'=>true,'message'=>'Membership session data archived.']);
}

function archiveExpensesSessionData($sessionYear) {
    global $pdo;
    $archiveTable = 'expenses_archive';
    $pdo->exec("CREATE TABLE IF NOT EXISTS $archiveTable LIKE expenses");
    $stmt = $pdo->prepare("INSERT INTO $archiveTable SELECT * FROM expenses WHERE YEAR(expense_date) = ?");
    $stmt->execute([$sessionYear]);
    $stmt = $pdo->prepare("DELETE FROM expenses WHERE YEAR(expense_date) = ?");
    $stmt->execute([$sessionYear]);
    echo json_encode(['success'=>true,'message'=>'Expenses session data archived.']);
}
?>


