<?php
require_once '../../../shared/api/config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

try {
    $stats = [
        'total_income' => 0,
        'total_expenses' => 0,
        'net_balance' => 0,
        'monthly_income' => 0,
        'monthly_expenses' => 0,
        'transactions_count' => 0,
        'recent_transactions' => [],
        'income_vs_expenses_chart' => [],
        'monthly_trends' => []
    ];
    
    // Get total income and expenses from finance table
    $stmt = $pdo->prepare("
        SELECT 
            transaction_type,
            SUM(amount) as total,
            COUNT(*) as count
        FROM finance
        WHERE status = 'completed'
        GROUP BY transaction_type
    ");
    $stmt->execute();
    $totals = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($totals as $total) {
        if ($total['transaction_type'] === 'membership_fee' || $total['transaction_type'] === 'donation' || $total['transaction_type'] === 'event_fee') {
            $stats['total_income'] += $total['total'];
        }
        $stats['transactions_count'] += $total['count'];
    }
    
    // Get total expenses from expenses table
    $stmt = $pdo->prepare("
        SELECT 
            SUM(amount) as total,
            COUNT(*) as count
        FROM expenses
        WHERE status IN ('approved', 'paid')
    ");
    $stmt->execute();
    $expenseData = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($expenseData) {
        $stats['total_expenses'] = $expenseData['total'] ?? 0;
        $stats['transactions_count'] += $expenseData['count'] ?? 0;
    }
    
    $stats['net_balance'] = $stats['total_income'] - $stats['total_expenses'];
    
    // Get monthly stats (current month)
    $currentMonth = date('Y-m');
    $stmt = $pdo->prepare("
        SELECT 
            SUM(amount) as total
        FROM finance 
        WHERE DATE_FORMAT(payment_date, '%Y-%m') = ? AND status = 'completed'
    ");
    $stmt->execute([$currentMonth]);
    $monthlyIncome = $stmt->fetchColumn() ?? 0;
    $stats['monthly_income'] = $monthlyIncome;
    
    $stmt = $pdo->prepare("
        SELECT 
            SUM(amount) as total
        FROM expenses 
        WHERE DATE_FORMAT(expense_date, '%Y-%m') = ? AND status IN ('approved', 'paid')
    ");
    $stmt->execute([$currentMonth]);
    $monthlyExpenses = $stmt->fetchColumn() ?? 0;
    $stats['monthly_expenses'] = $monthlyExpenses;
    
    // Get recent transactions (last 10 from both tables)
    $stmt = $pdo->prepare("
        SELECT id, transaction_type as type, amount, description, payment_date as date, 'finance' as source
        FROM finance 
        WHERE status = 'completed'
        UNION ALL
        SELECT id, expense_category as type, amount, expense_title as description, expense_date as date, 'expenses' as source
        FROM expenses 
        WHERE status IN ('approved', 'paid')
        ORDER BY date DESC 
        LIMIT 10
    ");
    $stmt->execute();
    $stats['recent_transactions'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get data for income vs expenses chart (last 12 months)
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(payment_date, '%Y-%m') as month,
            SUM(amount) as income,
            0 as expenses
        FROM finance 
        WHERE payment_date >= DATE_SUB(NOW(), INTERVAL 12 MONTH) AND status = 'completed'
        GROUP BY DATE_FORMAT(payment_date, '%Y-%m')
        UNION ALL
        SELECT 
            DATE_FORMAT(expense_date, '%Y-%m') as month,
            0 as income,
            SUM(amount) as expenses
        FROM expenses 
        WHERE expense_date >= DATE_SUB(NOW(), INTERVAL 12 MONTH) AND status IN ('approved', 'paid')
        GROUP BY DATE_FORMAT(expense_date, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute();
    $chartData = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process chart data
    $chartDataProcessed = [];
    foreach ($chartData as $data) {
        $month = $data['month'];
        if (!isset($chartDataProcessed[$month])) {
            $chartDataProcessed[$month] = [
                'month' => $month,
                'income' => 0,
                'expenses' => 0
            ];
        }
        $chartDataProcessed[$month]['income'] += (float)$data['income'];
        $chartDataProcessed[$month]['expenses'] += (float)$data['expenses'];
    }
                'expenses' => 0
            ];
        }
        
        if ($data['transaction_type'] === 'income') {
            $chartDataProcessed[$month]['income'] = $data['total'];
        } else {
            $chartDataProcessed[$month]['expenses'] = $data['total'];
        }
    }
    
    $stats['income_vs_expenses_chart'] = array_values($chartDataProcessed);
    
    // Get monthly trends (last 6 months)
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(transaction_date, '%Y-%m') as month,
            COUNT(*) as transaction_count,
            SUM(CASE WHEN transaction_type = 'income' THEN amount ELSE 0 END) as income,
            SUM(CASE WHEN transaction_type = 'expense' THEN amount ELSE 0 END) as expenses
        FROM finance_transactions 
        WHERE transaction_date >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
        GROUP BY DATE_FORMAT(transaction_date, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute();
    $trends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($trends as &$trend) {
        $trend['net'] = $trend['income'] - $trend['expenses'];
    }
    
    $stats['monthly_trends'] = $trends;
    
    echo json_encode([
        'success' => true,
        'data' => $stats
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>


