<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';
require_once '../../../shared/services/EmailService.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ApiResponse::error('Method not allowed', 405);
}

// Require admin authentication
ApiAuth::requireAdminAuth();

try {
    // Get input data
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        ApiResponse::error('Invalid JSON input', 400);
    }
    
    // Validate required fields
    ApiValidator::validateRequired($input, ['member_id']);
    
    $member_id = (int)$input['member_id'];
    $notes = ApiValidator::sanitizeInput($input['notes'] ?? '');
    $upi_tx_id = ApiValidator::sanitizeInput($input['upi_tx_id'] ?? '');
    $current_admin_id = ApiAuth::getCurrentAdminId();
    
    global $pdo;
    
    // Check if member exists and is pending
    $stmt = $pdo->prepare("
        SELECT id, kso_id, first_name, last_name, email, approval_status, membership_type 
        FROM members 
        WHERE id = ?
    ");
    $stmt->execute([$member_id]);
    $member = $stmt->fetch();
    
    if (!$member) {
        ApiResponse::error('Member not found', 404);
    }
    
    if ($member['approval_status'] !== 'pending' && $member['approval_status'] !== 'under_review') {
        ApiResponse::error('Member is not in a state that can be approved', 400);
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    try {
        // Update member status to approved
        $stmt = $pdo->prepare("
            UPDATE members 
            SET approval_status = 'approved',
                status = 'active',
                approved_by = ?,
                approved_at = NOW(),
                assignment_notes = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        
        $success = $stmt->execute([
            $current_admin_id,
            $notes,
            $member_id
        ]);
        
        if (!$success) {
            throw new Exception('Failed to update member status');
        }
        
        // Set membership expiry date (1 year from approval)
        $expiry_date = date('Y-m-d', strtotime('+1 year'));
        $stmt = $pdo->prepare("UPDATE members SET membership_expiry_date = ? WHERE id = ?");
        $stmt->execute([$expiry_date, $member_id]);
        
        // Create finance record for membership fee if UPI transaction ID provided
        if (!empty($upi_tx_id)) {
            // Get current membership fee setting
            $current_year = date('Y');
            $stmt = $pdo->prepare("
                SELECT fee_amount 
                FROM membership_fee_settings 
                WHERE membership_type = ? AND session_year = ? AND is_active = 1
                LIMIT 1
            ");
            $stmt->execute([$member['membership_type'], $current_year]);
            $fee_setting = $stmt->fetch();
            
            if ($fee_setting) {
                // Generate receipt number
                $receipt_number = 'KSO' . date('Y') . str_pad($member_id, 4, '0', STR_PAD_LEFT);
                
                $stmt = $pdo->prepare("
                    INSERT INTO finance (
                        member_id, transaction_type, amount, payment_method, 
                        payment_date, receipt_number, description, status,
                        upi_transaction_id, session_year, created_at
                    ) VALUES (?, 'membership_fee', ?, 'UPI', NOW(), ?, ?, 'completed', ?, ?, NOW())
                ");
                
                $stmt->execute([
                    $member_id,
                    $fee_setting['fee_amount'],
                    $receipt_number,
                    "Membership Fee - {$member['membership_type']} - {$current_year}",
                    $upi_tx_id,
                    $current_year
                ]);
            }
        }
        
        // Log activity
        ActivityLogger::log(
            'member_approved',
            "Approved member {$member['kso_id']} ({$member['first_name']} {$member['last_name']})" . 
            (!empty($notes) ? ". Notes: $notes" : "") .
            (!empty($upi_tx_id) ? ". UPI TX: $upi_tx_id" : ""),
            'member',
            $member_id
        );
        
        // Commit transaction
        $pdo->commit();
        
        // Send approval email notification
        try {
            $emailResult = EmailNotifications::sendApprovalEmail($member);
            if ($emailResult) {
                error_log("Approval email sent successfully to {$member['email']}");
            } else {
                error_log("Failed to send approval email to {$member['email']}");
            }
        } catch (Exception $emailError) {
            error_log("Email notification error: " . $emailError->getMessage());
            // Don't fail the approval if email fails
        }
        
        ApiResponse::success([
            'member_id' => $member_id,
            'kso_id' => $member['kso_id'],
            'status' => 'approved',
            'membership_expiry_date' => $expiry_date,
            'email_sent' => isset($emailResult) ? $emailResult : false
        ], 'Member approved successfully');
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Error in members/approve.php: " . $e->getMessage());
    ApiResponse::error('Failed to approve member: ' . $e->getMessage(), 500);
}
?>


