<?php
/**
 * Member Assignment API Endpoint
 * POST /api/members/assign.php
 */

// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

// Require admin authentication
ApiAuth::requireAdminAuth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'POST':
            handleAssignment();
            break;
        case 'GET':
            getAssignmentData();
            break;
        default:
            ApiResponse::error('Method not allowed', 405);
    }
} catch (Exception $e) {
    error_log("Error in members/assign.php: " . $e->getMessage());
    ApiResponse::error('Internal server error: ' . $e->getMessage(), 500);
}

function handleAssignment() {
    global $pdo;
    
    $input = json_decode(file_get_contents('php://input'), true);
    if (!$input) {
        ApiResponse::error('Invalid JSON input', 400);
    }
    
    $action = $input['action'] ?? '';
    $memberId = (int)($input['member_id'] ?? 0);
    $currentAdminId = ApiAuth::getCurrentAdminId();
    
    if (!$memberId) {
        ApiResponse::error('Member ID is required', 400);
    }
    
    switch ($action) {
        case 'assign':
            assignMember($pdo, $memberId, $input, $currentAdminId);
            break;
        case 'unassign':
            unassignMember($pdo, $memberId, $currentAdminId);
            break;
        default:
            ApiResponse::error('Invalid action', 400);
    }
}

function assignMember($pdo, $memberId, $input, $currentAdminId) {
    $assignedTo = (int)($input['assigned_to'] ?? 0);
    $priority = $input['priority'] ?? 'normal';
    $dueDate = $input['due_date'] ?? null;
    $notes = trim($input['notes'] ?? '');
    
    // Validate inputs
    if (!$assignedTo) {
        ApiResponse::error('Assigned admin ID is required', 400);
    }
    
    if (!in_array($priority, ['low', 'normal', 'high', 'urgent'])) {
        $priority = 'normal';
    }
    
    if ($dueDate && !ApiValidator::validateDateFormat($dueDate, 'Y-m-d H:i:s')) {
        if (ApiValidator::validateDateFormat($dueDate, 'Y-m-d')) {
            $dueDate .= ' 23:59:59';
        } else {
            ApiResponse::error('Invalid due date format', 400);
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Check if member exists and is pending
        $memberStmt = $pdo->prepare("SELECT id, first_name, last_name, status, assigned_to FROM members WHERE id = ?");
        $memberStmt->execute([$memberId]);
        $member = $memberStmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member) {
            throw new Exception('Member not found');
        }
        
        if ($member['status'] !== 'pending') {
            throw new Exception('Only pending members can be assigned');
        }
        
        // Check if target admin exists and is active
        $adminStmt = $pdo->prepare("SELECT id, first_name, last_name, status FROM admins WHERE id = ? AND status = 'active'");
        $adminStmt->execute([$assignedTo]);
        $admin = $adminStmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$admin) {
            throw new Exception('Target admin not found or inactive');
        }
        
        // Update member assignment
        $updateStmt = $pdo->prepare("
            UPDATE members 
            SET assigned_to = ?, 
                assigned_by = ?, 
                assignment_date = CURRENT_TIMESTAMP,
                assignment_notes = ?
            WHERE id = ?
        ");
        $updateStmt->execute([$assignedTo, $currentAdminId, $notes, $memberId]);
        
        // Create assignment record
        $assignmentStmt = $pdo->prepare("
            INSERT INTO admin_assignments 
            (member_id, assigned_to, assigned_by, assignment_type, priority, due_date, notes, status) 
            VALUES (?, ?, ?, 'review', ?, ?, ?, 'active')
        ");
        $assignmentStmt->execute([$memberId, $assignedTo, $currentAdminId, $priority, $dueDate, $notes]);
        $assignmentId = $pdo->lastInsertId();
        
        // Log in approval history
        $historyStmt = $pdo->prepare("
            INSERT INTO member_approval_history 
            (member_id, performed_by, action, from_status, to_status, assigned_to, notes) 
            VALUES (?, ?, 'assigned', 'pending', 'pending', ?, ?)
        ");
        $historyStmt->execute([$memberId, $currentAdminId, $assignedTo, $notes]);
        
        // Create notification for assigned admin
        $notificationStmt = $pdo->prepare("
            INSERT INTO notifications (type, title, message, target_audience, target_user_id, created_by, priority)
            VALUES ('assignment', 'Member Assigned for Review', ?, 'specific', ?, ?, ?)
        ");
        $message = "Member {$member['first_name']} {$member['last_name']} has been assigned to you for review" . 
                   ($notes ? ". Note: {$notes}" : "");
        $notificationStmt->execute([$message, $assignedTo, $currentAdminId, $priority]);
        $notificationId = $pdo->lastInsertId();
        
        // Create notification recipient
        $recipientStmt = $pdo->prepare("
            INSERT INTO notification_recipients (notification_id, user_id, user_type)
            VALUES (?, ?, 'admin')
        ");
        $recipientStmt->execute([$notificationId, $assignedTo]);
        
        $pdo->commit();
        
        // Log activity
        ActivityLogger::log(
            'member_assigned',
            "Member {$member['first_name']} {$member['last_name']} assigned to {$admin['first_name']} {$admin['last_name']}",
            'member',
            $memberId
        );
        
        ApiResponse::success([
            'assignment_id' => $assignmentId,
            'assigned_to' => $admin,
            'member' => $member,
            'priority' => $priority,
            'due_date' => $dueDate
        ], 'Member assigned successfully');
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function unassignMember($pdo, $memberId, $currentAdminId) {
    try {
        $pdo->beginTransaction();
        
        // Get current assignment
        $assignmentStmt = $pdo->prepare("
            SELECT aa.*, m.first_name, m.last_name, m.assigned_to, m.assigned_by,
                   assigned_admin.first_name as assigned_admin_name,
                   assigned_admin.last_name as assigned_admin_lastname
            FROM admin_assignments aa
            JOIN members m ON aa.member_id = m.id
            LEFT JOIN admins assigned_admin ON m.assigned_to = assigned_admin.id
            WHERE aa.member_id = ? AND aa.status = 'active'
            ORDER BY aa.created_at DESC
            LIMIT 1
        ");
        $assignmentStmt->execute([$memberId]);
        $assignment = $assignmentStmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$assignment) {
            throw new Exception('No active assignment found for this member');
        }
        
        // Check permissions (only assigned admin or assigner can unassign)
        if ($assignment['assigned_to'] != $currentAdminId && $assignment['assigned_by'] != $currentAdminId) {
            throw new Exception('You can only unassign members assigned to you or by you');
        }
        
        // Remove assignment from member
        $updateStmt = $pdo->prepare("
            UPDATE members 
            SET assigned_to = NULL, 
                assigned_by = NULL, 
                assignment_date = NULL,
                assignment_notes = NULL
            WHERE id = ?
        ");
        $updateStmt->execute([$memberId]);
        
        // Mark assignment as cancelled
        $cancelStmt = $pdo->prepare("
            UPDATE admin_assignments 
            SET status = 'cancelled', completed_at = CURRENT_TIMESTAMP 
            WHERE id = ?
        ");
        $cancelStmt->execute([$assignment['id']]);
        
        // Log in approval history
        $historyStmt = $pdo->prepare("
            INSERT INTO member_approval_history 
            (member_id, performed_by, action, from_status, to_status, notes) 
            VALUES (?, ?, 'unassigned', 'pending', 'pending', 'Assignment removed')
        ");
        $historyStmt->execute([$memberId, $currentAdminId]);
        
        $pdo->commit();
        
        // Log activity
        ActivityLogger::log(
            'member_unassigned',
            "Assignment removed for member {$assignment['first_name']} {$assignment['last_name']}",
            'member',
            $memberId
        );
        
        ApiResponse::success([
            'member_id' => $memberId,
            'assignment_id' => $assignment['id']
        ], 'Assignment removed successfully');
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getAssignmentData() {
    global $pdo;
    
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'available_admins':
            getAvailableAdmins($pdo);
            break;
        case 'my_assignments':
            getMyAssignments($pdo);
            break;
        case 'assignment_stats':
            getAssignmentStats($pdo);
            break;
        default:
            ApiResponse::error('Invalid action', 400);
    }
}

function getAvailableAdmins($pdo) {
    $currentAdminId = ApiAuth::getCurrentAdminId();
    
    $stmt = $pdo->prepare("
        SELECT id, first_name, last_name, role, 
               CONCAT(first_name, ' ', last_name) as full_name,
               (SELECT COUNT(*) FROM admin_assignments 
                WHERE assigned_to = admins.id AND status = 'active') as active_assignments
        FROM admins 
        WHERE status = 'active' AND id != ?
        ORDER BY first_name, last_name
    ");
    $stmt->execute([$currentAdminId]);
    $admins = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    ApiResponse::success($admins);
}

function getMyAssignments($pdo) {
    $currentAdminId = ApiAuth::getCurrentAdminId();
    $limit = min((int)($_GET['limit'] ?? 20), 50);
    $offset = (int)($_GET['offset'] ?? 0);
    
    // Get assigned members
    $stmt = $pdo->prepare("
        SELECT m.*, aa.priority, aa.due_date, aa.notes as assignment_notes,
               aa.created_at as assigned_at,
               CONCAT(m.first_name, ' ', m.last_name) as full_name,
               assigner.first_name as assigned_by_name,
               assigner.last_name as assigned_by_lastname,
               CONCAT(assigner.first_name, ' ', assigner.last_name) as assigned_by_full_name
        FROM members m
        JOIN admin_assignments aa ON m.id = aa.member_id
        LEFT JOIN admins assigner ON aa.assigned_by = assigner.id
        WHERE aa.assigned_to = ? AND aa.status = 'active'
        ORDER BY aa.priority DESC, aa.due_date ASC, aa.created_at ASC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$currentAdminId, $limit, $offset]);
    $assignments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format dates
    foreach ($assignments as &$assignment) {
        $assignment['assigned_at_formatted'] = ApiUtils::formatDateTime($assignment['assigned_at']);
        $assignment['due_date_formatted'] = $assignment['due_date'] ? 
            ApiUtils::formatDateTime($assignment['due_date']) : null;
        
        if ($assignment['due_date']) {
            $dueDate = new DateTime($assignment['due_date']);
            $now = new DateTime();
            $assignment['days_until_due'] = $now < $dueDate ? 
                $now->diff($dueDate)->days : -$dueDate->diff($now)->days;
            $assignment['is_overdue'] = $now > $dueDate;
        }
    }
    
    // Get total count
    $countStmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM admin_assignments 
        WHERE assigned_to = ? AND status = 'active'
    ");
    $countStmt->execute([$currentAdminId]);
    $total = $countStmt->fetchColumn();
    
    ApiResponse::success([
        'assignments' => $assignments,
        'total' => (int)$total,
        'limit' => $limit,
        'offset' => $offset
    ]);
}

function getAssignmentStats($pdo) {
    $currentAdminId = ApiAuth::getCurrentAdminId();
    
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_assignments,
            SUM(CASE WHEN priority = 'urgent' THEN 1 ELSE 0 END) as urgent_assignments,
            SUM(CASE WHEN priority = 'high' THEN 1 ELSE 0 END) as high_priority,
            SUM(CASE WHEN due_date < NOW() AND due_date IS NOT NULL THEN 1 ELSE 0 END) as overdue,
            SUM(CASE WHEN due_date BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 3 DAY) THEN 1 ELSE 0 END) as due_soon
        FROM admin_assignments 
        WHERE assigned_to = ? AND status = 'active'
    ");
    $stmt->execute([$currentAdminId]);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Convert to integers
    foreach ($stats as $key => $value) {
        $stats[$key] = (int)$value;
    }
    
    ApiResponse::success($stats);
}
?>


