<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';
require_once '../../../shared/services/EmailService.php';

// Require admin authentication for all member operations
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            getMembers();
            break;
        case 'POST':
            createMember();
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    error_log("Error in members/index.php: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

function getMembers() {
    global $pdo;
    
    try {
        // Get query parameters
        $status = $_GET['status'] ?? '';
        $search = $_GET['search'] ?? '';
        $membership_type = $_GET['membership_type'] ?? '';
        $approval_status = $_GET['approval_status'] ?? '';
        $assigned_to = $_GET['assigned_to'] ?? '';
        $limit = min((int)($_GET['limit'] ?? 50), 100); // Max 100 records
        $offset = (int)($_GET['offset'] ?? 0);
        $sort_by = $_GET['sort_by'] ?? 'created_at';
        $sort_order = strtoupper($_GET['sort_order'] ?? 'DESC');
        
        // Validate sort order
        if (!in_array($sort_order, ['ASC', 'DESC'])) {
            $sort_order = 'DESC';
        }
        
        // Build WHERE clause
        $where_conditions = ['1=1'];
        $params = [];
        
        if (!empty($status)) {
            $where_conditions[] = "m.status = ?";
            $params[] = $status;
        }
        
        if (!empty($approval_status)) {
            $where_conditions[] = "m.approval_status = ?";
            $params[] = $approval_status;
        }
        
        if (!empty($membership_type)) {
            $where_conditions[] = "m.membership_type = ?";
            $params[] = $membership_type;
        }
        
        if (!empty($assigned_to)) {
            if ($assigned_to === 'me') {
                $currentAdminId = ApiAuth::getCurrentAdminId();
                if ($currentAdminId) {
                    $where_conditions[] = "m.assigned_to = ?";
                    $params[] = $currentAdminId;
                }
            } else {
                $where_conditions[] = "m.assigned_to = ?";
                $params[] = $assigned_to;
            }
        }
        
        if (!empty($search)) {
            $where_conditions[] = "(m.first_name LIKE ? OR m.last_name LIKE ? OR m.email LIKE ? OR m.phone LIKE ? OR m.kso_id LIKE ? OR m.college LIKE ?)";
            $search_term = "%$search%";
            $params = array_merge($params, [$search_term, $search_term, $search_term, $search_term, $search_term, $search_term]);
        }
        
        $where_clause = implode(" AND ", $where_conditions);
        
        // Get total count
        $count_stmt = $pdo->prepare("SELECT COUNT(*) as total FROM members m WHERE $where_clause");
        $count_stmt->execute($params);
        $total = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Validate sort column
        $valid_sort_columns = ['kso_id', 'first_name', 'last_name', 'email', 'college', 'membership_type', 'approval_status', 'status', 'created_at'];
        if (!in_array($sort_by, $valid_sort_columns)) {
            $sort_by = 'created_at';
        }
        
        // Get members with pagination
        $stmt = $pdo->prepare("
            SELECT m.*, 
                   CONCAT(m.first_name, ' ', m.last_name) as full_name,
                   approver.first_name as approver_first_name,
                   approver.last_name as approver_last_name,
                   CONCAT(approver.first_name, ' ', approver.last_name) as approver_name,
                   assigner.first_name as assigner_first_name,
                   assigner.last_name as assigner_last_name,
                   CONCAT(assigner.first_name, ' ', assigner.last_name) as assigned_to_name,
                   (SELECT COUNT(*) FROM dependents WHERE member_id = m.id) as dependents_count
            FROM members m
            LEFT JOIN admins approver ON m.approved_by = approver.id
            LEFT JOIN admins assigner ON m.assigned_to = assigner.id
            WHERE $where_clause 
            ORDER BY m.$sort_by $sort_order 
            LIMIT ? OFFSET ?
        ");
        
        $params[] = $limit;
        $params[] = $offset;
        $stmt->execute($params);
        $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format dates and remove sensitive data
        foreach ($members as &$member) {
            // Format dates
            $member['created_at_formatted'] = ApiUtils::formatDateTime($member['created_at']);
            $member['updated_at_formatted'] = ApiUtils::formatDateTime($member['updated_at']);
            $member['approved_at_formatted'] = $member['approved_at'] ? ApiUtils::formatDateTime($member['approved_at']) : null;
            $member['assignment_date_formatted'] = $member['assignment_date'] ? ApiUtils::formatDateTime($member['assignment_date']) : null;
            
            // Calculate age if date of birth is available
            if ($member['date_of_birth']) {
                $dob = new DateTime($member['date_of_birth']);
                $now = new DateTime();
                $member['age'] = $now->diff($dob)->y;
                $member['date_of_birth_formatted'] = date('M j, Y', strtotime($member['date_of_birth']));
            } else {
                $member['age'] = null;
                $member['date_of_birth_formatted'] = null;
            }
            
            // Format membership expiry
            if ($member['membership_expiry_date']) {
                $member['membership_expiry_formatted'] = date('M j, Y', strtotime($member['membership_expiry_date']));
                $expiry = new DateTime($member['membership_expiry_date']);
                $now = new DateTime();
                $member['days_until_expiry'] = $now < $expiry ? $now->diff($expiry)->days : -$expiry->diff($now)->days;
            } else {
                $member['membership_expiry_formatted'] = null;
                $member['days_until_expiry'] = null;
            }
            
            // Remove sensitive data
            unset($member['password_hash'], $member['email_token']);
        }
        
        ApiResponse::success([
            'members' => $members,
            'pagination' => [
                'total' => (int)$total,
                'limit' => $limit,
                'offset' => $offset,
                'has_more' => ($offset + $limit) < $total,
                'current_page' => floor($offset / $limit) + 1,
                'total_pages' => ceil($total / $limit)
            ],
            'filters' => [
                'status' => $status,
                'approval_status' => $approval_status,
                'membership_type' => $membership_type,
                'search' => $search,
                'sort_by' => $sort_by,
                'sort_order' => $sort_order
            ]
        ], 'Members retrieved successfully');
        
    } catch (Exception $e) {
        error_log("Error in getMembers: " . $e->getMessage());
        ApiResponse::error('Failed to retrieve members: ' . $e->getMessage(), 500);
    }
}

function createMember() {
    global $pdo;
    
    // Require admin authentication for member creation
    ApiAuth::requireAdminAuth();
    
    try {
        // Get input data
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            ApiResponse::error('Invalid JSON input', 400);
        }
        
        // Sanitize input
        $input = ApiValidator::sanitizeInput($input);
        
        // Validate required fields
        ApiValidator::validateRequired($input, [
            'first_name', 'last_name', 'email', 'college', 'course', 'membership_type'
        ]);
        
        // Validate email format
        ApiValidator::validateEmail($input['email']);
        
        // Validate phone if provided
        if (!empty($input['phone'])) {
            ApiValidator::validatePhone($input['phone']);
        }
        
        // Validate membership type
        if (!ApiUtils::validateMembershipType($input['membership_type'])) {
            ApiResponse::error('Invalid membership type. Must be Individual or Family', 400);
        }
        
        // Validate date of birth if provided
        if (!empty($input['date_of_birth'])) {
            if (!ApiValidator::validateDateFormat($input['date_of_birth'])) {
                ApiResponse::error('Invalid date of birth format. Use YYYY-MM-DD', 400);
            }
        }
        
        // Check if email already exists
        $stmt = $pdo->prepare("SELECT id FROM members WHERE email = ?");
        $stmt->execute([$input['email']]);
        if ($stmt->fetch()) {
            ApiResponse::error('Email already exists', 409);
        }
        
        // Generate password hash (temporary password)
        $temp_password = bin2hex(random_bytes(8));
        $password_hash = password_hash($temp_password, PASSWORD_DEFAULT);
        
        // Generate email verification token
        $email_token = bin2hex(random_bytes(32));
        
        // Set default session years if not provided
        $session_from_year = $input['session_from_year'] ?? date('Y');
        $session_to_year = $input['session_to_year'] ?? (date('Y') + 1);
        
        // Insert new member (KSO ID will be auto-generated by trigger)
        $stmt = $pdo->prepare("
            INSERT INTO members (
                first_name, last_name, nickname, email, password_hash, email_token,
                phone, date_of_birth, gender, blood_group, college, course, 
                year_of_study, address, emergency_contact_name, emergency_contact,
                membership_type, session_from_year, session_to_year, notes,
                approval_status, status
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', 'pending')
        ");
        
        $success = $stmt->execute([
            $input['first_name'],
            $input['last_name'],
            $input['nickname'] ?? null,
            $input['email'],
            $password_hash,
            $email_token,
            $input['phone'] ?? null,
            $input['date_of_birth'] ?? null,
            $input['gender'] ?? null,
            $input['blood_group'] ?? null,
            $input['college'],
            $input['course'],
            $input['year_of_study'] ?? null,
            $input['address'] ?? null,
            $input['emergency_contact_name'] ?? null,
            $input['emergency_contact'] ?? null,
            $input['membership_type'],
            $session_from_year,
            $session_to_year,
            $input['notes'] ?? null
        ]);
        
        if ($success) {
            $member_id = $pdo->lastInsertId();
            
            // Get the generated KSO ID
            $stmt = $pdo->prepare("SELECT kso_id FROM members WHERE id = ?");
            $stmt->execute([$member_id]);
            $kso_id = $stmt->fetchColumn();
            
            // Log activity
            ActivityLogger::log(
                'member_created',
                "Created new member: {$input['first_name']} {$input['last_name']} ({$kso_id})",
                'member',
                $member_id
            );
            
            // Send welcome email
            $emailSent = false;
            try {
                $memberData = [
                    'first_name' => $input['first_name'],
                    'last_name' => $input['last_name'],
                    'kso_id' => $kso_id,
                    'membership_type' => $input['membership_type'],
                    'email' => $input['email']
                ];
                $emailSent = EmailNotifications::sendWelcomeEmail($memberData);
                if ($emailSent) {
                    error_log("Welcome email sent successfully to {$input['email']}");
                } else {
                    error_log("Failed to send welcome email to {$input['email']}");
                }
            } catch (Exception $emailError) {
                error_log("Email notification error: " . $emailError->getMessage());
                // Don't fail the registration if email fails
            }
            
            ApiResponse::success([
                'id' => (int)$member_id,
                'kso_id' => $kso_id,
                'temp_password' => $temp_password, // In production, send via email
                'email_token' => $email_token,
                'email_sent' => $emailSent
            ], 'Member created successfully', 201);
        } else {
            ApiResponse::error('Failed to create member', 500);
        }
        
    } catch (Exception $e) {
        error_log("Error in createMember: " . $e->getMessage());
        ApiResponse::error('Failed to create member: ' . $e->getMessage(), 500);
    }
}
?>
        
        if (!empty($search)) {
            $where_conditions[] = "(first_name LIKE ? OR last_name LIKE ? OR email LIKE ? OR phone LIKE ? OR kso_id LIKE ?)";
            $search_term = "%$search%";
            $params = array_merge($params, [$search_term, $search_term, $search_term, $search_term, $search_term]);
        }
        
        $where_clause = implode(" AND ", $where_conditions);
        
        // Get total count
        $count_stmt = $pdo->prepare("SELECT COUNT(*) as total FROM members WHERE $where_clause");
        $count_stmt->execute($params);
        $total = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get members with pagination
        $stmt = $pdo->prepare("
            SELECT m.*, 
                   CONCAT(m.first_name, ' ', m.last_name) as full_name,
                   approver.first_name as approver_first_name,
                   approver.last_name as approver_last_name,
                   assigner.first_name as assigner_first_name,
                   assigner.last_name as assigner_last_name
            FROM members m
            LEFT JOIN admins approver ON m.approved_by = approver.id
            LEFT JOIN admins assigner ON m.assigned_to = assigner.id
            WHERE $where_clause 
            ORDER BY m.created_at DESC 
            LIMIT ? OFFSET ?
        ");
        
        $params[] = $limit;
        $params[] = $offset;
        $stmt->execute($params);
        $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format dates and remove sensitive data
        foreach ($members as &$member) {
            $member['created_at_formatted'] = date('M j, Y g:i A', strtotime($member['created_at']));
            $member['updated_at_formatted'] = date('M j, Y g:i A', strtotime($member['updated_at']));
            
            if ($member['approved_at']) {
                $member['approved_at_formatted'] = date('M j, Y g:i A', strtotime($member['approved_at']));
            }
            
            if ($member['date_of_birth']) {
                $member['age'] = date_diff(date_create($member['date_of_birth']), date_create('today'))->y;
            }
            
            // Remove sensitive data
            unset($member['password_hash'], $member['email_token']);
        }
        
        ApiResponse::success([
            'members' => $members,
            'pagination' => [
                'total' => (int)$total,
                'limit' => $limit,
                'offset' => $offset,
                'has_more' => ($offset + $limit) < $total
            ]
        ], 'Members retrieved successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to retrieve members: ' . $e->getMessage(), 500);
    }
}

function createMember() {
    global $pdo;
    
    // Require admin authentication for member creation
    ApiAuth::requireAdminAuth();
    
    try {
        // Get input data
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            ApiResponse::error('Invalid JSON input', 400);
        }
        
        // Sanitize input
        $input = ApiValidator::sanitizeInput($input);
        
        // Validate required fields
        ApiValidator::validateRequired($input, [
            'first_name', 'last_name', 'email', 'college', 'course', 'membership_type'
        ]);
        
        // Validate email format
        ApiValidator::validateEmail($input['email']);
        
        // Validate phone if provided
        if (!empty($input['phone'])) {
            ApiValidator::validatePhone($input['phone']);
        }
        
        // Check if email already exists
        $stmt = $pdo->prepare("SELECT id FROM members WHERE email = ?");
        $stmt->execute([$input['email']]);
        if ($stmt->fetch()) {
            ApiResponse::error('Email already exists', 409);
        }
        
        // Generate password hash (temporary password)
        $temp_password = bin2hex(random_bytes(8));
        $password_hash = password_hash($temp_password, PASSWORD_DEFAULT);
        
        // Generate email verification token
        $email_token = bin2hex(random_bytes(32));
        
        // Insert new member (KSO ID will be auto-generated by trigger)
        $stmt = $pdo->prepare("
            INSERT INTO members (
                first_name, last_name, nickname, email, password_hash, email_token,
                phone, date_of_birth, gender, blood_group, college, course, 
                year_of_study, address, emergency_contact_name, emergency_contact,
                membership_type, session_from_year, session_to_year, notes
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $success = $stmt->execute([
            $input['first_name'],
            $input['last_name'],
            $input['nickname'] ?? null,
            $input['email'],
            $password_hash,
            $email_token,
            $input['phone'] ?? null,
            $input['date_of_birth'] ?? null,
            $input['gender'] ?? null,
            $input['blood_group'] ?? null,
            $input['college'],
            $input['course'],
            $input['year_of_study'] ?? null,
            $input['address'] ?? null,
            $input['emergency_contact_name'] ?? null,
            $input['emergency_contact'] ?? null,
            $input['membership_type'],
            $input['session_from_year'] ?? date('Y'),
            $input['session_to_year'] ?? (date('Y') + 1),
            $input['notes'] ?? null
        ]);
        
        if ($success) {
            $member_id = $pdo->lastInsertId();
            
            // Get the generated KSO ID
            $stmt = $pdo->prepare("SELECT kso_id FROM members WHERE id = ?");
            $stmt->execute([$member_id]);
            $kso_id = $stmt->fetchColumn();
            
            // Log activity
            ActivityLogger::log(
                'member_created',
                "Created new member: {$input['first_name']} {$input['last_name']} ({$kso_id})",
                'member',
                $member_id
            );
            
            ApiResponse::success([
                'id' => $member_id,
                'kso_id' => $kso_id,
                'temp_password' => $temp_password // In production, send via email
            ], 'Member created successfully', 201);
        } else {
            ApiResponse::error('Failed to create member', 500);
        }
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to create member: ' . $e->getMessage(), 500);
    }
}
?>


