<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

$method = $_SERVER['REQUEST_METHOD'];
$member_id = $_GET['id'] ?? null;

if (!$member_id) {
    ApiResponse::error('Member ID is required', 400);
}

try {
    switch ($method) {
        case 'GET':
            getMember($member_id);
            break;
        case 'PUT':
            updateMember($member_id);
            break;
        case 'DELETE':
            deleteMember($member_id);
            break;
        default:
            ApiResponse::error('Method not allowed', 405);
    }
} catch (Exception $e) {
    error_log("Error in members/member.php: " . $e->getMessage());
    ApiResponse::error('Internal server error: ' . $e->getMessage(), 500);
}

function getMember($member_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT m.*, 
                   CONCAT(m.first_name, ' ', m.last_name) as full_name,
                   approver.first_name as approver_first_name,
                   approver.last_name as approver_last_name,
                   CONCAT(approver.first_name, ' ', approver.last_name) as approver_name,
                   assigner.first_name as assigner_first_name,
                   assigner.last_name as assigner_last_name,
                   CONCAT(assigner.first_name, ' ', assigner.last_name) as assigned_to_name
            FROM members m
            LEFT JOIN admins approver ON m.approved_by = approver.id
            LEFT JOIN admins assigner ON m.assigned_to = assigner.id
            WHERE m.id = ?
        ");
        
        $stmt->execute([$member_id]);
        $member = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member) {
            ApiResponse::error('Member not found', 404);
        }
        
        // Format dates
        $member['created_at_formatted'] = ApiUtils::formatDateTime($member['created_at']);
        $member['updated_at_formatted'] = ApiUtils::formatDateTime($member['updated_at']);
        $member['approved_at_formatted'] = $member['approved_at'] ? ApiUtils::formatDateTime($member['approved_at']) : null;
        $member['assignment_date_formatted'] = $member['assignment_date'] ? ApiUtils::formatDateTime($member['assignment_date']) : null;
        
        // Calculate age if date of birth is available
        if ($member['date_of_birth']) {
            $dob = new DateTime($member['date_of_birth']);
            $now = new DateTime();
            $member['age'] = $now->diff($dob)->y;
            $member['date_of_birth_formatted'] = date('M j, Y', strtotime($member['date_of_birth']));
        }
        
        // Format membership expiry
        if ($member['membership_expiry_date']) {
            $member['membership_expiry_formatted'] = date('M j, Y', strtotime($member['membership_expiry_date']));
            $expiry = new DateTime($member['membership_expiry_date']);
            $now = new DateTime();
            $member['days_until_expiry'] = $now < $expiry ? $now->diff($expiry)->days : -$expiry->diff($now)->days;
        }
        
        // Get dependents if family membership
        if ($member['membership_type'] === 'Family') {
            $stmt = $pdo->prepare("SELECT * FROM dependents WHERE member_id = ? ORDER BY relationship, first_name");
            $stmt->execute([$member_id]);
            $member['dependents'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($member['dependents'] as &$dependent) {
                if ($dependent['date_of_birth']) {
                    $dob = new DateTime($dependent['date_of_birth']);
                    $now = new DateTime();
                    $dependent['age'] = $now->diff($dob)->y;
                    $dependent['date_of_birth_formatted'] = date('M j, Y', strtotime($dependent['date_of_birth']));
                }
                $dependent['created_at_formatted'] = ApiUtils::formatDateTime($dependent['created_at']);
            }
        } else {
            $member['dependents'] = [];
        }
        
        // Remove sensitive data
        unset($member['password_hash'], $member['email_token']);
        
        ApiResponse::success($member, 'Member retrieved successfully');
        
    } catch (Exception $e) {
        error_log("Error in getMember: " . $e->getMessage());
        ApiResponse::error('Failed to retrieve member: ' . $e->getMessage(), 500);
    }
}

function updateMember($member_id) {
    global $pdo;
    
    // Require admin authentication
    ApiAuth::requireAdminAuth();
    
    try {
        // Check if member exists
        $stmt = $pdo->prepare("SELECT id, kso_id, first_name, last_name FROM members WHERE id = ?");
        $stmt->execute([$member_id]);
        $existing_member = $stmt->fetch();
        
        if (!$existing_member) {
            ApiResponse::error('Member not found', 404);
        }
        
        // Get input data
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            ApiResponse::error('Invalid JSON input', 400);
        }
        
        // Sanitize input
        $input = ApiValidator::sanitizeInput($input);
        
        // Validate email format if provided
        if (!empty($input['email'])) {
            ApiValidator::validateEmail($input['email']);
            
            // Check if email already exists for another member
            $stmt = $pdo->prepare("SELECT id FROM members WHERE email = ? AND id != ?");
            $stmt->execute([$input['email'], $member_id]);
            if ($stmt->fetch()) {
                ApiResponse::error('Email already exists for another member', 409);
            }
        }
        
        // Validate phone if provided
        if (!empty($input['phone'])) {
            ApiValidator::validatePhone($input['phone']);
        }
        
        // Validate membership type if provided
        if (!empty($input['membership_type']) && !ApiUtils::validateMembershipType($input['membership_type'])) {
            ApiResponse::error('Invalid membership type. Must be Individual or Family', 400);
        }
        
        // Validate approval status if provided
        if (!empty($input['approval_status']) && !ApiUtils::validateApprovalStatus($input['approval_status'])) {
            ApiResponse::error('Invalid approval status', 400);
        }
        
        // Validate member status if provided
        if (!empty($input['status']) && !ApiUtils::validateMemberStatus($input['status'])) {
            ApiResponse::error('Invalid member status', 400);
        }
        
        // Validate date of birth if provided
        if (!empty($input['date_of_birth'])) {
            if (!ApiValidator::validateDateFormat($input['date_of_birth'])) {
                ApiResponse::error('Invalid date of birth format. Use YYYY-MM-DD', 400);
            }
        }
        
        // Build update query dynamically
        $update_fields = [];
        $params = [];
        
        $allowed_fields = [
            'first_name', 'last_name', 'nickname', 'email', 'phone', 'date_of_birth',
            'gender', 'blood_group', 'college', 'course', 'year_of_study', 'address',
            'emergency_contact_name', 'emergency_contact', 'membership_type',
            'session_from_year', 'session_to_year', 'notes', 'approval_status', 
            'status', 'membership_expiry_date'
        ];
        
        foreach ($allowed_fields as $field) {
            if (array_key_exists($field, $input)) {
                $update_fields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($update_fields)) {
            ApiResponse::error('No valid fields to update', 400);
        }
        
        // Add updated_at
        $update_fields[] = "updated_at = NOW()";
        $params[] = $member_id;
        
        $sql = "UPDATE members SET " . implode(', ', $update_fields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $success = $stmt->execute($params);
        
        if ($success) {
            // Log activity
            $changes = [];
            foreach ($allowed_fields as $field) {
                if (array_key_exists($field, $input)) {
                    $changes[] = $field;
                }
            }
            
            ActivityLogger::log(
                'member_updated',
                "Updated member {$existing_member['kso_id']} ({$existing_member['first_name']} {$existing_member['last_name']}). Fields: " . implode(', ', $changes),
                'member',
                $member_id
            );
            
            ApiResponse::success(['id' => (int)$member_id], 'Member updated successfully');
        } else {
            ApiResponse::error('Failed to update member', 500);
        }
        
    } catch (Exception $e) {
        error_log("Error in updateMember: " . $e->getMessage());
        ApiResponse::error('Failed to update member: ' . $e->getMessage(), 500);
    }
}

function deleteMember($member_id) {
    global $pdo;
    
    // Require admin authentication
    ApiAuth::requireAdminAuth();
    
    try {
        // Check if member exists
        $stmt = $pdo->prepare("SELECT id, kso_id, first_name, last_name FROM members WHERE id = ?");
        $stmt->execute([$member_id]);
        $member = $stmt->fetch();
        
        if (!$member) {
            ApiResponse::error('Member not found', 404);
        }
        
        // Begin transaction
        $pdo->beginTransaction();
        
        try {
            // Delete dependents first (if any)
            $stmt = $pdo->prepare("DELETE FROM dependents WHERE member_id = ?");
            $stmt->execute([$member_id]);
            
            // Delete the member
            $stmt = $pdo->prepare("DELETE FROM members WHERE id = ?");
            $stmt->execute([$member_id]);
            
            // Commit transaction
            $pdo->commit();
            
            // Log activity
            ActivityLogger::log(
                'member_deleted',
                "Deleted member {$member['kso_id']} ({$member['first_name']} {$member['last_name']})",
                'member',
                $member_id
            );
            
            ApiResponse::success(['id' => (int)$member_id], 'Member deleted successfully');
            
        } catch (Exception $e) {
            $pdo->rollBack();
            throw $e;
        }
        
    } catch (Exception $e) {
        error_log("Error in deleteMember: " . $e->getMessage());
        ApiResponse::error('Failed to delete member: ' . $e->getMessage(), 500);
    }
}
?>


