<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ApiResponse::error('Method not allowed', 405);
}

// Require admin authentication
ApiAuth::requireAdminAuth();

try {
    // Get input data
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        ApiResponse::error('Invalid JSON input', 400);
    }
    
    // Validate required fields
    ApiValidator::validateRequired($input, ['member_id', 'rejection_reason']);
    
    $member_id = (int)$input['member_id'];
    $rejection_reason = ApiValidator::sanitizeInput($input['rejection_reason']);
    $notes = ApiValidator::sanitizeInput($input['notes'] ?? '');
    $current_admin_id = ApiAuth::getCurrentAdminId();
    
    global $pdo;
    
    // Check if member exists and can be rejected
    $stmt = $pdo->prepare("
        SELECT id, kso_id, first_name, last_name, email, approval_status 
        FROM members 
        WHERE id = ?
    ");
    $stmt->execute([$member_id]);
    $member = $stmt->fetch();
    
    if (!$member) {
        ApiResponse::error('Member not found', 404);
    }
    
    if ($member['approval_status'] !== 'pending' && $member['approval_status'] !== 'under_review') {
        ApiResponse::error('Member is not in a state that can be rejected', 400);
    }
    
    // Begin transaction
    $pdo->beginTransaction();
    
    try {
        // Update member status to rejected
        $stmt = $pdo->prepare("
            UPDATE members 
            SET approval_status = 'rejected',
                status = 'rejected',
                approved_by = ?,
                approved_at = NOW(),
                rejection_reason = ?,
                assignment_notes = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        
        $success = $stmt->execute([
            $current_admin_id,
            $rejection_reason,
            $notes,
            $member_id
        ]);
        
        if (!$success) {
            throw new Exception('Failed to update member status');
        }
        
        // Log activity
        ActivityLogger::log(
            'member_rejected',
            "Rejected member {$member['kso_id']} ({$member['first_name']} {$member['last_name']}). Reason: $rejection_reason" . 
            (!empty($notes) ? ". Notes: $notes" : ""),
            'member',
            $member_id,
            'WARNING'
        );
        
        // Commit transaction
        $pdo->commit();
        
        ApiResponse::success([
            'member_id' => $member_id,
            'kso_id' => $member['kso_id'],
            'status' => 'rejected',
            'rejection_reason' => $rejection_reason
        ], 'Member rejected successfully');
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Error in members/reject.php: " . $e->getMessage());
    ApiResponse::error('Failed to reject member: ' . $e->getMessage(), 500);
}
?>


