<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                getNotification($_GET['id']);
            } else {
                getNotifications();
            }
            break;
        case 'POST':
            createNotification();
            break;
        case 'PUT':
            if (isset($_GET['id']) && isset($_GET['action']) && $_GET['action'] === 'mark_read') {
                markAsRead($_GET['id']);
            } elseif (isset($_GET['id'])) {
                updateNotification($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Notification ID required for update']);
            }
            break;
        case 'DELETE':
            if (isset($_GET['id'])) {
                deleteNotification($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Notification ID required for delete']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getNotifications() {
    global $pdo;
    
    try {
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = ($page - 1) * $limit;
        
        $search = isset($_GET['search']) ? $_GET['search'] : '';
        $type = isset($_GET['type']) ? $_GET['type'] : '';
        $status = isset($_GET['status']) ? $_GET['status'] : '';
        $priority = isset($_GET['priority']) ? $_GET['priority'] : '';
        
        $whereConditions = [];
        $params = [];
        
        if (!empty($search)) {
            $whereConditions[] = "(title LIKE ? OR message LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        if (!empty($type)) {
            $whereConditions[] = "type = ?";
            $params[] = $type;
        }
        
        if (!empty($status)) {
            $whereConditions[] = "status = ?";
            $params[] = $status;
        }
        
        if (!empty($priority)) {
            $whereConditions[] = "priority = ?";
            $params[] = $priority;
        }
        
        $whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";
        
        // Get total count
        $countSql = "SELECT COUNT(*) as total FROM notifications $whereClause";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute($params);
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get notifications
        $sql = "SELECT * FROM notifications $whereClause ORDER BY created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $notifications,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getNotification($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM notifications WHERE id = ?");
        $stmt->execute([$id]);
        $notification = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$notification) {
            http_response_code(404);
            echo json_encode(['error' => 'Notification not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'data' => $notification
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function createNotification() {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Validate required fields
        $required = ['title', 'message', 'type'];
        foreach ($required as $field) {
            if (!isset($input[$field]) || empty($input[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Field '$field' is required"]);
                return;
            }
        }
        
        // Validate type
        $validTypes = ['system', 'member', 'event', 'finance', 'general'];
        if (!in_array($input['type'], $validTypes)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid notification type']);
            return;
        }
        
        // Validate priority
        $priority = $input['priority'] ?? 'medium';
        $validPriorities = ['low', 'medium', 'high', 'urgent'];
        if (!in_array($priority, $validPriorities)) {
            $priority = 'medium';
        }
        
        $sql = "INSERT INTO notifications (title, message, type, priority, target_audience, status, created_by, created_at) VALUES (?, ?, ?, ?, ?, 'active', ?, NOW())";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['title'],
            $input['message'],
            $input['type'],
            $priority,
            $input['target_audience'] ?? 'all',
            $_SESSION['admin_id']
        ]);
        
        $notificationId = $pdo->lastInsertId();
        
        // Get the created notification
        $stmt = $pdo->prepare("SELECT * FROM notifications WHERE id = ?");
        $stmt->execute([$notificationId]);
        $notification = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Notification created successfully',
            'data' => $notification
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateNotification($id) {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Check if notification exists
        $stmt = $pdo->prepare("SELECT id FROM notifications WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Notification not found']);
            return;
        }
        
        $updateFields = [];
        $params = [];
        
        $allowedFields = ['title', 'message', 'type', 'priority', 'target_audience', 'status'];
        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateFields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($updateFields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        $updateFields[] = "updated_at = NOW()";
        $params[] = $id;
        
        $sql = "UPDATE notifications SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Get updated notification
        $stmt = $pdo->prepare("SELECT * FROM notifications WHERE id = ?");
        $stmt->execute([$id]);
        $notification = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Notification updated successfully',
            'data' => $notification
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function markAsRead($id) {
    global $pdo;
    
    try {
        // Check if notification exists
        $stmt = $pdo->prepare("SELECT id FROM notifications WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Notification not found']);
            return;
        }
        
        $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1, read_at = NOW() WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Notification marked as read'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteNotification($id) {
    global $pdo;
    
    try {
        // Check if notification exists
        $stmt = $pdo->prepare("SELECT id FROM notifications WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Notification not found']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM notifications WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Notification deleted successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}
?>


