<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../../../shared/api/config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['type'])) {
                generateReport($_GET['type']);
            } else {
                getAvailableReports();
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getAvailableReports() {
    $reports = [
        [
            'id' => 'members_summary',
            'name' => 'Members Summary Report',
            'description' => 'Overview of all members with statistics',
            'category' => 'Members'
        ],
        [
            'id' => 'financial_summary',
            'name' => 'Financial Summary Report',
            'description' => 'Income, expenses, and financial overview',
            'category' => 'Finance'
        ],
        [
            'id' => 'events_summary',
            'name' => 'Events Summary Report',
            'description' => 'Events and registrations overview',
            'category' => 'Events'
        ],
        [
            'id' => 'membership_fees',
            'name' => 'Membership Fees Report',
            'description' => 'Fee collection status and outstanding amounts',
            'category' => 'Finance'
        ],
        [
            'id' => 'expenses_report',
            'name' => 'Expenses Report',
            'description' => 'Detailed expense breakdown by category',
            'category' => 'Finance'
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $reports
    ]);
}

function generateReport($type) {
    global $pdo;
    
    $dateFrom = $_GET['date_from'] ?? date('Y-01-01');
    $dateTo = $_GET['date_to'] ?? date('Y-12-31');
    $format = $_GET['format'] ?? 'json'; // json or csv
    
    try {
        switch ($type) {
            case 'members_summary':
                $data = generateMembersSummaryReport($dateFrom, $dateTo);
                break;
            case 'financial_summary':
                $data = generateFinancialSummaryReport($dateFrom, $dateTo);
                break;
            case 'events_summary':
                $data = generateEventsSummaryReport($dateFrom, $dateTo);
                break;
            case 'membership_fees':
                $data = generateMembershipFeesReport($dateFrom, $dateTo);
                break;
            case 'expenses_report':
                $data = generateExpensesReport($dateFrom, $dateTo);
                break;
            default:
                http_response_code(400);
                echo json_encode(['error' => 'Invalid report type']);
                return;
        }
        
        if ($format === 'csv') {
            outputCSV($data, $type);
        } else {
            echo json_encode([
                'success' => true,
                'data' => $data,
                'generated_at' => date('Y-m-d H:i:s'),
                'date_range' => ['from' => $dateFrom, 'to' => $dateTo]
            ]);
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function generateMembersSummaryReport($dateFrom, $dateTo) {
    global $pdo;
    
    $report = [
        'summary' => [],
        'details' => [],
        'statistics' => []
    ];
    
    // Get member statistics
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_members,
            SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_members,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_members,
            SUM(CASE WHEN membership_type = 'individual' THEN 1 ELSE 0 END) as individual_members,
            SUM(CASE WHEN membership_type = 'family' THEN 1 ELSE 0 END) as family_members
        FROM members 
        WHERE created_at BETWEEN ? AND ?
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['summary'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get member details
    $stmt = $pdo->prepare("
        SELECT 
            kso_id, first_name, last_name, email, phone, 
            membership_type, status, created_at
        FROM members 
        WHERE created_at BETWEEN ? AND ?
        ORDER BY created_at DESC
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['details'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return $report;
}

function generateFinancialSummaryReport($dateFrom, $dateTo) {
    global $pdo;
    
    $report = [
        'summary' => [],
        'transactions' => [],
        'monthly_breakdown' => []
    ];
    
    // Get financial summary
    $stmt = $pdo->prepare("
        SELECT 
            SUM(CASE WHEN transaction_type = 'income' THEN amount ELSE 0 END) as total_income,
            SUM(CASE WHEN transaction_type = 'expense' THEN amount ELSE 0 END) as total_expenses,
            COUNT(*) as total_transactions
        FROM finance_transactions 
        WHERE transaction_date BETWEEN ? AND ?
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $summary = $stmt->fetch(PDO::FETCH_ASSOC);
    $summary['net_balance'] = $summary['total_income'] - $summary['total_expenses'];
    $report['summary'] = $summary;
    
    // Get transactions
    $stmt = $pdo->prepare("
        SELECT * FROM finance_transactions 
        WHERE transaction_date BETWEEN ? AND ?
        ORDER BY transaction_date DESC
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['transactions'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return $report;
}

function generateEventsSummaryReport($dateFrom, $dateTo) {
    global $pdo;
    
    $report = [
        'summary' => [],
        'events' => [],
        'registrations' => []
    ];
    
    // Get events summary
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_events,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_events,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_events
        FROM events 
        WHERE event_date BETWEEN ? AND ?
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['summary'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get events with registration counts
    $stmt = $pdo->prepare("
        SELECT 
            e.*,
            COUNT(er.id) as registration_count
        FROM events e 
        LEFT JOIN event_registrations er ON e.id = er.event_id
        WHERE e.event_date BETWEEN ? AND ?
        GROUP BY e.id
        ORDER BY e.event_date DESC
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['events'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return $report;
}

function generateMembershipFeesReport($dateFrom, $dateTo) {
    global $pdo;
    
    $report = [
        'summary' => [],
        'fees' => []
    ];
    
    // Get fees summary
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_fees,
            SUM(amount) as total_amount,
            SUM(CASE WHEN status = 'paid' THEN amount ELSE 0 END) as paid_amount,
            SUM(CASE WHEN status = 'pending' THEN amount ELSE 0 END) as pending_amount,
            COUNT(CASE WHEN status = 'paid' THEN 1 END) as paid_count,
            COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_count
        FROM membership_fees 
        WHERE created_at BETWEEN ? AND ?
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['summary'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get detailed fees
    $stmt = $pdo->prepare("
        SELECT 
            mf.*,
            m.first_name, m.last_name, m.kso_id
        FROM membership_fees mf
        JOIN members m ON mf.member_id = m.id
        WHERE mf.created_at BETWEEN ? AND ?
        ORDER BY mf.fee_year DESC, mf.created_at DESC
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['fees'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return $report;
}

function generateExpensesReport($dateFrom, $dateTo) {
    global $pdo;
    
    $report = [
        'summary' => [],
        'expenses' => [],
        'by_category' => []
    ];
    
    // Get expenses summary
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_expenses,
            SUM(amount) as total_amount,
            AVG(amount) as average_amount
        FROM expenses 
        WHERE expense_date BETWEEN ? AND ?
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['summary'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get expenses by category
    $stmt = $pdo->prepare("
        SELECT 
            category,
            COUNT(*) as count,
            SUM(amount) as total_amount
        FROM expenses 
        WHERE expense_date BETWEEN ? AND ?
        GROUP BY category
        ORDER BY total_amount DESC
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['by_category'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get detailed expenses
    $stmt = $pdo->prepare("
        SELECT * FROM expenses 
        WHERE expense_date BETWEEN ? AND ?
        ORDER BY expense_date DESC
    ");
    $stmt->execute([$dateFrom, $dateTo]);
    $report['expenses'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return $report;
}

function outputCSV($data, $reportType) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $reportType . '_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    switch ($reportType) {
        case 'members_summary':
            fputcsv($output, ['KSO ID', 'First Name', 'Last Name', 'Email', 'Phone', 'Membership Type', 'Status', 'Created At']);
            foreach ($data['details'] as $member) {
                fputcsv($output, [
                    $member['kso_id'],
                    $member['first_name'],
                    $member['last_name'],
                    $member['email'],
                    $member['phone'],
                    $member['membership_type'],
                    $member['status'],
                    $member['created_at']
                ]);
            }
            break;
        case 'financial_summary':
            fputcsv($output, ['Date', 'Type', 'Amount', 'Description', 'Category', 'Reference']);
            foreach ($data['transactions'] as $transaction) {
                fputcsv($output, [
                    $transaction['transaction_date'],
                    $transaction['transaction_type'],
                    $transaction['amount'],
                    $transaction['description'],
                    $transaction['category'],
                    $transaction['reference_number']
                ]);
            }
            break;
        // Add more cases for other report types
    }
    
    fclose($output);
    exit;
}
            'description' => 'Events and registrations overview',
            'category' => 'Events'
        ],
        [
            'id' => 'membership_fees',
            'name' => 'Membership Fees Report',
            'description' => 'Membership fee payments and pending dues',
            'category' => 'Finance'
        ],
        [
            'id' => 'expenses_summary',
            'name' => 'Expenses Summary Report',
            'description' => 'Expenses breakdown by category and time',
            'category' => 'Finance'
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $reports
    ]);
}

function generateReport($type) {
    global $pdo;
    
    try {
        $dateFrom = $_GET['date_from'] ?? '';
        $dateTo = $_GET['date_to'] ?? '';
        $format = $_GET['format'] ?? 'json'; // json, csv, pdf
        
        switch ($type) {
            case 'members_summary':
                $data = generateMembersSummaryReport($dateFrom, $dateTo);
                break;
            case 'financial_summary':
                $data = generateFinancialSummaryReport($dateFrom, $dateTo);
                break;
            case 'events_summary':
                $data = generateEventsSummaryReport($dateFrom, $dateTo);
                break;
            case 'membership_fees':
                $data = generateMembershipFeesReport($dateFrom, $dateTo);
                break;
            case 'expenses_summary':
                $data = generateExpensesSummaryReport($dateFrom, $dateTo);
                break;
            default:
                http_response_code(400);
                echo json_encode(['error' => 'Invalid report type']);
                return;
        }
        
        if ($format === 'csv') {
            outputCSV($data, $type);
        } else {
            echo json_encode([
                'success' => true,
                'data' => $data,
                'generated_at' => date('Y-m-d H:i:s'),
                'report_type' => $type
            ]);
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function generateMembersSummaryReport($dateFrom, $dateTo) {
    global $pdo;
    
    $whereClause = '';
    $params = [];
    
    if (!empty($dateFrom) || !empty($dateTo)) {
        $conditions = [];
        if (!empty($dateFrom)) {
            $conditions[] = "created_at >= ?";
            $params[] = $dateFrom;
        }
        if (!empty($dateTo)) {
            $conditions[] = "created_at <= ?";
            $params[] = $dateTo . ' 23:59:59';
        }
        $whereClause = "WHERE " . implode(" AND ", $conditions);
    }
    
    // Get summary statistics
    $stats = [];
    
    // Total members
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM members $whereClause");
    $stmt->execute($params);
    $stats['total_members'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Members by status
    $stmt = $pdo->prepare("SELECT status, COUNT(*) as count FROM members $whereClause GROUP BY status");
    $stmt->execute($params);
    $stats['by_status'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Members by type
    $stmt = $pdo->prepare("SELECT membership_type, COUNT(*) as count FROM members $whereClause GROUP BY membership_type");
    $stmt->execute($params);
    $stats['by_type'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Recent registrations (last 30 days)
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM members WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
    $stmt->execute();
    $stats['recent_registrations'] = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    // Get detailed member list
    $stmt = $pdo->prepare("SELECT * FROM members $whereClause ORDER BY created_at DESC");
    $stmt->execute($params);
    $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'summary' => $stats,
        'members' => $members
    ];
}

function generateFinancialSummaryReport($dateFrom, $dateTo) {
    global $pdo;
    
    $whereClause = '';
    $params = [];
    
    if (!empty($dateFrom) || !empty($dateTo)) {
        $conditions = [];
        if (!empty($dateFrom)) {
            $conditions[] = "transaction_date >= ?";
            $params[] = $dateFrom;
        }
        if (!empty($dateTo)) {
            $conditions[] = "transaction_date <= ?";
            $params[] = $dateTo;
        }
        $whereClause = "WHERE " . implode(" AND ", $conditions);
    }
    
    // Get financial summary
    $stmt = $pdo->prepare("
        SELECT 
            transaction_type,
            SUM(amount) as total,
            COUNT(*) as count
        FROM finance_transactions 
        $whereClause
        GROUP BY transaction_type
    ");
    $stmt->execute($params);
    $summary = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $income = 0;
    $expenses = 0;
    
    foreach ($summary as $item) {
        if ($item['transaction_type'] === 'income') {
            $income = $item['total'];
        } else {
            $expenses = $item['total'];
        }
    }
    
    // Get transactions by category
    $stmt = $pdo->prepare("
        SELECT 
            category,
            transaction_type,
            SUM(amount) as total,
            COUNT(*) as count
        FROM finance_transactions 
        $whereClause
        GROUP BY category, transaction_type
        ORDER BY total DESC
    ");
    $stmt->execute($params);
    $byCategory = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get monthly trends
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(transaction_date, '%Y-%m') as month,
            transaction_type,
            SUM(amount) as total
        FROM finance_transactions 
        $whereClause
        GROUP BY DATE_FORMAT(transaction_date, '%Y-%m'), transaction_type
        ORDER BY month
    ");
    $stmt->execute($params);
    $monthlyTrends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'summary' => [
            'total_income' => $income,
            'total_expenses' => $expenses,
            'net_balance' => $income - $expenses
        ],
        'by_category' => $byCategory,
        'monthly_trends' => $monthlyTrends
    ];
}

function generateEventsSummaryReport($dateFrom, $dateTo) {
    global $pdo;
    
    $whereClause = '';
    $params = [];
    
    if (!empty($dateFrom) || !empty($dateTo)) {
        $conditions = [];
        if (!empty($dateFrom)) {
            $conditions[] = "event_date >= ?";
            $params[] = $dateFrom;
        }
        if (!empty($dateTo)) {
            $conditions[] = "event_date <= ?";
            $params[] = $dateTo;
        }
        $whereClause = "WHERE " . implode(" AND ", $conditions);
    }
    
    // Get events summary
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM events $whereClause");
    $stmt->execute($params);
    $totalEvents = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Get events with registration counts
    $stmt = $pdo->prepare("
        SELECT 
            e.*,
            COUNT(er.id) as registration_count
        FROM events e
        LEFT JOIN event_registrations er ON e.id = er.event_id
        $whereClause
        GROUP BY e.id
        ORDER BY e.event_date DESC
    ");
    $stmt->execute($params);
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get total registrations
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM event_registrations er
        JOIN events e ON er.event_id = e.id
        $whereClause
    ");
    $stmt->execute($params);
    $totalRegistrations = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    return [
        'summary' => [
            'total_events' => $totalEvents,
            'total_registrations' => $totalRegistrations
        ],
        'events' => $events
    ];
}

function generateMembershipFeesReport($dateFrom, $dateTo) {
    global $pdo;
    
    $whereClause = '';
    $params = [];
    
    if (!empty($dateFrom) || !empty($dateTo)) {
        $conditions = [];
        if (!empty($dateFrom)) {
            $conditions[] = "mf.created_at >= ?";
            $params[] = $dateFrom;
        }
        if (!empty($dateTo)) {
            $conditions[] = "mf.created_at <= ?";
            $params[] = $dateTo . ' 23:59:59';
        }
        $whereClause = "WHERE " . implode(" AND ", $conditions);
    }
    
    // Get summary
    $stmt = $pdo->prepare("
        SELECT 
            status,
            COUNT(*) as count,
            SUM(amount) as total
        FROM membership_fees mf
        $whereClause
        GROUP BY status
    ");
    $stmt->execute($params);
    $summary = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get detailed fees
    $stmt = $pdo->prepare("
        SELECT 
            mf.*,
            m.first_name, m.last_name, m.kso_id, m.membership_type
        FROM membership_fees mf
        JOIN members m ON mf.member_id = m.id
        $whereClause
        ORDER BY mf.fee_year DESC, mf.created_at DESC
    ");
    $stmt->execute($params);
    $fees = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'summary' => $summary,
        'fees' => $fees
    ];
}

function generateExpensesSummaryReport($dateFrom, $dateTo) {
    global $pdo;
    
    $whereClause = '';
    $params = [];
    
    if (!empty($dateFrom) || !empty($dateTo)) {
        $conditions = [];
        if (!empty($dateFrom)) {
            $conditions[] = "expense_date >= ?";
            $params[] = $dateFrom;
        }
        if (!empty($dateTo)) {
            $conditions[] = "expense_date <= ?";
            $params[] = $dateTo;
        }
        $whereClause = "WHERE " . implode(" AND ", $conditions);
    }
    
    // Get summary by category
    $stmt = $pdo->prepare("
        SELECT 
            category,
            COUNT(*) as count,
            SUM(amount) as total
        FROM expenses 
        $whereClause
        GROUP BY category
        ORDER BY total DESC
    ");
    $stmt->execute($params);
    $byCategory = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get summary by status
    $stmt = $pdo->prepare("
        SELECT 
            status,
            COUNT(*) as count,
            SUM(amount) as total
        FROM expenses 
        $whereClause
        GROUP BY status
    ");
    $stmt->execute($params);
    $byStatus = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get detailed expenses
    $stmt = $pdo->prepare("SELECT * FROM expenses $whereClause ORDER BY expense_date DESC");
    $stmt->execute($params);
    $expenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'by_category' => $byCategory,
        'by_status' => $byStatus,
        'expenses' => $expenses
    ];
}

function outputCSV($data, $reportType) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $reportType . '_report_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // Output CSV based on report type
    switch ($reportType) {
        case 'members_summary':
            fputcsv($output, ['KSO ID', 'First Name', 'Last Name', 'Email', 'Phone', 'Membership Type', 'Status', 'Created Date']);
            foreach ($data['members'] as $member) {
                fputcsv($output, [
                    $member['kso_id'],
                    $member['first_name'],
                    $member['last_name'],
                    $member['email'],
                    $member['phone'],
                    $member['membership_type'],
                    $member['status'],
                    $member['created_at']
                ]);
            }
            break;
        // Add other report types as needed
    }
    
    fclose($output);
    exit;
}
?>


