// KSO Chandigarh Management System - Admin API Service
class ApiService {
    constructor() {
        this.baseUrl = window.CONFIG ? CONFIG.API_BASE_URL : '../api';
        this.headers = {
            'Content-Type': 'application/json',
        };
    }

    async request(endpoint, options = {}) {
        const url = `${this.baseUrl}/${endpoint}`;
        const config = {
            method: options.method || 'GET',
            headers: { ...this.headers, ...options.headers },
            credentials: 'same-origin', // Include session cookies
            ...options
        };

        if (config.body && typeof config.body === 'object' && !(config.body instanceof FormData)) {
            config.body = JSON.stringify(config.body);
        }

        try {
            const response = await fetch(url, config);
            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            }

            return data;
        } catch (error) {
            console.error('API request failed:', error);
            throw error;
        }
    }

    // Members API
    async getMembers(filters = {}) {
        const queryParams = new URLSearchParams(filters).toString();
        const endpoint = queryParams ? `members/?${queryParams}` : 'members/';
        return this.request(endpoint);
    }

    async getMember(id) {
        return this.request(`members/member?id=${id}`);
    }

    async createMember(memberData) {
        return this.request('members/', {
            method: 'POST',
            body: memberData
        });
    }

    async updateMember(id, memberData) {
        return this.request(`members/member?id=${id}`, {
            method: 'PUT',
            body: memberData
        });
    }

    async deleteMember(id) {
        return this.request(`members/member?id=${id}`, {
            method: 'DELETE'
        });
    }

    async approveMember(memberId, notes = '', upiTxId = '') {
        return this.request('members/approve', {
            method: 'POST',
            body: {
                member_id: memberId,
                notes: notes,
                upi_tx_id: upiTxId
            }
        });
    }

    async rejectMember(memberId, rejectionReason, notes = '') {
        return this.request('members/reject', {
            method: 'POST',
            body: {
                member_id: memberId,
                rejection_reason: rejectionReason,
                notes: notes
            }
        });
    }

    async getMembersStats() {
        return this.request('members/stats');
    }

    // Dashboard API
    async getDashboardStats() {
        return this.request('dashboard/stats');
    }

    // Events API (placeholder for future implementation)
    async getEvents(filters = {}) {
        const queryParams = new URLSearchParams(filters).toString();
        const endpoint = queryParams ? `events/?${queryParams}` : 'events/';
        return this.request(endpoint);
    }

    async createEvent(eventData) {
        return this.request('events/', {
            method: 'POST',
            body: eventData
        });
    }

    // Finance API (placeholder for future implementation)
    async getFinanceRecords(filters = {}) {
        const queryParams = new URLSearchParams(filters).toString();
        const endpoint = queryParams ? `finance/?${queryParams}` : 'finance/';
        return this.request(endpoint);
    }

    async createFinanceRecord(financeData) {
        return this.request('finance/', {
            method: 'POST',
            body: financeData
        });
    }

    // Authentication API (placeholder for future implementation)
    async login(credentials) {
        return this.request('auth/login', {
            method: 'POST',
            body: credentials
        });
    }

    async logout() {
        return this.request('auth/logout', {
            method: 'POST'
        });
    }

    // Dashboard API
    async getDashboardStats() {
        return this.request('dashboard');
    }

    // Enhanced methods for better functionality
    async getRecentMembers(limit = 8) {
        return this.request(`members/?limit=${limit}&sort=created_at&order=desc`);
    }

    async getRecentActivities(limit = 10) {
        return this.request(`dashboard?action=activities&limit=${limit}`);
    }

    async getMonthlyStats() {
        return this.request(`dashboard?action=monthly`);
    }

    // Notifications API
    async getNotifications(filters = {}) {
        const queryParams = new URLSearchParams(filters).toString();
        const endpoint = queryParams ? `notifications/?${queryParams}` : 'notifications/';
        return this.request(endpoint);
    }

    async getNotification(id) {
        return this.request(`notifications/?id=${id}`);
    }

    async createNotification(notificationData) {
        return this.request('notifications/', {
            method: 'POST',
            body: notificationData
        });
    }

    async updateNotification(id, notificationData) {
        return this.request(`notifications/?id=${id}`, {
            method: 'PUT',
            body: notificationData
        });
    }

    async deleteNotification(id) {
        return this.request(`notifications/?id=${id}`, {
            method: 'DELETE'
        });
    }

    async markNotificationAsRead(id) {
        return this.request(`notifications/?id=${id}&action=mark_read`, {
            method: 'PUT'
        });
    }

    async getNotificationStats() {
        return this.request('notifications/stats');
    }

    async getNotificationRecipients() {
        return this.request('notifications/recipients');
    }
}

// Utility functions
class Utils {
    static showAlert(message, type = 'info', containerId = 'alertContainer') {
        const container = document.getElementById(containerId);
        if (!container) {
            console.warn(`Alert container '${containerId}' not found`);
            return;
        }

        const alertId = 'alert-' + Date.now();
        const alertClass = this.getBootstrapAlertClass(type);
        
        const alertHtml = `
            <div id="${alertId}" class="alert ${alertClass} alert-dismissible fade show" role="alert">
                <i class="${this.getAlertIcon(type)} me-2"></i>
                ${this.escapeHtml(message)}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `;
        
        container.innerHTML = alertHtml;
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            const alertElement = document.getElementById(alertId);
            if (alertElement) {
                const bsAlert = new bootstrap.Alert(alertElement);
                bsAlert.close();
            }
        }, 5000);
    }

    static getBootstrapAlertClass(type) {
        const typeMap = {
            'success': 'alert-success',
            'error': 'alert-danger',
            'warning': 'alert-warning',
            'info': 'alert-info'
        };
        return typeMap[type] || 'alert-info';
    }

    static getAlertIcon(type) {
        const iconMap = {
            'success': 'bi bi-check-circle-fill',
            'error': 'bi bi-exclamation-triangle-fill',
            'warning': 'bi bi-exclamation-triangle-fill',
            'info': 'bi bi-info-circle-fill'
        };
        return iconMap[type] || 'bi bi-info-circle-fill';
    }

    static escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    static formatDate(dateString, format = 'short') {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        
        if (isNaN(date.getTime())) return 'Invalid Date';
        
        if (format === 'short') {
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'short',
                day: 'numeric'
            });
        } else if (format === 'long') {
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'long',
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
        } else if (format === 'time') {
            return date.toLocaleTimeString('en-US', {
                hour: '2-digit',
                minute: '2-digit'
            });
        }
        
        return date.toLocaleDateString();
    }

    static getStatusBadgeClass(status) {
        const statusClasses = {
            'pending': 'text-bg-warning',
            'approved': 'text-bg-success',
            'rejected': 'text-bg-danger',
            'under_review': 'text-bg-info',
            'active': 'text-bg-success',
            'inactive': 'text-bg-secondary',
            'suspended': 'text-bg-warning',
            'expired': 'text-bg-dark'
        };
        return statusClasses[status] || 'text-bg-secondary';
    }

    static showLoading(containerId) {
        const container = document.getElementById(containerId);
        if (container) {
            container.innerHTML = `
                <div class="d-flex justify-content-center align-items-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <span class="ms-2 text-muted">Loading...</span>
                </div>
            `;
        }
    }

    static showError(containerId, message) {
        const container = document.getElementById(containerId);
        if (container) {
            container.innerHTML = `
                <div class="alert alert-danger" role="alert">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i>
                    ${this.escapeHtml(message)}
                </div>
            `;
        }
    }

    static validateForm(formElement, rules = {}) {
        const errors = [];
        const formData = new FormData(formElement);
        
        for (const [field, rule] of Object.entries(rules)) {
            const value = formData.get(field);
            
            if (rule.required && (!value || value.trim() === '')) {
                errors.push(`${rule.label || field} is required`);
                continue;
            }
            
            if (value && rule.email && !this.isValidEmail(value)) {
                errors.push(`${rule.label || field} must be a valid email`);
            }
            
            if (value && rule.phone && !this.isValidPhone(value)) {
                errors.push(`${rule.label || field} must be a valid phone number`);
            }
            
            if (value && rule.minLength && value.length < rule.minLength) {
                errors.push(`${rule.label || field} must be at least ${rule.minLength} characters`);
            }
            
            if (value && rule.maxLength && value.length > rule.maxLength) {
                errors.push(`${rule.label || field} must be no more than ${rule.maxLength} characters`);
            }
        }
        
        return errors;
    }

    static isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    static isValidPhone(phone) {
        const phoneRegex = /^[0-9+\-\s()]{10,15}$/;
        return phoneRegex.test(phone);
    }

    static debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    static formatCurrency(amount, currency = 'INR') {
        if (amount === null || amount === undefined) return 'N/A';
        
        if (currency === 'INR') {
            return new Intl.NumberFormat('en-IN', {
                style: 'currency',
                currency: 'INR'
            }).format(amount);
        }
        
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency
        }).format(amount);
    }

    static truncateText(text, maxLength = 50) {
        if (!text) return '';
        if (text.length <= maxLength) return text;
        return text.substring(0, maxLength) + '...';
    }

    static capitalizeFirst(str) {
        if (!str) return '';
        return str.charAt(0).toUpperCase() + str.slice(1);
    }

    static getRelativeTime(dateString) {
        if (!dateString) return 'N/A';
        
        const date = new Date(dateString);
        const now = new Date();
        const diffInSeconds = Math.floor((now - date) / 1000);
        
        if (diffInSeconds < 60) return 'Just now';
        if (diffInSeconds < 3600) return `${Math.floor(diffInSeconds / 60)} minutes ago`;
        if (diffInSeconds < 86400) return `${Math.floor(diffInSeconds / 3600)} hours ago`;
        if (diffInSeconds < 2592000) return `${Math.floor(diffInSeconds / 86400)} days ago`;
        
        return this.formatDate(dateString, 'short');
    }

    static confirmAction(message, title = 'Confirm Action') {
        return new Promise((resolve) => {
            if (confirm(`${title}\n\n${message}`)) {
                resolve(true);
            } else {
                resolve(false);
            }
        });
    }
}

// Global instances
window.apiService = new ApiService();
window.utils = Utils;

// Global utility functions for compatibility
function showError(message) {
    const alertContainer = document.getElementById('alertContainer') || document.body;
    const alert = document.createElement('div');
    alert.className = 'alert alert-danger alert-dismissible fade show';
    alert.innerHTML = `
        <i class="bi bi-exclamation-triangle-fill me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    alertContainer.insertBefore(alert, alertContainer.firstChild);
    
    // Auto-dismiss after 5 seconds
    setTimeout(() => {
        if (alert.parentNode) {
            alert.remove();
        }
    }, 5000);
}

function showSuccess(message) {
    const alertContainer = document.getElementById('alertContainer') || document.body;
    const alert = document.createElement('div');
    alert.className = 'alert alert-success alert-dismissible fade show';
    alert.innerHTML = `
        <i class="bi bi-check-circle-fill me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    alertContainer.insertBefore(alert, alertContainer.firstChild);
    
    // Auto-dismiss after 3 seconds
    setTimeout(() => {
        if (alert.parentNode) {
            alert.remove();
        }
    }, 3000);
}

function showWarning(message) {
    const alertContainer = document.getElementById('alertContainer') || document.body;
    const alert = document.createElement('div');
    alert.className = 'alert alert-warning alert-dismissible fade show';
    alert.innerHTML = `
        <i class="bi bi-exclamation-triangle-fill me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    alertContainer.insertBefore(alert, alertContainer.firstChild);
    
    // Auto-dismiss after 4 seconds
    setTimeout(() => {
        if (alert.parentNode) {
            alert.remove();
        }
    }, 4000);
}

function showInfo(message) {
    const alertContainer = document.getElementById('alertContainer') || document.body;
    const alert = document.createElement('div');
    alert.className = 'alert alert-info alert-dismissible fade show';
    alert.innerHTML = `
        <i class="bi bi-info-circle-fill me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    alertContainer.insertBefore(alert, alertContainer.firstChild);
    
    // Auto-dismiss after 4 seconds
    setTimeout(() => {
        if (alert.parentNode) {
            alert.remove();
        }
    }, 4000);
}

function showLoading() {
    // Create or show loading overlay
    let loadingOverlay = document.getElementById('loadingOverlay');
    if (!loadingOverlay) {
        loadingOverlay = document.createElement('div');
        loadingOverlay.id = 'loadingOverlay';
        loadingOverlay.className = 'position-fixed top-0 start-0 w-100 h-100 d-flex align-items-center justify-content-center';
        loadingOverlay.style.backgroundColor = 'rgba(0,0,0,0.5)';
        loadingOverlay.style.zIndex = '9999';
        loadingOverlay.innerHTML = `
            <div class="spinner-border text-light" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
        `;
        document.body.appendChild(loadingOverlay);
    }
    loadingOverlay.style.display = 'flex';
}

function hideLoading() {
    const loadingOverlay = document.getElementById('loadingOverlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = 'none';
    }
}
