/**
 * Simplified Authentication Service for KSO Chandigarh Management System
 */

class AuthService {
    constructor() {
        this.baseUrl = window.CONFIG ? CONFIG.getApiUrl(CONFIG.ENDPOINTS.AUTH) : '/api/auth';
        this.currentAdmin = null;
        this.isAuthenticated = false;
        
        // Try to restore from localStorage first to avoid redirect loop
        const storedAuth = localStorage.getItem('kso_auth_state');
        if (storedAuth) {
            try {
                const authData = JSON.parse(storedAuth);
                const timestamp = authData.timestamp || 0;
                const now = Date.now();
                const fourHoursInMs = 4 * 60 * 60 * 1000;
                
                // Only use stored auth if it's less than 4 hours old
                if (now - timestamp < fourHoursInMs) {
                    this.currentAdmin = authData.admin;
                    this.isAuthenticated = true;
                }
            } catch (e) {
                // Ignore parse errors
            }
        }
        
        // Initialize auth state after a small delay
        setTimeout(() => this.init(), 100);
    }
    
    async init() {
        // Check if user is already authenticated
        await this.checkSession();
    }
    
    /**
     * Login with username/email and password
     */
    async login(username, password, rememberMe = false) {
        try {
            const response = await fetch(`${this.baseUrl}/login.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                credentials: 'include',
                body: JSON.stringify({
                    username: username,
                    password: password,
                    remember_me: rememberMe
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.currentAdmin = result.data.admin;
                this.isAuthenticated = true;
                
                // Enhanced storage with metadata
                const authData = {
                    admin: this.currentAdmin,
                    timestamp: Date.now(),
                    rememberMe: rememberMe,
                    sessionId: result.data.session_id || Date.now().toString()
                };
                
                localStorage.setItem('kso_auth_state', JSON.stringify(authData));
                
                // Store username separately if remember me is checked
                if (rememberMe) {
                    localStorage.setItem('kso_remembered_username', username);
                } else {
                    localStorage.removeItem('kso_remembered_username');
                }
                
                return result;
            } else {
                throw new Error(result.message || 'Login failed');
            }
        } catch (error) {
            console.error('Login error:', error);
            throw error;
        }
    }
    
    /**
     * Logout current user
     */
    async logout() {
        try {
            const response = await fetch(`${this.baseUrl}/logout.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                }
            });
            
            const result = await response.json();
            
            // Clear auth state regardless of server response
            this.clearAuthState();
            
            return { success: true };
        } catch (error) {
            console.error('Logout error:', error);
            // Clear auth state even if logout request fails
            this.clearAuthState();
            return { success: true };
        }
    }
    
    /**
     * Check current session status
     */
    async checkSession() {
        try {
            const response = await fetch(`${this.baseUrl}/session.php`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                },
                credentials: 'include'
            });
            
            const result = await response.json();
            
            if (result.success && result.data && result.data.logged_in) {
                this.currentAdmin = result.data.admin;
                this.isAuthenticated = true;
                
                // Update localStorage with fresh data
                const storedAuth = localStorage.getItem('kso_auth_state');
                let authData = { timestamp: Date.now() };
                
                if (storedAuth) {
                    try {
                        const existing = JSON.parse(storedAuth);
                        authData = { ...existing, timestamp: Date.now() };
                    } catch (e) {
                        // Use default authData
                    }
                }
                
                authData.admin = this.currentAdmin;
                localStorage.setItem('kso_auth_state', JSON.stringify(authData));
                
                return { success: true, admin: this.currentAdmin };
            } else {
                this.clearAuthState();
                return { success: false };
            }
        } catch (error) {
            console.error('Session check error:', error);
            this.clearAuthState();
            return { success: false };
        }
    }
    
    /**
     * Clear authentication state
     */
    clearAuthState() {
        this.currentAdmin = null;
        this.isAuthenticated = false;
        localStorage.removeItem('kso_auth_state');
        localStorage.removeItem('adminUsername');
        
        // Also clear session cookie by setting expired date
        document.cookie = "KSO_CHANDIGARH=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;";
    }
    
    /**
     * Get current admin information
     */
    getCurrentAdmin() {
        return this.currentAdmin;
    }
    
    /**
     * Check if user is logged in
     */
    isLoggedIn() {
        return this.isAuthenticated;
    }
    
    /**
     * Check if user has specific role
     */
    hasRole(role) {
        return this.currentAdmin && this.currentAdmin.role === role;
    }
    
    /**
     * Redirect to login page
     */
    redirectToLogin() {
        // Prevent redirect loops
        if (document.cookie.includes('KSO_REDIRECT_PREVENTION')) {
            console.log('Preventing redirect loop');
            return;
        }
        
        // Set a cookie to prevent redirect loops
        document.cookie = "KSO_REDIRECT_PREVENTION=true; path=/; max-age=5";
        
        // Always redirect to the main login page in the root directory
        const rootLoginPath = '../../login.html';
        
        // Force redirection to main login page
        window.location.href = rootLoginPath;
    }
    
    /**
     * Redirect to dashboard
     */
    redirectToDashboard() {
        const currentPath = window.location.pathname;
        
        // Determine the correct dashboard path based on current location
        let dashboardPath;
        if (currentPath.includes('/frontend/pages/')) {
            dashboardPath = 'dashboard.html';
        } else if (currentPath.includes('/admin/')) {
            dashboardPath = 'frontend/pages/dashboard.html';
        } else {
            // From root login page
            dashboardPath = 'access/admin/frontend/pages/dashboard.html';
        }
        
        // Smooth redirect with loading indication
        const loadingOverlay = this.createLoadingOverlay('Redirecting to dashboard...');
        document.body.appendChild(loadingOverlay);
        
        setTimeout(() => {
            window.location.href = dashboardPath;
        }, 800);
    }
    
    /**
     * Get remembered username
     */
    getRememberedUsername() {
        return localStorage.getItem('kso_remembered_username');
    }
    
    /**
     * Clear remembered username
     */
    clearRememberedUsername() {
        localStorage.removeItem('kso_remembered_username');
    }
    
    /**
     * Create loading overlay for smooth transitions
     */
    createLoadingOverlay(message = 'Loading...') {
        const overlay = document.createElement('div');
        overlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.9);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 9999;
            font-family: 'Source Sans 3', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
        `;
        
        overlay.innerHTML = `
            <div style="text-align: center;">
                <div class="spinner-border text-primary mb-3" role="status"></div>
                <div style="color: #333; font-size: 16px;">${message}</div>
            </div>
        `;
        
        return overlay;
    }
}

// Create global auth service instance
window.authService = new AuthService();

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
    module.exports = AuthService;
}
