/**
 * KSO Chandigarh Dashboard JavaScript
 * Handles dashboard functionality and data loading with AdminLTE frontend
 */

// Redirect prevention flag - prevents redirect loops
let redirectAttempted = false;

class DashboardController {
    constructor() {
        this.apiService = new ApiService();
        this.refreshInterval = null;
    }

    async init() {
        console.log('Initializing KSO Chandigarh Dashboard...');
        
        // Load initial data
        await this.loadDashboardData();
        
        // Set up auto-refresh (every 5 minutes)
        this.refreshInterval = setInterval(() => {
            this.loadDashboardData();
        }, 300000);
        
        // Set up event listeners
        this.setupEventListeners();
        
        console.log('Dashboard initialized successfully');
    }

    async loadDashboardData() {
        try {
            // Show loading state
            this.showLoading();
            
            // Fetch dashboard statistics
            const response = await this.apiService.getDashboardStats();
            
            if (response.success) {
                this.updateStatCards(response.data);
                this.updateRecentMembers(response.data.recent_members);
                this.updateUpcomingEvents(response.data.upcoming_events);
            } else {
                throw new Error(response.message || 'Failed to load dashboard data');
            }
            
        } catch (error) {
            console.error('Error loading dashboard data:', error);
            this.showError('Failed to load dashboard data. Please refresh the page.');
        } finally {
            this.hideLoading();
        }
    }

    updateStatCards(data) {
        // Update stat cards with animation
        this.animateCounter('#totalMembers', data.total_members || 0);
        this.animateCounter('#totalEvents', data.total_events || 0);
        this.animateCounter('#pendingApprovals', data.pending_approvals || 0);
        
        // Format revenue with currency
        const revenueElement = document.querySelector('#recentRevenue');
        if (revenueElement) {
            revenueElement.textContent = '₹' + (data.recent_revenue || 0).toLocaleString();
        }
    }

    updateRecentMembers(members) {
        const container = document.querySelector('#recentMembersList');
        if (!container) return;

        if (!members || members.length === 0) {
            container.innerHTML = `
                <li class="text-center p-3">
                    <span class="text-muted">No recent members</span>
                </li>
            `;
            return;
        }

        container.innerHTML = members.map(member => `
            <li>
                <img src="../assets/img/user2-160x160.jpg" alt="User Image" class="img-circle">
                <a class="users-list-name" href="member-profile.html?id=${member.id}">
                    ${this.escapeHtml(member.first_name)} ${this.escapeHtml(member.last_name)}
                </a>
                <span class="users-list-date">${member.formatted_date}</span>
            </li>
        `).join('');
    }

    updateUpcomingEvents(events) {
        const container = document.querySelector('#upcomingEventsList');
        if (!container) return;

        if (!events || events.length === 0) {
            container.innerHTML = `
                <li class="list-group-item text-center">
                    <span class="text-muted">No upcoming events</span>
                </li>
            `;
            return;
        }

        container.innerHTML = events.map(event => `
            <li class="list-group-item">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <h6 class="mb-1">
                            <a href="event-details.html?id=${event.id}" class="text-decoration-none">
                                ${this.escapeHtml(event.title)}
                            </a>
                        </h6>
                        <small class="text-muted">
                            <i class="bi bi-calendar-event me-1"></i>
                            ${event.formatted_date}
                        </small>
                    </div>
                    <span class="badge bg-primary">${this.capitalize(event.status)}</span>
                </div>
            </li>
        `).join('');
    }

    animateCounter(selector, targetValue) {
        const element = document.querySelector(selector);
        if (!element) return;

        const startValue = 0;
        const duration = 1000; // 1 second
        const startTime = performance.now();

        const animate = (currentTime) => {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            // Easing function
            const easeOutQuart = 1 - Math.pow(1 - progress, 4);
            const currentValue = Math.floor(startValue + (targetValue - startValue) * easeOutQuart);
            
            element.textContent = currentValue.toLocaleString();
            
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };

        requestAnimationFrame(animate);
    }

    setupEventListeners() {
        // Refresh button
        const refreshBtn = document.getElementById('refreshBtn');
        if (refreshBtn) {
            refreshBtn.addEventListener('click', () => {
                this.loadDashboardData();
            });
        }

        // Handle page visibility change (pause/resume auto-refresh)
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                if (this.refreshInterval) {
                    clearInterval(this.refreshInterval);
                }
            } else {
                this.refreshInterval = setInterval(() => {
                    this.loadDashboardData();
                }, 300000);
                // Refresh immediately when page becomes visible
                this.loadDashboardData();
            }
        });
    }

    showLoading() {
        // Add loading state to stat cards
        const statCards = document.querySelectorAll('.small-box h3');
        statCards.forEach(card => {
            if (!card.textContent.includes('...')) {
                card.setAttribute('data-original', card.textContent);
                card.textContent = '...';
            }
        });
    }

    hideLoading() {
        // Loading state will be updated by data
    }

    showError(message) {
        // Create or update error alert
        let errorAlert = document.querySelector('.dashboard-error-alert');
        if (!errorAlert) {
            errorAlert = document.createElement('div');
            errorAlert.className = 'alert alert-danger alert-dismissible fade show dashboard-error-alert';
            errorAlert.innerHTML = `
                <i class="bi bi-exclamation-triangle"></i>
                <span class="error-message">${message}</span>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            const container = document.querySelector('.app-content .container-fluid');
            if (container) {
                container.insertBefore(errorAlert, container.firstChild);
            }
        } else {
            errorAlert.querySelector('.error-message').textContent = message;
        }
    }

    // Utility functions
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    capitalize(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
    }

    // Public methods
    refresh() {
        this.loadDashboardData();
    }

    destroy() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
        }
    }
}

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Prevent redirect loops
    if (redirectAttempted) {
        console.log("Preventing redirect loop");
        return;
    }
    
    // Add a small delay to ensure authService is fully initialized
    setTimeout(function() {
        // Check authentication first - if not authenticated, this will redirect to login
        try {
            if (typeof requireAuth === 'function') {
                redirectAttempted = true; // Set flag before attempting redirect
                if (!requireAuth()) return; // Stop if not authenticated
                redirectAttempted = false; // Reset flag if auth successful
            } else {
                console.error('requireAuth function not found - redirecting to login');
                window.location.href = '../../login.html';
                return;
            }
        } catch (e) {
            console.error('Authentication check failed:', e);
            window.location.href = '../../login.html';
            return;
        }
        
        // Initialize dashboard controller
        console.log('Starting dashboard initialization...');
        window.dashboardController = new DashboardController();
        window.dashboardController.init();
        
        // Initialize legacy dashboard components for compatibility
        initializeDashboard();
    }, 200);
});

/**
 * Initialize dashboard components
 */
function initializeDashboard() {
    // Load statistics
    loadDashboardStats();
    
    // Load recent activity
    loadRecentActivity();
    
    // Update last update time
    updateLastUpdateTime();
    
    // Setup refresh functionality
    setupRefreshHandler();
    
    // Setup logout handler
    setupLogoutHandler();
    
    // Setup sidebar toggle
    setupSidebarToggle();
}

/**
 * Load dashboard statistics (updated to use new API)
 */
async function loadDashboardStats() {
    try {
        // Use the new dashboard API
        const apiService = new ApiService();
        const response = await apiService.getDashboardStats();
        
        if (response.success) {
            const stats = response.data;
            
            // Update stat displays with real data
            updateStatDisplay('totalMembers', stats.total_members || 0);
            updateStatDisplay('totalEvents', stats.total_events || 0);
            updateStatDisplay('pendingApprovals', stats.pending_approvals || 0);
            updateStatDisplay('monthlyRevenue', stats.monthly_revenue || '₹0');
        }
    } catch (error) {
        console.error('Error loading dashboard stats:', error);
        
        // Fallback to placeholder data
        const stats = {
            totalMembers: 150,
            totalEvents: 25,
            pendingApprovals: 8,
            monthlyRevenue: '₹25,000'
        };
        
        updateStatDisplay('totalMembers', stats.totalMembers);
        updateStatDisplay('totalEvents', stats.totalEvents);
        updateStatDisplay('pendingApprovals', stats.pendingApprovals);
        updateStatDisplay('monthlyRevenue', stats.monthlyRevenue);
    }
}

/**
 * Update individual stat display
 */
function updateStatDisplay(elementId, value) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = value;
    }
}

/**
 * Load recent activity
 */
function loadRecentActivity() {
    const activityContainer = document.getElementById('recentActivity');
    if (!activityContainer) return;
    
    // Simulate recent activity data
    const activities = [
        {
            time: '2 mins ago',
            icon: 'bi-person-plus',
            color: 'success',
            text: 'New member registration: John Doe'
        },
        {
            time: '15 mins ago',
            icon: 'bi-calendar-event',
            color: 'info',
            text: 'Event "Cultural Night" was approved'
        },
        {
            time: '1 hour ago',
            icon: 'bi-credit-card',
            color: 'warning',
            text: 'Payment received from membership fees'
        },
        {
            time: '2 hours ago',
            icon: 'bi-file-earmark-text',
            color: 'primary',
            text: 'Monthly report generated'
        }
    ];
    
    // Generate activity HTML
    let activityHTML = '';
    activities.forEach(activity => {
        activityHTML += `
            <div class="timeline-item">
                <div class="timeline-marker">
                    <i class="${activity.icon} text-${activity.color}"></i>
                </div>
                <div class="timeline-content">
                    <p class="timeline-text">${activity.text}</p>
                    <small class="text-muted">${activity.time}</small>
                </div>
            </div>
        `;
    });
    
    activityContainer.innerHTML = activityHTML;
}

/**
 * Update last update time
 */
function updateLastUpdateTime() {
    const lastUpdateElement = document.getElementById('lastUpdate');
    if (lastUpdateElement) {
        const now = new Date();
        const timeString = now.toLocaleTimeString();
        lastUpdateElement.textContent = timeString;
    }
}

/**
 * Setup refresh handler
 */
function setupRefreshHandler() {
    const refreshBtn = document.getElementById('refreshBtn');
    if (refreshBtn) {
        refreshBtn.addEventListener('click', function() {
            // Add loading state
            const icon = refreshBtn.querySelector('i');
            icon.classList.add('fa-spin');
            
            // Reload data
            setTimeout(() => {
                loadDashboardStats();
                loadRecentActivity();
                updateLastUpdateTime();
                
                // Remove loading state
                icon.classList.remove('fa-spin');
                
                // Show success message
                showNotification('Dashboard refreshed successfully!', 'success');
            }, 1000);
        });
    }
}

/**
 * Setup logout handler
 */
function setupLogoutHandler() {
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', async function(e) {
            e.preventDefault();
            
            try {
                // Use the auth service to handle logout
                if (window.authService) {
                    await window.authService.logout();
                    window.authService.redirectToLogin();
                } else {
                    // Fallback if auth service is not available
                    localStorage.removeItem('kso_auth');
                    sessionStorage.removeItem('kso_auth');
                    window.location.href = '../../login.html'; // Relative path to root login page
                }
            } catch (error) {
                console.error('Logout error:', error);
                // Force redirect even if there's an error
                window.location.href = '../../login.html';
            }
        });
    }
}

/**
 * Setup sidebar toggle
 */
function setupSidebarToggle() {
    const sidebarToggle = document.querySelector('[data-lte-toggle="sidebar"]');
    if (sidebarToggle) {
        sidebarToggle.addEventListener('click', function(e) {
            e.preventDefault();
            document.body.classList.toggle('sidebar-collapse');
        });
    }
}

/**
 * Show notification
 */
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 5000);
}

/**
 * Check authentication status
 */
function checkAuth() {
    const auth = localStorage.getItem('kso_auth') || sessionStorage.getItem('kso_auth');
    if (!auth) {
        window.location.href = 'login.html';
        return false;
    }
    return true;
}

// Check authentication on page load
checkAuth();
