/**
 * KSO Chandigarh - Documents Management
 * Enhanced with DataTables functionality
 */

class DocumentsManager {
    constructor() {
        this.documentsTable = null;
        this.init();
    }

    init() {
        this.initDataTable();
        this.bindEvents();
        this.loadDocuments();
    }

    initDataTable() {
        if ($.fn.DataTable.isDataTable('#documentsTable')) {
            $('#documentsTable').DataTable().destroy();
        }

        this.documentsTable = $('#documentsTable').DataTable({
            responsive: true,
            pageLength: 25,
            lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
            order: [[6, 'desc']], // Sort by date (latest first)
            columnDefs: [
                {
                    targets: [0], // Preview column
                    orderable: false,
                    searchable: false,
                    width: "60px"
                },
                {
                    targets: [7], // Actions column
                    orderable: false,
                    searchable: false,
                    width: "120px"
                },
                {
                    targets: [4], // Size column
                    render: function(data, type, row) {
                        if (type === 'display') {
                            return formatFileSize(data);
                        }
                        return data;
                    }
                },
                {
                    targets: [6], // Date column
                    render: function(data, type, row) {
                        if (type === 'display') {
                            return formatDate(data);
                        }
                        return data;
                    }
                }
            ],
            dom: '<"row"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>' +
                 '<"row"<"col-sm-12"tr>>' +
                 '<"row"<"col-sm-12 col-md-5"i><"col-sm-12 col-md-7"p>>',
            language: {
                search: "Search documents:",
                lengthMenu: "Show _MENU_ documents per page",
                info: "Showing _START_ to _END_ of _TOTAL_ documents",
                infoEmpty: "No documents found",
                infoFiltered: "(filtered from _MAX_ total documents)",
                emptyTable: "No documents available",
                zeroRecords: "No matching documents found"
            }
        });
    }

    bindEvents() {
        // Add document button
        $('#addDocumentBtn').on('click', () => {
            this.showAddDocumentModal();
        });

        // View toggle buttons
        $('#listViewBtn').on('click', () => {
            this.showListView();
        });

        $('#gridViewBtn').on('click', () => {
            this.showGridView();
        });

        // Category filter
        $('#categoryFilter').on('change', () => {
            this.filterByCategory();
        });

        // Type filter
        $('#typeFilter').on('change', () => {
            this.filterByType();
        });
    }

    loadDocuments() {
        // Show loading state
        this.showLoading();

        // API call to load documents
        fetch('/access/admin/api/documents/list.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    this.populateTable(data.documents);
                } else {
                    this.showError('Failed to load documents: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error loading documents:', error);
                this.showError('Failed to load documents. Please try again.');
            })
            .finally(() => {
                this.hideLoading();
            });
    }

    populateTable(documents) {
        const tableData = documents.map(doc => [
            this.createPreviewColumn(doc),
            doc.name,
            doc.category,
            doc.type,
            doc.size,
            doc.uploaded_by,
            doc.uploaded_date,
            this.createActionsColumn(doc)
        ]);

        this.documentsTable.clear().rows.add(tableData).draw();
    }

    createPreviewColumn(doc) {
        const isImage = ['jpg', 'jpeg', 'png', 'gif', 'bmp'].includes(doc.extension.toLowerCase());
        const isPDF = doc.extension.toLowerCase() === 'pdf';
        
        if (isImage) {
            return `<img src="${doc.thumbnail_url || doc.url}" class="img-thumbnail" style="width: 40px; height: 40px; object-fit: cover;" alt="Preview">`;
        } else if (isPDF) {
            return `<i class="bi bi-file-earmark-pdf text-danger" style="font-size: 2rem;"></i>`;
        } else {
            return `<i class="bi bi-file-earmark text-secondary" style="font-size: 2rem;"></i>`;
        }
    }

    createActionsColumn(doc) {
        return `
            <div class="btn-group btn-group-sm" role="group">
                <button type="button" class="btn btn-outline-primary" onclick="documentsManager.viewDocument('${doc.id}')" title="View">
                    <i class="bi bi-eye"></i>
                </button>
                <button type="button" class="btn btn-outline-secondary" onclick="documentsManager.downloadDocument('${doc.id}')" title="Download">
                    <i class="bi bi-download"></i>
                </button>
                <button type="button" class="btn btn-outline-danger" onclick="documentsManager.deleteDocument('${doc.id}')" title="Delete">
                    <i class="bi bi-trash"></i>
                </button>
            </div>
        `;
    }

    showListView() {
        $('#listViewContainer').show();
        $('#gridViewContainer').hide();
        $('#listViewBtn').addClass('active');
        $('#gridViewBtn').removeClass('active');
        
        if (this.documentsTable) {
            this.documentsTable.columns.adjust().draw();
        }
    }

    showGridView() {
        $('#listViewContainer').hide();
        $('#gridViewContainer').show();
        $('#listViewBtn').removeClass('active');
        $('#gridViewBtn').addClass('active');
    }

    filterByCategory() {
        const category = $('#categoryFilter').val();
        if (category === 'all') {
            this.documentsTable.column(2).search('').draw();
        } else {
            this.documentsTable.column(2).search(category).draw();
        }
    }

    filterByType() {
        const type = $('#typeFilter').val();
        if (type === 'all') {
            this.documentsTable.column(3).search('').draw();
        } else {
            this.documentsTable.column(3).search(type).draw();
        }
    }

    viewDocument(id) {
        // Implementation for viewing document
        console.log('Viewing document:', id);
    }

    downloadDocument(id) {
        // Implementation for downloading document
        window.open(`/access/admin/api/documents/download.php?id=${id}`, '_blank');
    }

    deleteDocument(id) {
        if (confirm('Are you sure you want to delete this document?')) {
            // Implementation for deleting document
            console.log('Deleting document:', id);
        }
    }

    showAddDocumentModal() {
        // Implementation for showing add document modal
        $('#addDocumentModal').modal('show');
    }

    showLoading() {
        // Show loading indicator
        if (this.documentsTable) {
            this.documentsTable.processing(true);
        }
    }

    hideLoading() {
        // Hide loading indicator
        if (this.documentsTable) {
            this.documentsTable.processing(false);
        }
    }

    showError(message) {
        // Show error message
        console.error(message);
        // You can implement a toast notification here
    }
}

// Utility functions
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Initialize when document is ready
let documentsManager;
$(document).ready(function() {
    documentsManager = new DocumentsManager();
});
