/**
 * Enhanced DataTable Library with AdminLTE Integration
 * Provides comprehensive data management functionality
 */

class EnhancedDataTable {
    constructor(options) {
        this.tableName = options.tableName;
        this.containerId = options.containerId || 'dataTableContainer';
        this.apiUrl = options.apiUrl || '../api/datatable.php';
        this.columns = options.columns || [];
        this.searchable = options.searchable || [];
        this.filterable = options.filterable || [];
        this.exportable = options.exportable || [];
        this.showActions = options.showActions !== false;
        this.pageLength = options.pageLength || 25;
        this.responsive = options.responsive !== false;
        this.stateSave = options.stateSave !== false;
        this.callbacks = options.callbacks || {};
        
        this.table = null;
        this.filters = {};
        this.isLoading = false;
        
        this.init();
    }
    
    init() {
        this.loadConfiguration().then(() => {
            this.renderTable();
            this.bindEvents();
            this.setupFilters();
            this.setupBulkActions();
        });
    }
    
    async loadConfiguration() {
        try {
            const response = await fetch(`${this.apiUrl}/${this.tableName}/config`);
            const config = await response.json();
            
            if (config.columns) {
                this.columns = config.columns;
                this.searchable = config.searchableColumns || [];
                this.filterable = config.filterableColumns || [];
                this.exportable = config.exportableColumns || [];
            }
        } catch (error) {
            console.error('Failed to load table configuration:', error);
        }
    }
    
    renderTable() {
        const container = document.getElementById(this.containerId);
        
        if (!container) {
            console.error(`Container with ID '${this.containerId}' not found`);
            return;
        }
        
        // Create table structure
        const tableHtml = `
            <div class="card">
                <div class="card-header">
                    <div class="row">
                        <div class="col-md-6">
                            <h3 class="card-title">${this.getTableTitle()}</h3>
                        </div>
                        <div class="col-md-6">
                            <div class="float-right">
                                ${this.renderToolbar()}
                            </div>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    ${this.renderFiltersPanel()}
                    ${this.renderBulkActionsPanel()}
                    <div class="table-responsive">
                        <table id="${this.tableName}Table" class="table table-bordered table-striped table-hover">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="selectAll" class="form-check-input"></th>
                                    ${this.renderColumnHeaders()}
                                    ${this.showActions ? '<th>Actions</th>' : ''}
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Data will be loaded via AJAX -->
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="card-footer">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="dataTables_info" id="${this.tableName}Info"></div>
                        </div>
                        <div class="col-md-6">
                            <div class="dataTables_paginate" id="${this.tableName}Paginate"></div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        container.innerHTML = tableHtml;
        
        this.initializeDataTable();
    }
    
    renderToolbar() {
        return `
            <div class="btn-group">
                <button type="button" class="btn btn-primary" onclick="window.${this.tableName}Table.showCreateModal()">
                    <i class="fas fa-plus"></i> Add New
                </button>
                <button type="button" class="btn btn-info" onclick="window.${this.tableName}Table.refreshTable()">
                    <i class="fas fa-refresh"></i> Refresh
                </button>
                <div class="btn-group">
                    <button type="button" class="btn btn-secondary dropdown-toggle" data-bs-toggle="dropdown">
                        <i class="fas fa-download"></i> Export
                    </button>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="#" onclick="window.${this.tableName}Table.exportData('csv')">CSV</a></li>
                        <li><a class="dropdown-item" href="#" onclick="window.${this.tableName}Table.exportData('excel')">Excel</a></li>
                        <li><a class="dropdown-item" href="#" onclick="window.${this.tableName}Table.exportData('pdf')">PDF</a></li>
                    </ul>
                </div>
                <button type="button" class="btn btn-outline-secondary" onclick="window.${this.tableName}Table.showFilters()">
                    <i class="fas fa-filter"></i> Filters
                </button>
            </div>
        `;
    }
    
    renderFiltersPanel() {
        let filtersHtml = '<div id="filtersPanel" class="mb-3" style="display: none;">';
        filtersHtml += '<div class="row">';
        
        this.filterable.forEach(column => {
            const columnConfig = this.columns[column];
            if (columnConfig) {
                filtersHtml += `
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="filter_${column}">${columnConfig.title}</label>
                            ${this.renderFilterInput(column, columnConfig)}
                        </div>
                    </div>
                `;
            }
        });
        
        filtersHtml += `
                <div class="col-md-12">
                    <div class="form-group">
                        <button type="button" class="btn btn-primary" onclick="window.${this.tableName}Table.applyFilters()">
                            Apply Filters
                        </button>
                        <button type="button" class="btn btn-secondary" onclick="window.${this.tableName}Table.clearFilters()">
                            Clear Filters
                        </button>
                    </div>
                </div>
            </div>
        </div>`;
        
        return filtersHtml;
    }
    
    renderFilterInput(column, config) {
        switch (config.type) {
            case 'select':
                return `<select class="form-control" id="filter_${column}">
                    <option value="">All</option>
                    ${config.options ? config.options.map(opt => `<option value="${opt}">${opt}</option>`).join('') : ''}
                </select>`;
            case 'date':
                return `<input type="date" class="form-control" id="filter_${column}">`;
            case 'number':
                return `<input type="number" class="form-control" id="filter_${column}" placeholder="Filter by ${config.title}">`;
            default:
                return `<input type="text" class="form-control" id="filter_${column}" placeholder="Filter by ${config.title}">`;
        }
    }
    
    renderBulkActionsPanel() {
        return `
            <div id="bulkActionsPanel" class="mb-3" style="display: none;">
                <div class="btn-group">
                    <span class="selected-count me-3">0 selected</span>
                    <button type="button" class="btn btn-sm btn-danger" onclick="window.${this.tableName}Table.bulkDelete()">
                        <i class="fas fa-trash"></i> Delete Selected
                    </button>
                    <button type="button" class="btn btn-sm btn-success" onclick="window.${this.tableName}Table.bulkActivate()">
                        <i class="fas fa-check"></i> Activate Selected
                    </button>
                    <button type="button" class="btn btn-sm btn-warning" onclick="window.${this.tableName}Table.bulkDeactivate()">
                        <i class="fas fa-ban"></i> Deactivate Selected
                    </button>
                </div>
            </div>
        `;
    }
    
    renderColumnHeaders() {
        return Object.entries(this.columns).map(([key, config]) => {
            return `<th data-column="${key}">${config.title}</th>`;
        }).join('');
    }
    
    initializeDataTable() {
        const tableId = `${this.tableName}Table`;
        
        this.table = $(`#${tableId}`).DataTable({
            processing: true,
            serverSide: true,
            ajax: {
                url: `${this.apiUrl}/${this.tableName}/data`,
                type: 'GET',
                data: (params) => {
                    return {
                        ...params,
                        filters: JSON.stringify(this.filters)
                    };
                },
                error: (xhr, error, code) => {
                    console.error('DataTable Ajax Error:', error);
                    this.showError('Failed to load data');
                }
            },
            columns: this.buildColumnsConfig(),
            pageLength: this.pageLength,
            responsive: this.responsive,
            stateSave: this.stateSave,
            dom: '<"top"l>rt<"bottom"ip><"clear">',
            language: {
                processing: '<i class="fas fa-spinner fa-spin"></i> Loading...',
                emptyTable: 'No data available',
                info: 'Showing _START_ to _END_ of _TOTAL_ entries',
                infoEmpty: 'No entries to show',
                infoFiltered: '(filtered from _MAX_ total entries)',
                lengthMenu: 'Show _MENU_ entries',
                search: 'Search:',
                zeroRecords: 'No matching records found'
            },
            drawCallback: () => {
                this.updateBulkActionsPanel();
            }
        });
        
        // Store reference globally for callbacks
        window[`${this.tableName}Table`] = this;
    }
    
    buildColumnsConfig() {
        const columns = [
            {
                data: null,
                orderable: false,
                searchable: false,
                render: (data, type, row) => {
                    return `<input type="checkbox" class="form-check-input row-select" value="${row.id}">`;
                }
            }
        ];
        
        Object.entries(this.columns).forEach(([key, config]) => {
            columns.push({
                data: key,
                title: config.title,
                searchable: this.searchable.includes(key),
                render: (data, type, row) => {
                    if (type === 'display') {
                        return this.formatCellData(data, config.type, row);
                    }
                    return data;
                }
            });
        });
        
        if (this.showActions) {
            columns.push({
                data: 'actions',
                orderable: false,
                searchable: false,
                render: (data, type, row) => {
                    return this.renderActionButtons(row);
                }
            });
        }
        
        return columns;
    }
    
    formatCellData(data, type, row) {
        if (data === null || data === undefined) {
            return '';
        }
        
        switch (type) {
            case 'currency':
                return `₹${parseFloat(data).toLocaleString('en-IN', { minimumFractionDigits: 2 })}`;
            case 'date':
                return data ? new Date(data).toLocaleDateString() : '';
            case 'datetime':
                return data ? new Date(data).toLocaleString() : '';
            case 'boolean':
                return data ? '<span class="badge bg-success">Yes</span>' : '<span class="badge bg-danger">No</span>';
            case 'email':
                return `<a href="mailto:${data}">${data}</a>`;
            case 'phone':
                return `<a href="tel:${data}">${data}</a>`;
            case 'filesize':
                return this.formatFileSize(data);
            case 'select':
                return `<span class="badge bg-primary">${data}</span>`;
            default:
                return data;
        }
    }
    
    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
    renderActionButtons(row) {
        const buttons = [];
        
        // View button
        buttons.push(`<button class="btn btn-sm btn-info" onclick="window.${this.tableName}Table.viewRecord(${row.id})" title="View">
            <i class="fas fa-eye"></i>
        </button>`);
        
        // Edit button
        buttons.push(`<button class="btn btn-sm btn-warning" onclick="window.${this.tableName}Table.editRecord(${row.id})" title="Edit">
            <i class="fas fa-edit"></i>
        </button>`);
        
        // Delete button
        buttons.push(`<button class="btn btn-sm btn-danger" onclick="window.${this.tableName}Table.deleteRecord(${row.id})" title="Delete">
            <i class="fas fa-trash"></i>
        </button>`);
        
        // Table-specific buttons
        if (this.callbacks.customButtons) {
            buttons.push(...this.callbacks.customButtons(row));
        }
        
        return `<div class="btn-group">${buttons.join('')}</div>`;
    }
    
    bindEvents() {
        // Global search
        const globalSearch = document.getElementById('globalSearch');
        if (globalSearch) {
            globalSearch.addEventListener('input', (e) => {
                this.table.search(e.target.value).draw();
            });
        }
        
        // Select all checkbox
        const selectAllCheckbox = document.getElementById('selectAll');
        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', (e) => {
                const checkboxes = document.querySelectorAll('.row-select');
                checkboxes.forEach(checkbox => {
                    checkbox.checked = e.target.checked;
                });
                this.updateBulkActionsPanel();
            });
        }
        
        // Individual row selection
        document.addEventListener('change', (e) => {
            if (e.target.classList.contains('row-select')) {
                this.updateBulkActionsPanel();
            }
        });
    }
    
    setupFilters() {
        this.filterable.forEach(column => {
            const filterElement = document.getElementById(`filter_${column}`);
            if (filterElement) {
                filterElement.addEventListener('change', () => {
                    this.applyFilters();
                });
            }
        });
    }
    
    setupBulkActions() {
        // Bulk actions are handled by individual methods
    }
    
    // Public methods
    refreshTable() {
        this.table.ajax.reload();
    }
    
    showFilters() {
        const filtersPanel = document.getElementById('filtersPanel');
        if (filtersPanel) {
            filtersPanel.style.display = filtersPanel.style.display === 'none' ? 'block' : 'none';
        }
    }
    
    applyFilters() {
        this.filters = {};
        
        this.filterable.forEach(column => {
            const filterElement = document.getElementById(`filter_${column}`);
            if (filterElement && filterElement.value) {
                this.filters[column] = filterElement.value;
            }
        });
        
        this.table.ajax.reload();
    }
    
    clearFilters() {
        this.filters = {};
        
        this.filterable.forEach(column => {
            const filterElement = document.getElementById(`filter_${column}`);
            if (filterElement) {
                filterElement.value = '';
            }
        });
        
        this.table.ajax.reload();
    }
    
    exportData(format) {
        const params = new URLSearchParams({
            format: format,
            filters: JSON.stringify(this.filters)
        });
        
        window.open(`${this.apiUrl}/${this.tableName}/export?${params.toString()}`, '_blank');
    }
    
    getSelectedIds() {
        const selected = [];
        document.querySelectorAll('.row-select:checked').forEach(checkbox => {
            selected.push(parseInt(checkbox.value));
        });
        return selected;
    }
    
    updateBulkActionsPanel() {
        const selected = this.getSelectedIds();
        const bulkActionsPanel = document.getElementById('bulkActionsPanel');
        const selectedCount = bulkActionsPanel.querySelector('.selected-count');
        
        if (selected.length > 0) {
            bulkActionsPanel.style.display = 'block';
            selectedCount.textContent = `${selected.length} selected`;
        } else {
            bulkActionsPanel.style.display = 'none';
        }
    }
    
    // CRUD operations
    showCreateModal() {
        if (this.callbacks.onCreate) {
            this.callbacks.onCreate();
        } else {
            this.showError('Create functionality not implemented');
        }
    }
    
    viewRecord(id) {
        if (this.callbacks.onView) {
            this.callbacks.onView(id);
        } else {
            this.showError('View functionality not implemented');
        }
    }
    
    editRecord(id) {
        if (this.callbacks.onEdit) {
            this.callbacks.onEdit(id);
        } else {
            this.showError('Edit functionality not implemented');
        }
    }
    
    async deleteRecord(id) {
        if (!confirm('Are you sure you want to delete this record?')) {
            return;
        }
        
        try {
            const response = await fetch(`${this.apiUrl}/${this.tableName}/${id}`, {
                method: 'DELETE'
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showSuccess('Record deleted successfully');
                this.refreshTable();
            } else {
                this.showError(result.message || 'Failed to delete record');
            }
        } catch (error) {
            this.showError('Error deleting record: ' + error.message);
        }
    }
    
    // Bulk operations
    async bulkDelete() {
        const selected = this.getSelectedIds();
        
        if (selected.length === 0) {
            this.showError('Please select records to delete');
            return;
        }
        
        if (!confirm(`Are you sure you want to delete ${selected.length} records?`)) {
            return;
        }
        
        await this.performBulkAction('delete', selected);
    }
    
    async bulkActivate() {
        const selected = this.getSelectedIds();
        
        if (selected.length === 0) {
            this.showError('Please select records to activate');
            return;
        }
        
        await this.performBulkAction('activate', selected);
    }
    
    async bulkDeactivate() {
        const selected = this.getSelectedIds();
        
        if (selected.length === 0) {
            this.showError('Please select records to deactivate');
            return;
        }
        
        await this.performBulkAction('deactivate', selected);
    }
    
    async performBulkAction(action, ids) {
        try {
            const response = await fetch(`${this.apiUrl}/${this.tableName}/bulk-action`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ action, ids })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showSuccess(result.message);
                this.refreshTable();
                this.updateBulkActionsPanel();
            } else {
                this.showError(result.message || 'Bulk action failed');
            }
        } catch (error) {
            this.showError('Error performing bulk action: ' + error.message);
        }
    }
    
    // Utility methods
    getTableTitle() {
        const titles = {
            'members': 'Members Management',
            'events': 'Events Management',
            'finance': 'Finance Records',
            'expenses': 'Expenses Management',
            'admins': 'Administrators',
            'documents': 'Documents Management',
            'notifications': 'Notifications'
        };
        
        return titles[this.tableName] || this.tableName.charAt(0).toUpperCase() + this.tableName.slice(1);
    }
    
    showSuccess(message) {
        // Use AdminLTE toast or SweetAlert
        if (typeof Swal !== 'undefined') {
            Swal.fire('Success', message, 'success');
        } else if (typeof toastr !== 'undefined') {
            toastr.success(message);
        } else {
            alert(message);
        }
    }
    
    showError(message) {
        // Use AdminLTE toast or SweetAlert
        if (typeof Swal !== 'undefined') {
            Swal.fire('Error', message, 'error');
        } else if (typeof toastr !== 'undefined') {
            toastr.error(message);
        } else {
            alert(message);
        }
    }
}

// Export for use in other scripts
window.EnhancedDataTable = EnhancedDataTable;
