/**
 * KSO Chandigarh - Event Registrations Management
 * Enhanced with DataTables functionality
 */

class EventRegistrationsManager {
    constructor() {
        this.registrationsTable = null;
        this.currentEventId = null;
        this.init();
    }

    init() {
        // Get event ID from URL if present
        const urlParams = new URLSearchParams(window.location.search);
        this.currentEventId = urlParams.get('event_id');
        
        this.initDataTable();
        this.bindEvents();
        this.loadRegistrations();
    }

    initDataTable() {
        if ($.fn.DataTable.isDataTable('#registrationsTable')) {
            $('#registrationsTable').DataTable().destroy();
        }

        this.registrationsTable = $('#registrationsTable').DataTable({
            responsive: true,
            pageLength: 25,
            lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
            order: [[7, 'desc']], // Sort by registration date (latest first)
            columnDefs: [
                {
                    targets: [0], // Photo column
                    orderable: false,
                    searchable: false,
                    width: "50px"
                },
                {
                    targets: [8], // Actions column
                    orderable: false,
                    searchable: false,
                    width: "120px"
                },
                {
                    targets: [5], // Status column
                    render: function(data, type, row) {
                        if (type === 'display') {
                            return getRegistrationStatusBadge(data);
                        }
                        return data;
                    }
                },
                {
                    targets: [6], // Payment status column
                    render: function(data, type, row) {
                        if (type === 'display') {
                            return getPaymentStatusBadge(data);
                        }
                        return data;
                    }
                },
                {
                    targets: [7], // Registration date column
                    render: function(data, type, row) {
                        if (type === 'display' && data) {
                            return formatDateTime(data);
                        }
                        return data;
                    }
                }
            ],
            dom: '<"row"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>' +
                 '<"row"<"col-sm-12"tr>>' +
                 '<"row"<"col-sm-12 col-md-5"i><"col-sm-12 col-md-7"p>>',
            language: {
                search: "Search registrations:",
                lengthMenu: "Show _MENU_ registrations per page",
                info: "Showing _START_ to _END_ of _TOTAL_ registrations",
                infoEmpty: "No registrations found",
                infoFiltered: "(filtered from _MAX_ total registrations)",
                emptyTable: "No registrations available",
                zeroRecords: "No matching registrations found"
            }
        });
    }

    bindEvents() {
        // Event filter
        $('#eventFilter').on('change', () => {
            this.filterByEvent();
        });

        // Status filter
        $('#statusFilter').on('change', () => {
            this.filterByStatus();
        });

        // Payment status filter
        $('#paymentStatusFilter').on('change', () => {
            this.filterByPaymentStatus();
        });

        // Export buttons
        $('#exportRegistrationsBtn').on('click', () => {
            this.exportRegistrations();
        });

        $('#exportAttendeesBtn').on('click', () => {
            this.exportAttendeesList();
        });

        // Bulk actions
        $('#selectAllRegistrations').on('change', () => {
            this.toggleSelectAll();
        });

        $('#bulkActionBtn').on('click', () => {
            this.handleBulkAction();
        });

        // Send notifications
        $('#sendNotificationBtn').on('click', () => {
            this.showSendNotificationModal();
        });
    }

    loadRegistrations() {
        this.showLoading();

        const url = this.currentEventId 
            ? `/access/admin/api/registrations/list.php?event_id=${this.currentEventId}`
            : '/access/admin/api/registrations/list.php';

        fetch(url)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    this.populateTable(data.registrations);
                    this.updateStats(data.stats);
                    if (data.event_info) {
                        this.updateEventInfo(data.event_info);
                    }
                } else {
                    this.showError('Failed to load registrations: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error loading registrations:', error);
                this.showError('Failed to load registrations. Please try again.');
            })
            .finally(() => {
                this.hideLoading();
            });
    }

    populateTable(registrations) {
        const tableData = registrations.map(registration => [
            this.createPhotoColumn(registration.member),
            registration.member.name,
            registration.member.email,
            registration.member.phone,
            registration.event_title,
            registration.status,
            registration.payment_status,
            registration.registration_date,
            this.createActionsColumn(registration)
        ]);

        this.registrationsTable.clear().rows.add(tableData).draw();
    }

    createPhotoColumn(member) {
        if (member.photo_url) {
            return `<img src="${member.photo_url}" class="img-thumbnail rounded-circle" style="width: 35px; height: 35px; object-fit: cover;" alt="${member.name}">`;
        } else {
            return `<div class="bg-secondary rounded-circle d-flex align-items-center justify-content-center" style="width: 35px; height: 35px;">
                        <i class="bi bi-person text-white" style="font-size: 12px;"></i>
                    </div>`;
        }
    }

    createActionsColumn(registration) {
        return `
            <div class="btn-group btn-group-sm" role="group">
                <button type="button" class="btn btn-outline-primary" onclick="registrationsManager.viewRegistration('${registration.id}')" title="View">
                    <i class="bi bi-eye"></i>
                </button>
                <button type="button" class="btn btn-outline-success" onclick="registrationsManager.approveRegistration('${registration.id}')" title="Approve">
                    <i class="bi bi-check-circle"></i>
                </button>
                <button type="button" class="btn btn-outline-warning" onclick="registrationsManager.rejectRegistration('${registration.id}')" title="Reject">
                    <i class="bi bi-x-circle"></i>
                </button>
                <button type="button" class="btn btn-outline-danger" onclick="registrationsManager.deleteRegistration('${registration.id}')" title="Delete">
                    <i class="bi bi-trash"></i>
                </button>
            </div>
        `;
    }

    filterByEvent() {
        const eventId = $('#eventFilter').val();
        if (eventId === 'all') {
            this.registrationsTable.column(4).search('').draw();
        } else {
            // Reload with specific event filter
            this.currentEventId = eventId;
            this.loadRegistrations();
        }
    }

    filterByStatus() {
        const status = $('#statusFilter').val();
        if (status === 'all') {
            this.registrationsTable.column(5).search('').draw();
        } else {
            this.registrationsTable.column(5).search(status).draw();
        }
    }

    filterByPaymentStatus() {
        const paymentStatus = $('#paymentStatusFilter').val();
        if (paymentStatus === 'all') {
            this.registrationsTable.column(6).search('').draw();
        } else {
            this.registrationsTable.column(6).search(paymentStatus).draw();
        }
    }

    updateStats(stats) {
        $('#totalRegistrationsCount').text(stats.total_registrations || 0);
        $('#confirmedRegistrationsCount').text(stats.confirmed_registrations || 0);
        $('#pendingRegistrationsCount').text(stats.pending_registrations || 0);
        $('#rejectedRegistrationsCount').text(stats.rejected_registrations || 0);
    }

    updateEventInfo(eventInfo) {
        $('#eventTitle').text(eventInfo.title);
        $('#eventDate').text(formatDateTime(eventInfo.event_date));
        $('#registrationDeadline').text(formatDateTime(eventInfo.registration_deadline));
        $('#maxParticipants').text(eventInfo.max_participants || 'Unlimited');
    }

    viewRegistration(id) {
        // Implementation for viewing registration details
        console.log('Viewing registration:', id);
    }

    approveRegistration(id) {
        if (confirm('Are you sure you want to approve this registration?')) {
            this.updateRegistrationStatus(id, 'confirmed');
        }
    }

    rejectRegistration(id) {
        if (confirm('Are you sure you want to reject this registration?')) {
            this.updateRegistrationStatus(id, 'rejected');
        }
    }

    updateRegistrationStatus(id, status) {
        fetch('/access/admin/api/registrations/update-status.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ id: id, status: status })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                this.loadRegistrations(); // Reload the table
                this.showSuccess(`Registration ${status} successfully`);
            } else {
                this.showError('Failed to update registration: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error updating registration:', error);
            this.showError('Failed to update registration. Please try again.');
        });
    }

    deleteRegistration(id) {
        if (confirm('Are you sure you want to delete this registration? This action cannot be undone.')) {
            fetch('/access/admin/api/registrations/delete.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ id: id })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    this.loadRegistrations(); // Reload the table
                    this.showSuccess('Registration deleted successfully');
                } else {
                    this.showError('Failed to delete registration: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error deleting registration:', error);
                this.showError('Failed to delete registration. Please try again.');
            });
        }
    }

    exportRegistrations() {
        const params = new URLSearchParams({
            event_id: this.currentEventId || '',
            status: $('#statusFilter').val(),
            payment_status: $('#paymentStatusFilter').val()
        });
        
        window.open(`/access/admin/api/registrations/export.php?${params.toString()}`, '_blank');
    }

    exportAttendeesList() {
        const params = new URLSearchParams({
            event_id: this.currentEventId || '',
            status: 'confirmed'
        });
        
        window.open(`/access/admin/api/registrations/attendees-list.php?${params.toString()}`, '_blank');
    }

    toggleSelectAll() {
        const isChecked = $('#selectAllRegistrations').is(':checked');
        $('.registration-checkbox').prop('checked', isChecked);
    }

    handleBulkAction() {
        const selectedRegistrations = $('.registration-checkbox:checked').map(function() {
            return $(this).val();
        }).get();

        if (selectedRegistrations.length === 0) {
            alert('Please select at least one registration.');
            return;
        }

        const action = $('#bulkActionSelect').val();
        if (!action) {
            alert('Please select an action.');
            return;
        }

        // Implementation for bulk actions
        console.log('Bulk action:', action, 'Registrations:', selectedRegistrations);
    }

    showSendNotificationModal() {
        $('#sendNotificationModal').modal('show');
    }

    showLoading() {
        if (this.registrationsTable) {
            this.registrationsTable.processing(true);
        }
    }

    hideLoading() {
        if (this.registrationsTable) {
            this.registrationsTable.processing(false);
        }
    }

    showError(message) {
        console.error(message);
        // Implement toast notification
    }

    showSuccess(message) {
        console.log(message);
        // Implement toast notification
    }
}

// Utility functions
function getRegistrationStatusBadge(status) {
    const badges = {
        'pending': '<span class="badge bg-warning">Pending</span>',
        'confirmed': '<span class="badge bg-success">Confirmed</span>',
        'rejected': '<span class="badge bg-danger">Rejected</span>',
        'cancelled': '<span class="badge bg-secondary">Cancelled</span>'
    };
    return badges[status] || `<span class="badge bg-light text-dark">${status}</span>`;
}

function getPaymentStatusBadge(status) {
    const badges = {
        'pending': '<span class="badge bg-warning">Pending</span>',
        'paid': '<span class="badge bg-success">Paid</span>',
        'failed': '<span class="badge bg-danger">Failed</span>',
        'refunded': '<span class="badge bg-info">Refunded</span>'
    };
    return badges[status] || `<span class="badge bg-light text-dark">${status}</span>`;
}

function formatDateTime(dateTimeString) {
    if (!dateTimeString) return '-';
    const date = new Date(dateTimeString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Initialize when document is ready
let registrationsManager;
$(document).ready(function() {
    registrationsManager = new EventRegistrationsManager();
});
