/**
 * Events Management JavaScript
 * Handles all events-related functionality including CRUD operations, filtering, and UI interactions
 */

// Initialize services
const apiService = new ApiService();
const authService = new AuthService();

// Global variables
let currentPage = 1;
let totalPages = 1;
let currentFilters = {};
let events = [];

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    requireAuth();
    initializeEventListeners();
    loadEvents();
    loadEventStatistics();
});

/**
 * Initialize all event listeners
 */
function initializeEventListeners() {
    // Logout button
    document.getElementById('logoutBtn')?.addEventListener('click', async function(e) {
        e.preventDefault();
        await authService.logout();
    });
    
    // Search and filter controls
    document.getElementById('searchBtn')?.addEventListener('click', searchEvents);
    document.getElementById('searchEvents')?.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            searchEvents();
        }
    });
    
    // Filter dropdowns
    document.getElementById('filterEventType')?.addEventListener('change', filterEvents);
    document.getElementById('filterStatus')?.addEventListener('change', filterEvents);
    document.getElementById('filterDateFrom')?.addEventListener('change', filterEvents);
    document.getElementById('filterDateTo')?.addEventListener('change', filterEvents);
    
    // Action buttons
    document.getElementById('clearFilters')?.addEventListener('click', clearFilters);
    document.getElementById('refreshEvents')?.addEventListener('click', function() {
        currentPage = 1;
        loadEvents();
        loadEventStatistics();
    });
    
    // Event form submission
    document.getElementById('eventForm')?.addEventListener('submit', handleEventSubmission);
    
    // Modal events
    const eventModal = document.getElementById('eventModal');
    if (eventModal) {
        eventModal.addEventListener('show.bs.modal', function(event) {
            const button = event.relatedTarget;
            const eventId = button?.getAttribute('data-event-id');
            
            if (eventId) {
                loadEventForEdit(eventId);
            } else {
                resetEventForm();
            }
        });
        
        eventModal.addEventListener('hidden.bs.modal', function() {
            resetEventForm();
        });
    }
}

/**
 * Load events with current filters and pagination
 */
async function loadEvents(page = 1) {
    currentPage = page;
    const container = document.getElementById('eventsContainer');
    
    try {
        // Show loading state
        container.innerHTML = `
            <div class="col-12 text-center py-5">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-3 text-muted">Loading events...</p>
            </div>
        `;
        
        // Build query parameters
        const params = new URLSearchParams({
            limit: 12,
            offset: (page - 1) * 12,
            ...currentFilters
        });
        
        const response = await apiService.get(`/events/?${params.toString()}`);
        
        if (response.success) {
            events = response.data.events || [];
            const total = response.data.total || 0;
            totalPages = Math.ceil(total / 12);
            
            renderEvents(events);
            renderPagination();
        } else {
            throw new Error(response.message || 'Failed to load events');
        }
        
    } catch (error) {
        console.error('Error loading events:', error);
        container.innerHTML = `
            <div class="col-12 text-center py-5">
                <i class="bi bi-exclamation-triangle text-warning" style="font-size: 3rem;"></i>
                <h5 class="mt-3">Error Loading Events</h5>
                <p class="text-muted">${error.message}</p>
                <button class="btn btn-primary" onclick="loadEvents()">
                    <i class="bi bi-arrow-clockwise"></i> Retry
                </button>
            </div>
        `;
        showNotification('Error loading events: ' + error.message, 'error');
    }
}

/**
 * Render events in grid layout
 */
function renderEvents(events) {
    const container = document.getElementById('eventsContainer');
    
    if (!events || events.length === 0) {
        container.innerHTML = `
            <div class="col-12 text-center py-5">
                <i class="bi bi-calendar-x text-muted" style="font-size: 4rem;"></i>
                <h4 class="mt-3">No Events Found</h4>
                <p class="text-muted">No events match your current filters. Try adjusting your search criteria.</p>
                <button class="btn btn-outline-primary" onclick="clearFilters()">
                    <i class="bi bi-arrow-clockwise"></i> Clear Filters
                </button>
            </div>
        `;
        return;
    }
    
    const eventsHTML = events.map(event => createEventCard(event)).join('');
    container.innerHTML = eventsHTML;
}

/**
 * Create HTML for an event card
 */
function createEventCard(event) {
    const eventDate = new Date(event.event_date);
    const formattedDate = eventDate.toLocaleDateString('en-US', { 
        weekday: 'short', 
        year: 'numeric', 
        month: 'short', 
        day: 'numeric' 
    });
    
    const registrationPercentage = event.max_participants ? 
        Math.round((event.total_registrations / event.max_participants) * 100) : 0;
    
    const statusColors = {
        'upcoming': 'success',
        'ongoing': 'primary',
        'completed': 'secondary',
        'cancelled': 'danger'
    };
    
    return `
        <div class="col-lg-4 col-md-6 mb-4">
            <div class="card event-card">
                <div class="event-banner ${event.event_type || 'meeting'}">
                    ${event.event_banner_url ? 
                        `<img src="${event.event_banner_url}" alt="${event.event_title}" class="w-100 h-100 object-fit-cover">` :
                        `<div class="text-center">
                            <i class="bi bi-calendar-event" style="font-size: 3rem;"></i>
                            <h5 class="mt-2">${event.event_type?.toUpperCase() || 'EVENT'}</h5>
                        </div>`
                    }
                    <span class="event-status-badge bg-${statusColors[event.status] || 'secondary'}">
                        ${event.status?.toUpperCase() || 'UNKNOWN'}
                    </span>
                </div>
                <div class="card-body">
                    <h5 class="card-title">${event.event_title}</h5>
                    <p class="card-text text-muted small">${event.event_description || 'No description available'}</p>
                    
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <div>
                            <i class="bi bi-calendar3 text-primary"></i>
                            <small class="ms-1">${formattedDate}</small>
                        </div>
                        ${event.event_time ? `
                            <div>
                                <i class="bi bi-clock text-success"></i>
                                <small class="ms-1">${event.event_time}</small>
                            </div>
                        ` : ''}
                    </div>
                    
                    ${event.event_location ? `
                        <div class="mb-3">
                            <i class="bi bi-geo-alt text-info"></i>
                            <small class="ms-1">${event.event_location}</small>
                        </div>
                    ` : ''}
                    
                    ${event.max_participants ? `
                        <div class="mb-3">
                            <div class="d-flex justify-content-between small">
                                <span>Registrations</span>
                                <span>${event.total_registrations || 0}/${event.max_participants}</span>
                            </div>
                            <div class="registration-progress">
                                <div class="registration-progress-bar" style="width: ${registrationPercentage}%"></div>
                            </div>
                        </div>
                    ` : ''}
                    
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            ${event.registration_fee > 0 ? 
                                `<span class="badge bg-warning">₹${event.registration_fee}</span>` :
                                `<span class="badge bg-success">Free</span>`
                            }
                            <span class="badge bg-info ms-1">${event.event_type || 'General'}</span>
                        </div>
                        <div class="btn-group" role="group">
                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="viewEvent(${event.id})" title="View Details">
                                <i class="bi bi-eye"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-success" onclick="editEvent(${event.id})" title="Edit Event">
                                <i class="bi bi-pencil"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-info" onclick="viewRegistrations(${event.id})" title="View Registrations">
                                <i class="bi bi-people"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-danger" onclick="deleteEvent(${event.id})" title="Delete Event">
                                <i class="bi bi-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
}

/**
 * Load event statistics
 */
async function loadEventStatistics() {
    try {
        const response = await apiService.get('/events/stats');
        
        if (response.success) {
            const stats = response.data;
            
            document.getElementById('totalEvents').textContent = stats.total_events || 0;
            document.getElementById('upcomingEvents').textContent = stats.upcoming_events || 0;
            document.getElementById('totalRegistrations').textContent = stats.total_registrations || 0;
            document.getElementById('totalRevenue').textContent = `₹${stats.total_revenue || 0}`;
        }
    } catch (error) {
        console.error('Error loading event statistics:', error);
    }
}

/**
 * Handle event form submission
 */
async function handleEventSubmission(e) {
    e.preventDefault();
    
    const form = e.target;
    const formData = new FormData(form);
    const submitButton = form.querySelector('button[type="submit"]');
    const spinner = document.getElementById('submitSpinner');
    const isEdit = form.dataset.eventId;
    
    try {
        // Show loading state
        submitButton.disabled = true;
        spinner.classList.remove('d-none');
        
        // Prepare data
        const eventData = {
            event_title: formData.get('event_title'),
            event_description: formData.get('event_description'),
            event_date: formData.get('event_date'),
            event_time: formData.get('event_time'),
            event_location: formData.get('event_location'),
            event_type: formData.get('event_type'),
            max_participants: formData.get('max_participants') ? parseInt(formData.get('max_participants')) : null,
            registration_fee: parseFloat(formData.get('registration_fee')) || 0,
            status: formData.get('status'),
            event_banner_url: formData.get('event_banner_url') || null
        };
        
        let response;
        if (isEdit) {
            response = await apiService.put(`/events/${isEdit}`, eventData);
        } else {
            response = await apiService.post('/events/', eventData);
        }
        
        if (response.success) {
            showNotification(
                isEdit ? 'Event updated successfully!' : 'Event created successfully!', 
                'success'
            );
            
            // Close modal and refresh data
            const modal = bootstrap.Modal.getInstance(document.getElementById('eventModal'));
            modal.hide();
            
            await loadEvents(currentPage);
            await loadEventStatistics();
        } else {
            throw new Error(response.message || 'Failed to save event');
        }
        
    } catch (error) {
        console.error('Error saving event:', error);
        showNotification('Error saving event: ' + error.message, 'error');
    } finally {
        submitButton.disabled = false;
        spinner.classList.add('d-none');
    }
}

/**
 * Load event data for editing
 */
async function loadEventForEdit(eventId) {
    try {
        const response = await apiService.get(`/events/${eventId}`);
        
        if (response.success) {
            const event = response.data;
            const form = document.getElementById('eventForm');
            
            // Populate form fields
            form.querySelector('#eventTitle').value = event.event_title || '';
            form.querySelector('#eventDescription').value = event.event_description || '';
            form.querySelector('#eventDate').value = event.event_date || '';
            form.querySelector('#eventTime').value = event.event_time || '';
            form.querySelector('#eventLocation').value = event.event_location || '';
            form.querySelector('#eventType').value = event.event_type || '';
            form.querySelector('#maxParticipants').value = event.max_participants || '';
            form.querySelector('#registrationFee').value = event.registration_fee || 0;
            form.querySelector('#eventStatus').value = event.status || 'upcoming';
            form.querySelector('#eventBanner').value = event.event_banner_url || '';
            
            // Update modal title and form state
            document.getElementById('eventModalLabel').textContent = 'Edit Event';
            form.dataset.eventId = eventId;
            form.querySelector('button[type="submit"]').textContent = 'Update Event';
        } else {
            throw new Error(response.message || 'Failed to load event');
        }
    } catch (error) {
        console.error('Error loading event for edit:', error);
        showNotification('Error loading event: ' + error.message, 'error');
    }
}

/**
 * Reset event form to default state
 */
function resetEventForm() {
    const form = document.getElementById('eventForm');
    form.reset();
    delete form.dataset.eventId;
    document.getElementById('eventModalLabel').textContent = 'Create New Event';
    form.querySelector('button[type="submit"]').textContent = 'Create Event';
}

/**
 * Filter events based on current filter values
 */
function filterEvents() {
    currentFilters = {};
    
    const eventType = document.getElementById('filterEventType')?.value;
    const status = document.getElementById('filterStatus')?.value;
    const dateFrom = document.getElementById('filterDateFrom')?.value;
    const dateTo = document.getElementById('filterDateTo')?.value;
    
    if (eventType) currentFilters.event_type = eventType;
    if (status) currentFilters.status = status;
    if (dateFrom) currentFilters.date_from = dateFrom;
    if (dateTo) currentFilters.date_to = dateTo;
    
    currentPage = 1;
    loadEvents();
}

/**
 * Search events
 */
function searchEvents() {
    const searchTerm = document.getElementById('searchEvents')?.value?.trim();
    
    if (searchTerm) {
        currentFilters.search = searchTerm;
    } else {
        delete currentFilters.search;
    }
    
    currentPage = 1;
    loadEvents();
}

/**
 * Clear all filters
 */
function clearFilters() {
    currentFilters = {};
    
    // Reset filter controls
    document.getElementById('filterEventType').value = '';
    document.getElementById('filterStatus').value = '';
    document.getElementById('filterDateFrom').value = '';
    document.getElementById('filterDateTo').value = '';
    document.getElementById('searchEvents').value = '';
    
    currentPage = 1;
    loadEvents();
}

/**
 * Render pagination controls
 */
function renderPagination() {
    const pagination = document.getElementById('eventsPagination');
    
    if (totalPages <= 1) {
        pagination.innerHTML = '';
        return;
    }
    
    let paginationHTML = '';
    
    // Previous button
    paginationHTML += `
        <li class="page-item ${currentPage === 1 ? 'disabled' : ''}">
            <a class="page-link" href="#" onclick="loadEvents(${currentPage - 1}); return false;">
                <i class="bi bi-chevron-left"></i>
            </a>
        </li>
    `;
    
    // Page numbers
    const startPage = Math.max(1, currentPage - 2);
    const endPage = Math.min(totalPages, startPage + 4);
    
    for (let i = startPage; i <= endPage; i++) {
        paginationHTML += `
            <li class="page-item ${i === currentPage ? 'active' : ''}">
                <a class="page-link" href="#" onclick="loadEvents(${i}); return false;">${i}</a>
            </li>
        `;
    }
    
    // Next button
    paginationHTML += `
        <li class="page-item ${currentPage === totalPages ? 'disabled' : ''}">
            <a class="page-link" href="#" onclick="loadEvents(${currentPage + 1}); return false;">
                <i class="bi bi-chevron-right"></i>
            </a>
        </li>
    `;
    
    pagination.innerHTML = paginationHTML;
}

/**
 * Event action functions
 */
function viewEvent(eventId) {
    // TODO: Implement event details view
    window.location.href = `event-details.html?id=${eventId}`;
}

function editEvent(eventId) {
    const modal = new bootstrap.Modal(document.getElementById('eventModal'));
    modal.show();
    // The modal event listener will handle loading the event data
}

function viewRegistrations(eventId) {
    window.location.href = `event-registrations.html?event_id=${eventId}`;
}

async function deleteEvent(eventId) {
    if (!confirm('Are you sure you want to delete this event? This action cannot be undone.')) {
        return;
    }
    
    try {
        const response = await apiService.delete(`/events/${eventId}`);
        
        if (response.success) {
            showNotification('Event deleted successfully!', 'success');
            await loadEvents(currentPage);
            await loadEventStatistics();
        } else {
            throw new Error(response.message || 'Failed to delete event');
        }
    } catch (error) {
        console.error('Error deleting event:', error);
        showNotification('Error deleting event: ' + error.message, 'error');
    }
}

/**
 * Show notification toast
 */
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}
