/**
 * KSO Chandigarh - Expenses Management
 * Enhanced with DataTables functionality
 */

class ExpensesManager {
    constructor() {
        this.expensesTable = null;
        this.init();
    }

    init() {
        this.initDataTable();
        this.bindEvents();
        this.loadExpenses();
    }

    initDataTable() {
        if ($.fn.DataTable.isDataTable('#expensesTable')) {
            $('#expensesTable').DataTable().destroy();
        }

        this.expensesTable = $('#expensesTable').DataTable({
            responsive: true,
            pageLength: 25,
            lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
            order: [[5, 'desc']], // Sort by date (latest first)
            columnDefs: [
                {
                    targets: [0], // Receipt/Invoice column
                    orderable: false,
                    searchable: false,
                    width: "60px"
                },
                {
                    targets: [7], // Actions column
                    orderable: false,
                    searchable: false,
                    width: "120px"
                },
                {
                    targets: [3], // Amount column
                    render: function(data, type, row) {
                        if (type === 'display') {
                            return formatCurrency(data);
                        }
                        return data;
                    }
                },
                {
                    targets: [4], // Status column
                    render: function(data, type, row) {
                        if (type === 'display') {
                            return getExpenseStatusBadge(data);
                        }
                        return data;
                    }
                },
                {
                    targets: [5], // Date column
                    render: function(data, type, row) {
                        if (type === 'display' && data) {
                            return formatDate(data);
                        }
                        return data;
                    }
                }
            ],
            dom: '<"row"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>' +
                 '<"row"<"col-sm-12"tr>>' +
                 '<"row"<"col-sm-12 col-md-5"i><"col-sm-12 col-md-7"p>>',
            language: {
                search: "Search expenses:",
                lengthMenu: "Show _MENU_ expenses per page",
                info: "Showing _START_ to _END_ of _TOTAL_ expenses",
                infoEmpty: "No expenses found",
                infoFiltered: "(filtered from _MAX_ total expenses)",
                emptyTable: "No expenses available",
                zeroRecords: "No matching expenses found"
            },
            footerCallback: function(row, data, start, end, display) {
                const api = this.api();
                
                // Calculate total for displayed data
                const total = api
                    .column(3, { page: 'current' })
                    .data()
                    .reduce(function(a, b) {
                        return parseFloat(a) + parseFloat(b);
                    }, 0);

                // Update footer
                $(api.column(3).footer()).html(formatCurrency(total));
            }
        });
    }

    bindEvents() {
        // Add expense button
        $('#addExpenseBtn').on('click', () => {
            this.showAddExpenseModal();
        });

        // Category filter
        $('#categoryFilter').on('change', () => {
            this.filterByCategory();
        });

        // Status filter
        $('#statusFilter').on('change', () => {
            this.filterByStatus();
        });

        // Date range filter
        $('#dateRangeFilter').on('change', () => {
            this.filterByDateRange();
        });

        // Export buttons
        $('#exportExpensesBtn').on('click', () => {
            this.exportExpenses();
        });

        // Generate report button
        $('#generateReportBtn').on('click', () => {
            this.generateReport();
        });
    }

    loadExpenses() {
        this.showLoading();

        fetch('/access/admin/api/expenses/list.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    this.populateTable(data.expenses);
                    this.updateStats(data.stats);
                } else {
                    this.showError('Failed to load expenses: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error loading expenses:', error);
                this.showError('Failed to load expenses. Please try again.');
            })
            .finally(() => {
                this.hideLoading();
            });
    }

    populateTable(expenses) {
        const tableData = expenses.map(expense => [
            this.createReceiptColumn(expense),
            expense.description,
            expense.category,
            expense.amount,
            expense.status,
            expense.date,
            expense.vendor_name || '-',
            this.createActionsColumn(expense)
        ]);

        this.expensesTable.clear().rows.add(tableData).draw();
    }

    createReceiptColumn(expense) {
        if (expense.receipt_url) {
            return `<button type="button" class="btn btn-sm btn-outline-primary" onclick="expensesManager.viewReceipt('${expense.id}')" title="View Receipt">
                        <i class="bi bi-receipt"></i>
                    </button>`;
        } else {
            return `<span class="text-muted">
                        <i class="bi bi-receipt" title="No receipt"></i>
                    </span>`;
        }
    }

    createActionsColumn(expense) {
        return `
            <div class="btn-group btn-group-sm" role="group">
                <button type="button" class="btn btn-outline-primary" onclick="expensesManager.viewExpense('${expense.id}')" title="View">
                    <i class="bi bi-eye"></i>
                </button>
                <button type="button" class="btn btn-outline-secondary" onclick="expensesManager.editExpense('${expense.id}')" title="Edit">
                    <i class="bi bi-pencil"></i>
                </button>
                <button type="button" class="btn btn-outline-danger" onclick="expensesManager.deleteExpense('${expense.id}')" title="Delete">
                    <i class="bi bi-trash"></i>
                </button>
            </div>
        `;
    }

    filterByCategory() {
        const category = $('#categoryFilter').val();
        if (category === 'all') {
            this.expensesTable.column(2).search('').draw();
        } else {
            this.expensesTable.column(2).search(category).draw();
        }
    }

    filterByStatus() {
        const status = $('#statusFilter').val();
        if (status === 'all') {
            this.expensesTable.column(4).search('').draw();
        } else {
            this.expensesTable.column(4).search(status).draw();
        }
    }

    filterByDateRange() {
        const range = $('#dateRangeFilter').val();
        // Implementation for date range filtering
        console.log('Filtering by date range:', range);
    }

    updateStats(stats) {
        $('#totalExpensesAmount').text(formatCurrency(stats.total_amount || 0));
        $('#thisMonthExpenses').text(formatCurrency(stats.this_month_amount || 0));
        $('#pendingExpensesCount').text(stats.pending_count || 0);
        $('#approvedExpensesCount').text(stats.approved_count || 0);
    }

    viewExpense(id) {
        // Implementation for viewing expense details
        console.log('Viewing expense:', id);
    }

    editExpense(id) {
        // Implementation for editing expense
        console.log('Editing expense:', id);
    }

    deleteExpense(id) {
        if (confirm('Are you sure you want to delete this expense?')) {
            fetch(`/access/admin/api/expenses/delete.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ id: id })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    this.loadExpenses(); // Reload the table
                    this.showSuccess('Expense deleted successfully');
                } else {
                    this.showError('Failed to delete expense: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error deleting expense:', error);
                this.showError('Failed to delete expense. Please try again.');
            });
        }
    }

    viewReceipt(id) {
        window.open(`/access/admin/api/expenses/receipt.php?id=${id}`, '_blank');
    }

    showAddExpenseModal() {
        $('#addExpenseModal').modal('show');
    }

    exportExpenses() {
        const params = new URLSearchParams({
            category: $('#categoryFilter').val(),
            status: $('#statusFilter').val(),
            date_range: $('#dateRangeFilter').val()
        });
        
        window.open(`/access/admin/api/expenses/export.php?${params.toString()}`, '_blank');
    }

    generateReport() {
        const params = new URLSearchParams({
            category: $('#categoryFilter').val(),
            status: $('#statusFilter').val(),
            date_range: $('#dateRangeFilter').val()
        });
        
        window.open(`/access/admin/api/expenses/report.php?${params.toString()}`, '_blank');
    }

    showLoading() {
        if (this.expensesTable) {
            this.expensesTable.processing(true);
        }
    }

    hideLoading() {
        if (this.expensesTable) {
            this.expensesTable.processing(false);
        }
    }

    showError(message) {
        console.error(message);
        // Implement toast notification
    }

    showSuccess(message) {
        console.log(message);
        // Implement toast notification
    }
}

// Utility functions
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', {
        style: 'currency',
        currency: 'INR',
        minimumFractionDigits: 2
    }).format(amount);
}

function getExpenseStatusBadge(status) {
    const badges = {
        'pending': '<span class="badge bg-warning">Pending</span>',
        'approved': '<span class="badge bg-success">Approved</span>',
        'rejected': '<span class="badge bg-danger">Rejected</span>',
        'paid': '<span class="badge bg-info">Paid</span>'
    };
    return badges[status] || `<span class="badge bg-light text-dark">${status}</span>`;
}

function formatDate(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Initialize when document is ready
let expensesManager;
$(document).ready(function() {
    expensesManager = new ExpensesManager();
});
