// Finance Management JavaScript Module
class FinanceManager {
    constructor() {
        // Use a centralized API service for making requests
        this.api = new AdminAPIService();
        this.transactions = [];
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadInitialData();
    }

    bindEvents() {
        // Add transaction form submission
        document.getElementById('addTransactionForm')?.addEventListener('submit', e => this.handleAddTransaction(e));

        // Edit transaction form submission
        document.getElementById('editTransactionForm')?.addEventListener('submit', e => this.handleEditTransaction(e));

        // Filters and search
        document.getElementById('financeFilterForm')?.addEventListener('submit', e => {
            e.preventDefault();
            this.loadTransactions();
        });
        document.getElementById('globalSearch')?.addEventListener('input', debounce(() => this.loadTransactions(), 500));

        // Export buttons
        document.getElementById('exportExcelBtn')?.addEventListener('click', () => this.export('excel'));
        document.getElementById('exportPdfBtn')?.addEventListener('click', () => this.export('pdf'));

        // Modal for adding a transaction
        const addTransactionModal = document.getElementById('addTransactionModal');
        addTransactionModal?.addEventListener('show.bs.modal', () => this.resetAddForm());
    }

    async loadInitialData() {
        showLoading();
        try {
            // Fetch transactions and dashboard stats in parallel
            const [transactionsResponse, statsResponse] = await Promise.all([
                this.api.get('finance'),
                this.api.get('dashboard')
            ]);

            if (transactionsResponse.success) {
                this.transactions = transactionsResponse.data;
                this.displayTransactions();
            } else {
                showError('Failed to load transactions: ' + transactionsResponse.message);
            }

            if (statsResponse.success) {
                this.displayStatistics(statsResponse.data);
                this.updateCharts(statsResponse.data);
            } else {
                showError('Failed to load statistics: ' + statsResponse.message);
            }
        } catch (error) {
            showError('Error loading initial data: ' + error.message);
            console.error(error);
        } finally {
            hideLoading();
        }
    }

    async loadTransactions() {
        showLoading();
        try {
            // Collect filter parameters
            const params = new URLSearchParams(new FormData(document.getElementById('financeFilterForm'))).toString();
            const search = document.getElementById('globalSearch')?.value;
            const query = params + (search ? `&search=${search}` : '');

            const response = await this.api.get(`finance?${query}`);
            if (response.success) {
                this.transactions = response.data;
                this.displayTransactions();
            } else {
                showError('Failed to load transactions: ' + response.message);
            }
        } catch (error) {
            showError('Error loading transactions: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    displayTransactions() {
        const tbody = document.getElementById('transactionsTableBody');
        if (!tbody) return;

        tbody.innerHTML = '';
        if (!this.transactions || this.transactions.length === 0) {
            tbody.innerHTML = '<tr><td colspan="8" class="text-center">No transactions found.</td></tr>';
            return;
        }

        this.transactions.forEach(transaction => {
            tbody.appendChild(this.createTransactionRow(transaction));
        });
    }

    createTransactionRow(transaction) {
        const row = document.createElement('tr');
        const isIncome = ['membership_fee', 'donation', 'event_fee', 'other_income'].includes(transaction.transaction_type);
        const statusConfig = {
            completed: { class: 'success', icon: 'check-circle' },
            pending: { class: 'warning', icon: 'hourglass-split' },
            failed: { class: 'danger', icon: 'x-circle' },
            cancelled: { class: 'secondary', icon: 'slash-circle' }
        };
        const status = statusConfig[transaction.status] || { class: 'secondary', icon: 'question-circle' };

        row.innerHTML = `
            <td>${transaction.id}</td>
            <td>
                <span class="badge bg-${isIncome ? 'success' : 'danger'}">
                    ${isIncome ? 'Income' : 'Expense'}
                </span>
            </td>
            <td>${escapeHtml(transaction.description)}</td>
            <td>${formatCurrency(transaction.amount)}</td>
            <td><span class="badge bg-info">${escapeHtml(transaction.transaction_type.replace('_', ' '))}</span></td>
            <td><span class="badge bg-${status.class}"><i class="bi bi-${status.icon}"></i> ${escapeHtml(transaction.status)}</span></td>
            <td>${formatDate(transaction.created_at)}</td>
            <td>
                <button class="btn btn-sm btn-outline-primary" onclick="financeManager.openEditModal(${transaction.id})"><i class="bi bi-pencil"></i></button>
                <button class="btn btn-sm btn-outline-danger" onclick="financeManager.deleteTransaction(${transaction.id})"><i class="bi bi-trash"></i></button>
            </td>
        `;
        return row;
    }

    displayStatistics(stats) {
        document.getElementById('totalIncome').textContent = formatCurrency(stats.finance.total_income);
        document.getElementById('monthlyIncome').textContent = formatCurrency(stats.finance.income_this_month);
        document.getElementById('totalExpenses').textContent = formatCurrency(stats.finance.total_expenses);
        document.getElementById('monthlyExpenses').textContent = formatCurrency(stats.finance.expenses_this_month);
        document.getElementById('pendingPayments').textContent = formatCurrency(stats.fees.pending);
        document.getElementById('netBalance').textContent = formatCurrency(stats.finance.total_revenue);
    }

    updateCharts(stats) {
        // Financial Trends Chart
        const financialChartCtx = document.getElementById('financialChart')?.getContext('2d');
        if (financialChartCtx && stats.monthly_data) {
            this.renderFinancialChart(financialChartCtx, stats.monthly_data);
        }

        // Transaction Types Chart
        const transactionTypeChartCtx = document.getElementById('transactionTypeChart')?.getContext('2d');
        if (transactionTypeChartCtx && this.transactions) {
            this.renderTransactionTypeChart(transactionTypeChartCtx, this.transactions);
        }
    }

    renderFinancialChart(ctx, monthlyData) {
        const labels = monthlyData.map(d => d.month);
        const revenueData = monthlyData.map(d => d.revenue);
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels,
                datasets: [{
                    label: 'Monthly Revenue',
                    data: revenueData,
                    borderColor: '#4facfe',
                    backgroundColor: 'rgba(79, 172, 254, 0.2)',
                    fill: true,
                    tension: 0.3
                }]
            },
            options: { responsive: true, maintainAspectRatio: false }
        });
    }

    renderTransactionTypeChart(ctx, transactions) {
        const typeCounts = transactions.reduce((acc, t) => {
            acc[t.transaction_type] = (acc[t.transaction_type] || 0) + 1;
            return acc;
        }, {});

        new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: Object.keys(typeCounts),
                datasets: [{
                    data: Object.values(typeCounts),
                    backgroundColor: ['#4facfe', '#fa709a', '#43e97b', '#f093fb', '#f5576c']
                }]
            },
            options: { responsive: true, maintainAspectRatio: false }
        });
    }

    resetAddForm() {
        document.getElementById('addTransactionForm')?.reset();
    }

    async handleAddTransaction(e) {
        e.preventDefault();
        const form = e.target;
        const formData = new FormData(form);
        const data = Object.fromEntries(formData.entries());

        showLoading();
        try {
            const response = await this.api.post('finance', data);
            if (response.success) {
                showSuccess('Transaction added successfully.');
                bootstrap.Modal.getInstance(document.getElementById('addTransactionModal')).hide();
                this.loadInitialData();
            } else {
                showError(response.message || 'Failed to add transaction.');
            }
        } catch (error) {
            showError('An error occurred: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    async openEditModal(id) {
        showLoading();
        try {
            const response = await this.api.get(`finance?id=${id}`);
            if (response.success) {
                const form = document.getElementById('editTransactionForm');
                const transaction = response.data;
                form.querySelector('#editTransactionId').value = transaction.id;
                form.querySelector('#editTransactionType').value = transaction.transaction_type;
                form.querySelector('#editAmount').value = transaction.amount;
                form.querySelector('#editDescription').value = transaction.description;
                form.querySelector('#editStatus').value = transaction.status;
                
                new bootstrap.Modal(document.getElementById('editTransactionModal')).show();
            } else {
                showError(response.message || 'Failed to load transaction details.');
            }
        } catch (error) {
            showError('An error occurred: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    async handleEditTransaction(e) {
        e.preventDefault();
        const form = e.target;
        const formData = new FormData(form);
        const data = Object.fromEntries(formData.entries());
        data.id = document.getElementById('editTransactionId').value;

        showLoading();
        try {
            const response = await this.api.put('finance', data);
            if (response.success) {
                showSuccess('Transaction updated successfully.');
                bootstrap.Modal.getInstance(document.getElementById('editTransactionModal')).hide();
                this.loadInitialData();
            } else {
                showError(response.message || 'Failed to update transaction.');
            }
        } catch (error) {
            showError('An error occurred: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    async deleteTransaction(id) {
        if (!confirm('Are you sure you want to delete this transaction?')) return;

        showLoading();
        try {
            const response = await this.api.delete('finance', { id });
            if (response.success) {
                showSuccess('Transaction deleted successfully.');
                this.loadInitialData();
            } else {
                showError(response.message || 'Failed to delete transaction.');
            }
        } catch (error) {
            showError('An error occurred: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    export(format) {
        // This is a placeholder. A proper implementation would generate the file on the server.
        alert(`Exporting to ${format} is not yet implemented.`);
    }
}

// Initialize the FinanceManager when the DOM is fully loaded
document.addEventListener('DOMContentLoaded', () => {
    window.financeManager = new FinanceManager();
});

// Helper functions (assuming these are globally available or part of a utility script)
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', { style: 'currency', currency: 'INR' }).format(amount);
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    return new Date(dateString).toLocaleDateString('en-GB', { day: '2-digit', month: 'short', year: 'numeric' });
}

function escapeHtml(str) {
    return str?.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&#039;") || '';
}

function debounce(func, delay) {
    let timeout;
    return function(...args) {
        clearTimeout(timeout);
        timeout = setTimeout(() => func.apply(this, args), delay);
    };
}

// Assuming showLoading, hideLoading, showError, showSuccess are defined globally
// e.g., in a file like `utils.js`
function showLoading() { console.log("Loading..."); }
function hideLoading() { console.log("Loading complete."); }
function showError(message) { console.error(message); alert(`Error: ${message}`); }
function showSuccess(message) { console.log(message); }
