/**
 * Real-time Notification Manager with Browser Notifications
 */

class RealTimeNotificationManager {
    constructor() {
        this.eventSource = null;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 5;
        this.reconnectInterval = 5000; // 5 seconds
        this.notificationSound = null;
        this.userId = null;
        this.userType = 'member';
        this.isConnected = false;
        this.unreadCount = 0;
        
        this.initializeAudio();
        this.requestNotificationPermission();
        this.initializeUI();
    }
    
    /**
     * Initialize notification audio
     */
    initializeAudio() {
        this.notificationSound = new Audio('../assets/sounds/notification.mp3');
        this.notificationSound.volume = 0.5;
    }
    
    /**
     * Request browser notification permission
     */
    async requestNotificationPermission() {
        if ('Notification' in window) {
            if (Notification.permission === 'default') {
                const permission = await Notification.requestPermission();
                console.log('Notification permission:', permission);
            }
        }
    }
    
    /**
     * Initialize UI elements
     */
    initializeUI() {
        // Create notification container if it doesn't exist
        if (!document.getElementById('realtime-notifications')) {
            const container = document.createElement('div');
            container.id = 'realtime-notifications';
            container.className = 'realtime-notifications-container';
            container.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 9999;
                max-width: 350px;
            `;
            document.body.appendChild(container);
        }
        
        // Update notification badge
        this.updateNotificationBadge();
    }
    
    /**
     * Disconnect from real-time notification stream
     */
    disconnect() {
        if (this.eventSource) {
            this.eventSource.close();
            this.eventSource = null;
        }
        
        this.isConnected = false;
        this.updateConnectionStatus(false);
        
        console.log('Disconnected from notification stream');
    }
    
    /**
     * Reconnect to notification stream
     */
    reconnect() {
        if (this.userId) {
            this.connect(this.userId, this.userType);
        }
    }

    /**
     * Connect to real-time notification stream
     */
    connect(userId, userType = 'member') {
        this.userId = userId;
        this.userType = userType;
        
        if (this.eventSource) {
            this.eventSource.close();
        }
        
        const url = `/KSOCHD/api/realtime/notifications.php`;
        this.eventSource = new EventSource(url);
        
        this.eventSource.onopen = (event) => {
            console.log('Connected to notification stream');
            this.isConnected = true;
            this.reconnectAttempts = 0;
            this.updateConnectionStatus(true);
        };
        
        this.eventSource.onmessage = (event) => {
            console.log('Received message:', event.data);
        };
        
        this.eventSource.addEventListener('connected', (event) => {
            console.log('Connection confirmed:', event.data);
        });
        
        this.eventSource.addEventListener('notification', (event) => {
            const notification = JSON.parse(event.data);
            this.handleNewNotification(notification);
        });
        
        this.eventSource.addEventListener('heartbeat', (event) => {
            const data = JSON.parse(event.data);
            this.unreadCount = data.unread_count;
            this.updateNotificationBadge();
        });
        
        this.eventSource.addEventListener('error', (event) => {
            console.error('SSE error:', event);
            this.isConnected = false;
            this.updateConnectionStatus(false);
            this.attemptReconnect();
        });
        
        this.eventSource.addEventListener('close', (event) => {
            console.log('Connection closed:', event.data);
            this.isConnected = false;
            this.updateConnectionStatus(false);
        });
        
        this.eventSource.onerror = (event) => {
            console.error('EventSource failed:', event);
            this.isConnected = false;
            this.updateConnectionStatus(false);
            this.attemptReconnect();
        };
    }
    
    /**
     * Handle new notification
     */
    handleNewNotification(notification) {
        console.log('New notification received:', notification);
        
        // Update unread count
        this.unreadCount++;
        this.updateNotificationBadge();
        
        // Show browser notification
        this.showBrowserNotification(notification);
        
        // Show in-app notification
        this.showInAppNotification(notification);
        
        // Play sound
        this.playNotificationSound();
        
        // Update notifications list if visible
        this.updateNotificationsList(notification);
        
        // Trigger custom event for other components
        this.dispatchNotificationEvent(notification);
    }
    
    /**
     * Show browser notification
     */
    showBrowserNotification(notification) {
        if (Notification.permission === 'granted') {
            const options = {
                body: notification.message,
                icon: '../assets/img/logo.png',
                badge: '../assets/img/notification-badge.png',
                tag: `notification-${notification.id}`,
                requireInteraction: notification.priority === 'urgent',
                silent: false,
                timestamp: Date.now(),
                data: {
                    id: notification.id,
                    url: '/KSOCHD/frontend/pages/notifications.html'
                }
            };
            
            const browserNotification = new Notification(notification.title, options);
            
            browserNotification.onclick = function() {
                window.focus();
                window.location.href = options.data.url;
                browserNotification.close();
            };
            
            // Auto-close after 10 seconds unless urgent
            if (notification.priority !== 'urgent') {
                setTimeout(() => {
                    browserNotification.close();
                }, 10000);
            }
        }
    }
    
    /**
     * Show in-app notification toast
     */
    showInAppNotification(notification) {
        const container = document.getElementById('realtime-notifications');
        
        const toast = document.createElement('div');
        toast.className = `notification-toast notification-${notification.type} priority-${notification.priority}`;
        toast.style.cssText = `
            background: white;
            border-left: 4px solid ${this.getTypeColor(notification.type)};
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            margin-bottom: 10px;
            padding: 15px;
            transform: translateX(100%);
            transition: transform 0.3s ease;
            cursor: pointer;
        `;
        
        toast.innerHTML = `
            <div style="display: flex; justify-content: between; align-items: flex-start;">
                <div style="flex: 1;">
                    <div style="font-weight: bold; color: #333; margin-bottom: 5px;">
                        ${this.getTypeIcon(notification.type)} ${notification.title}
                    </div>
                    <div style="color: #666; font-size: 14px; line-height: 1.4;">
                        ${notification.message}
                    </div>
                    <div style="color: #999; font-size: 12px; margin-top: 5px;">
                        ${this.formatTime(notification.created_at)}
                    </div>
                </div>
                <button onclick="this.parentElement.parentElement.remove()" 
                        style="background: none; border: none; color: #999; cursor: pointer; font-size: 18px; padding: 0; margin-left: 10px;">
                    ×
                </button>
            </div>
        `;
        
        toast.onclick = () => {
            window.location.href = '/KSOCHD/frontend/pages/notifications.html';
        };
        
        container.appendChild(toast);
        
        // Animate in
        setTimeout(() => {
            toast.style.transform = 'translateX(0)';
        }, 100);
        
        // Auto-remove after delay based on priority
        const delay = notification.priority === 'urgent' ? 15000 : 8000;
        setTimeout(() => {
            if (toast.parentElement) {
                toast.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (toast.parentElement) {
                        toast.remove();
                    }
                }, 300);
            }
        }, delay);
    }
    
    /**
     * Play notification sound
     */
    playNotificationSound() {
        if (this.notificationSound) {
            this.notificationSound.play().catch(e => {
                console.log('Could not play notification sound:', e);
            });
        }
    }
    
    /**
     * Update notification badge
     */
    updateNotificationBadge() {
        const badge = document.querySelector('.notification-badge');
        if (badge) {
            badge.textContent = this.unreadCount;
            badge.style.display = this.unreadCount > 0 ? 'inline' : 'none';
        }
        
        // Update page title
        if (this.unreadCount > 0) {
            document.title = `(${this.unreadCount}) ${document.title.replace(/^\(\d+\)\s/, '')}`;
        } else {
            document.title = document.title.replace(/^\(\d+\)\s/, '');
        }
    }
    
    /**
     * Update connection status indicator
     */
    updateConnectionStatus(connected) {
        const indicator = document.querySelector('.connection-status');
        if (indicator) {
            indicator.className = `connection-status ${connected ? 'connected' : 'disconnected'}`;
            indicator.title = connected ? 'Connected to notifications' : 'Disconnected from notifications';
        }
    }
    
    /**
     * Attempt to reconnect
     */
    attemptReconnect() {
        if (this.reconnectAttempts < this.maxReconnectAttempts) {
            this.reconnectAttempts++;
            console.log(`Attempting to reconnect (${this.reconnectAttempts}/${this.maxReconnectAttempts})...`);
            
            setTimeout(() => {
                this.connect(this.userId, this.userType);
            }, this.reconnectInterval * this.reconnectAttempts);
        } else {
            console.error('Max reconnection attempts reached');
            this.showConnectionError();
        }
    }
    
    /**
     * Show connection error
     */
    showConnectionError() {
        const container = document.getElementById('realtime-notifications');
        const errorToast = document.createElement('div');
        errorToast.className = 'notification-toast notification-error';
        errorToast.innerHTML = `
            <div style="color: #dc3545; font-weight: bold;">
                ⚠️ Connection Lost
            </div>
            <div style="color: #666; font-size: 14px; margin-top: 5px;">
                Real-time notifications are unavailable. Please refresh the page.
            </div>
        `;
        container.appendChild(errorToast);
    }
    
    /**
     * Update notifications list if visible
     */
    updateNotificationsList(notification) {
        // If we're on the notifications page, refresh the list
        if (window.location.pathname.includes('notifications.html')) {
            if (typeof window.notificationsManager !== 'undefined') {
                window.notificationsManager.loadNotifications();
            }
        }
    }
    
    /**
     * Dispatch custom notification event
     */
    dispatchNotificationEvent(notification) {
        const event = new CustomEvent('newNotification', {
            detail: notification
        });
        document.dispatchEvent(event);
    }
    
    /**
     * Get type color
     */
    getTypeColor(type) {
        const colors = {
            'info': '#17a2b8',
            'success': '#28a745',
            'warning': '#ffc107',
            'error': '#dc3545',
            'announcement': '#6f42c1'
        };
        return colors[type] || '#007bff';
    }
    
    /**
     * Get type icon
     */
    getTypeIcon(type) {
        const icons = {
            'info': 'ℹ️',
            'success': '✅',
            'warning': '⚠️',
            'error': '❌',
            'announcement': '📢'
        };
        return icons[type] || '🔔';
    }
    
    /**
     * Format time
     */
    formatTime(timestamp) {
        const date = new Date(timestamp);
        const now = new Date();
        const diff = now - date;
        
        if (diff < 60000) { // Less than 1 minute
            return 'Just now';
        } else if (diff < 3600000) { // Less than 1 hour
            return `${Math.floor(diff / 60000)} min ago`;
        } else if (diff < 86400000) { // Less than 1 day
            return `${Math.floor(diff / 3600000)} hour ago`;
        } else {
            return date.toLocaleDateString();
        }
    }
    
    /**
     * Disconnect from notification stream
     */
    disconnect() {
        if (this.eventSource) {
            this.eventSource.close();
            this.eventSource = null;
        }
        this.isConnected = false;
        this.updateConnectionStatus(false);
    }
    
    /**
     * Mark notification as read
     */
    async markAsRead(notificationId) {
        try {
            const response = await fetch(`/KSOCHD/api/notifications/?id=${notificationId}&action=mark_read`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json'
                }
            });
            
            if (response.ok) {
                this.unreadCount = Math.max(0, this.unreadCount - 1);
                this.updateNotificationBadge();
            }
        } catch (error) {
            console.error('Error marking notification as read:', error);
        }
    }
}

// Global instance
window.realtimeNotifications = new RealTimeNotificationManager();
