// ... (existing code)
class ReceiptManager {
    constructor() {
        this.api = new AdminAPIService();
        this.receipts = [];
        this.pagination = {};
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadInitialData();
        this.populateYearFilter();
    }

    bindEvents() {
        document.getElementById('searchReceipts').addEventListener('input', debounce(() => this.loadReceipts(), 500));
        document.getElementById('filterTransactionType').addEventListener('change', () => this.loadReceipts());
        document.getElementById('filterYear').addEventListener('change', () => this.loadReceipts());
        document.getElementById('generateReceiptBtn').addEventListener('click', () => this.generateReceipt());
        document.getElementById('bulkGenerateBtn').addEventListener('click', () => this.bulkGenerateReceipts());
        document.getElementById('prevPage').addEventListener('click', () => this.changePage(-1));
        document.getElementById('nextPage').addEventListener('click', () => this.changePage(1));
    }

    async loadInitialData() {
        showLoading();
        try {
            const [receiptsResponse, statsResponse] = await Promise.all([
                this.api.get('receipts'),
                this.api.get('receipts/stats')
            ]);

            if (receiptsResponse.success) {
                this.receipts = receiptsResponse.data.receipts;
                this.pagination = receiptsResponse.data.pagination;
                this.displayReceipts();
                this.updatePagination();
            } else {
                showError('Failed to load receipts: ' + receiptsResponse.message);
            }

            if (statsResponse.success) {
                this.displayStats(statsResponse.data);
            } else {
                showError('Failed to load stats: ' + statsResponse.message);
            }
        } catch (error) {
            showError('Error loading initial data: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    async loadReceipts(page = 1) {
        showLoading();
        try {
            const search = document.getElementById('searchReceipts').value;
            const type = document.getElementById('filterTransactionType').value;
            const year = document.getElementById('filterYear').value;
            
            const params = new URLSearchParams({ page, search, type, year });
            const response = await this.api.get(`receipts?${params}`);

            if (response.success) {
                this.receipts = response.data.receipts;
                this.pagination = response.data.pagination;
                this.displayReceipts();
                this.updatePagination();
            } else {
                showError('Failed to load receipts: ' + response.message);
            }
        } catch (error) {
            showError('Error loading receipts: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    displayReceipts() {
        const tbody = document.getElementById('receiptsTableBody');
        tbody.innerHTML = '';
        if (this.receipts.length === 0) {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center">No receipts found.</td></tr>';
            return;
        }
        this.receipts.forEach(r => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td><a href="#" onclick="receiptManager.viewReceipt(${r.id})">${r.receipt_number}</a></td>
                <td>${escapeHtml(r.member_name)} (${r.kso_id})</td>
                <td><span class="badge bg-info">${escapeHtml(r.transaction_type.replace('_', ' '))}</span></td>
                <td>${formatCurrency(r.amount)}</td>
                <td>${formatDate(r.receipt_date)}</td>
                <td><span class="badge bg-success">${r.status}</span></td>
                <td>
                    <button class="btn btn-sm btn-outline-primary" onclick="receiptManager.viewReceipt(${r.id})"><i class="bi bi-eye"></i></button>
                    <button class="btn btn-sm btn-outline-danger" onclick="receiptManager.deleteReceipt(${r.id})"><i class="bi bi-trash"></i></button>
                </td>
            `;
            tbody.appendChild(row);
        });
    }

    displayStats(stats) {
        document.getElementById('totalReceipts').textContent = stats.total_receipts;
        document.getElementById('todayReceipts').textContent = stats.today_receipts;
        document.getElementById('thisMonthReceipts').textContent = stats.this_month_receipts;
        document.getElementById('totalAmount').textContent = formatCurrency(stats.total_amount);
    }

    updatePagination() {
        const { page, pages, total } = this.pagination;
        document.getElementById('paginationInfo').textContent = `Page ${page} of ${pages} (${total} receipts)`;
        document.getElementById('prevPage').disabled = page <= 1;
        document.getElementById('nextPage').disabled = page >= pages;
    }

    changePage(direction) {
        const newPage = this.pagination.page + direction;
        if (newPage > 0 && newPage <= this.pagination.pages) {
            this.loadReceipts(newPage);
        }
    }

    populateYearFilter() {
        const yearFilter = document.getElementById('filterYear');
        const currentYear = new Date().getFullYear();
        for (let i = 0; i < 5; i++) {
            const year = currentYear - i;
            yearFilter.innerHTML += `<option value="${year}">${year}</option>`;
        }
    }

    async showGenerateReceiptModal() {
        const select = document.getElementById('transactionSelect');
        select.innerHTML = '<option>Loading transactions...</option>';
        const modal = new bootstrap.Modal(document.getElementById('generateReceiptModal'));
        modal.show();

        const response = await this.api.get('receipts?action=list_transactions');
        if (response.success && response.data.length > 0) {
            select.innerHTML = '<option value="">Select a transaction...</option>';
            response.data.forEach(t => {
                select.innerHTML += `<option value="${t.id}">${t.member_name} - ${t.description} (${formatCurrency(t.amount)})</option>`;
            });
        } else {
            select.innerHTML = '<option>No eligible transactions found.</option>';
        }
    }

    async generateReceipt() {
        const transactionId = document.getElementById('transactionSelect').value;
        if (!transactionId) {
            showError('Please select a transaction.');
            return;
        }
        
        showLoading();
        try {
            const response = await this.api.post('receipts', { transaction_id: transactionId });
            if (response.success) {
                showSuccess('Receipt generated successfully!');
                bootstrap.Modal.getInstance(document.getElementById('generateReceiptModal')).hide();
                this.loadInitialData();
            } else {
                showError(response.message);
            }
        } catch (error) {
            showError('An error occurred: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    async bulkGenerateReceipts() {
        if (!confirm('Are you sure you want to generate receipts for all eligible transactions?')) return;

        showLoading();
        try {
            const response = await this.api.post('receipts', { action: 'bulk_generate' });
            if (response.success) {
                showSuccess(`${response.data.generated_count} receipts were generated.`);
                this.loadInitialData();
            } else {
                showError(response.message);
            }
        } catch (error) {
            showError('An error occurred: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    async viewReceipt(id) {
        showLoading();
        const modal = new bootstrap.Modal(document.getElementById('receiptPreviewModal'));
        const content = document.getElementById('receiptPreviewContent');
        content.innerHTML = 'Loading receipt...';
        modal.show();

        try {
            const response = await this.api.get(`receipts?id=${id}`);
            if (response.success) {
                const r = response.data;
                content.innerHTML = `
                    <div class="text-center mb-4">
                        <h3>KSO Chandigarh</h3>
                        <p>Official Receipt</p>
                    </div>
                    <p><strong>Receipt No:</strong> ${r.receipt_number}</p>
                    <p><strong>Date:</strong> ${formatDate(r.receipt_date)}</p>
                    <hr>
                    <p><strong>Received from:</strong> ${r.member_name} (${r.kso_id})</p>
                    <p><strong>Address:</strong> ${r.address || 'N/A'}</p>
                    <hr>
                    <table class="table">
                        <tr><th>Description</th><th class="text-end">Amount</th></tr>
                        <tr><td>${r.description}</td><td class="text-end">${formatCurrency(r.amount)}</td></tr>
                    </table>
                    <hr>
                    <p class="text-end"><strong>Total: ${formatCurrency(r.amount)}</strong></p>
                `;
                // Add print/download functionality
                document.getElementById('printReceiptBtn').onclick = () => this.printReceipt(content.innerHTML);
                document.getElementById('downloadReceiptBtn').onclick = () => this.downloadReceipt(r.receipt_number, content.innerHTML);
            } else {
                content.innerHTML = `<div class="alert alert-danger">${response.message}</div>`;
            }
        } catch (error) {
            content.innerHTML = `<div class="alert alert-danger">An error occurred: ${error.message}</div>`;
        } finally {
            hideLoading();
        }
    }

    async deleteReceipt(id) {
        if (!confirm('Are you sure you want to delete this receipt? This action cannot be undone.')) return;

        showLoading();
        try {
            const response = await this.api.delete(`receipts?id=${id}`);
            if (response.success) {
                showSuccess('Receipt deleted successfully.');
                this.loadInitialData();
            } else {
                showError(response.message);
            }
        } catch (error) {
            showError('An error occurred: ' + error.message);
        } finally {
            hideLoading();
        }
    }

    printReceipt(content) {
        const printWindow = window.open('', '', 'height=600,width=800');
        printWindow.document.write('<html><head><title>Print Receipt</title>');
        printWindow.document.write('<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">');
        printWindow.document.write('</head><body>');
        printWindow.document.write(content);
        printWindow.document.write('</body></html>');
        printWindow.document.close();
        printWindow.print();
    }

    downloadReceipt(receiptNumber, content) {
        // For a real implementation, use a library like jsPDF
        alert(`Downloading ${receiptNumber}.pdf (feature in development)`);
    }
}

document.addEventListener('DOMContentLoaded', () => {
    window.receiptManager = new ReceiptManager();
});

// Assumes global helper functions from a utility script
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', { style: 'currency', currency: 'INR' }).format(amount || 0);
}
function formatDate(dateString) {
    return new Date(dateString).toLocaleDateString('en-GB');
}
function escapeHtml(str) {
    return str?.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&#039;") || '';
}
function showLoading() { console.log("Loading..."); }
function hideLoading() { console.log("Loading complete."); }
function showError(message) { console.error(message); alert(`Error: ${message}`); }
function showSuccess(message) { console.log(message); }
