// Reports Management JavaScript
class ReportsManager {
    constructor() {
        this.currentCategory = null;
        this.globalFilters = {};
        this.charts = {};
        this.reportData = {};
    }

    // Load report statistics
    async loadReportStats() {
        try {
            const response = await fetch('../api/reports/stats.php');
            const data = await response.json();
            
            if (data.success) {
                // Update member stats
                document.getElementById('totalMembers').textContent = data.data.members.total || 0;
                document.getElementById('activeMembers').textContent = data.data.members.active || 0;
                
                // Update event stats
                document.getElementById('totalEvents').textContent = data.data.events.total || 0;
                document.getElementById('totalRegistrations').textContent = data.data.events.registrations || 0;
                
                // Update financial stats
                document.getElementById('totalIncome').textContent = '₹' + (data.data.finance.income || 0).toLocaleString();
                document.getElementById('totalExpenses').textContent = '₹' + (data.data.finance.expenses || 0).toLocaleString();
                
                // Update membership fees stats
                document.getElementById('paidFees').textContent = '₹' + (data.data.fees.paid || 0).toLocaleString();
                document.getElementById('pendingFees').textContent = '₹' + (data.data.fees.pending || 0).toLocaleString();
                
                // Update ID cards stats
                document.getElementById('activeCards').textContent = data.data.cards.active || 0;
                document.getElementById('expiredCards').textContent = data.data.cards.expired || 0;
                
                // Update custom reports stats
                document.getElementById('savedReports').textContent = data.data.custom.saved || 0;
            }
        } catch (error) {
            console.error('Error loading report stats:', error);
        }
    }

    // Load quick analytics
    async loadQuickAnalytics() {
        try {
            const response = await fetch('../api/reports/analytics.php');
            const data = await response.json();
            
            if (data.success) {
                this.renderMembersGrowthChart(data.data.member_growth);
                this.renderFinancialChart(data.data.financial_overview);
                this.renderEventParticipationChart(data.data.event_participation);
                this.renderMembershipStatusChart(data.data.membership_status);
            }
        } catch (error) {
            console.error('Error loading analytics:', error);
            // Render demo charts
            this.renderDemoCharts();
        }
    }

    // Render demo charts (fallback)
    renderDemoCharts() {
        this.renderMembersGrowthChart([
            { month: 'Jan', count: 45 },
            { month: 'Feb', count: 52 },
            { month: 'Mar', count: 48 },
            { month: 'Apr', count: 63 },
            { month: 'May', count: 71 },
            { month: 'Jun', count: 82 }
        ]);

        this.renderFinancialChart({
            income: [12000, 15000, 18000, 16000, 22000, 25000],
            expenses: [8000, 9500, 11000, 12000, 13500, 15000],
            months: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun']
        });

        this.renderEventParticipationChart([
            { event: 'Annual Meeting', participants: 120 },
            { event: 'Cultural Program', participants: 85 },
            { event: 'Sports Day', participants: 95 },
            { event: 'Festival Celebration', participants: 150 },
            { event: 'Community Service', participants: 65 }
        ]);

        this.renderMembershipStatusChart([
            { status: 'Active', count: 234 },
            { status: 'Pending', count: 45 },
            { status: 'Expired', count: 23 },
            { status: 'Suspended', count: 8 }
        ]);
    }

    // Render Members Growth Chart
    renderMembersGrowthChart(data) {
        const ctx = document.getElementById('membersGrowthChart').getContext('2d');
        
        if (this.charts.membersGrowth) {
            this.charts.membersGrowth.destroy();
        }

        this.charts.membersGrowth = new Chart(ctx, {
            type: 'line',
            data: {
                labels: data.map(item => item.month),
                datasets: [{
                    label: 'New Members',
                    data: data.map(item => item.count),
                    borderColor: 'rgb(75, 192, 192)',
                    backgroundColor: 'rgba(75, 192, 192, 0.2)',
                    tension: 0.1,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }

    // Render Financial Chart
    renderFinancialChart(data) {
        const ctx = document.getElementById('financialChart').getContext('2d');
        
        if (this.charts.financial) {
            this.charts.financial.destroy();
        }

        this.charts.financial = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: data.months,
                datasets: [{
                    label: 'Income',
                    data: data.income,
                    backgroundColor: 'rgba(54, 162, 235, 0.8)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 1
                }, {
                    label: 'Expenses',
                    data: data.expenses,
                    backgroundColor: 'rgba(255, 99, 132, 0.8)',
                    borderColor: 'rgba(255, 99, 132, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₹' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });
    }

    // Render Event Participation Chart
    renderEventParticipationChart(data) {
        const ctx = document.getElementById('eventParticipationChart').getContext('2d');
        
        if (this.charts.eventParticipation) {
            this.charts.eventParticipation.destroy();
        }

        this.charts.eventParticipation = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: data.map(item => item.event),
                datasets: [{
                    data: data.map(item => item.participants),
                    backgroundColor: [
                        'rgba(255, 99, 132, 0.8)',
                        'rgba(54, 162, 235, 0.8)',
                        'rgba(255, 205, 86, 0.8)',
                        'rgba(75, 192, 192, 0.8)',
                        'rgba(153, 102, 255, 0.8)'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    }

    // Render Membership Status Chart
    renderMembershipStatusChart(data) {
        const ctx = document.getElementById('membershipStatusChart').getContext('2d');
        
        if (this.charts.membershipStatus) {
            this.charts.membershipStatus.destroy();
        }

        this.charts.membershipStatus = new Chart(ctx, {
            type: 'pie',
            data: {
                labels: data.map(item => item.status),
                datasets: [{
                    data: data.map(item => item.count),
                    backgroundColor: [
                        'rgba(40, 167, 69, 0.8)',   // Active - Green
                        'rgba(255, 193, 7, 0.8)',   // Pending - Yellow
                        'rgba(220, 53, 69, 0.8)',   // Expired - Red
                        'rgba(108, 117, 125, 0.8)'  // Suspended - Gray
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    }

    // Show report category
    showReportCategory(category) {
        this.currentCategory = category;
        document.getElementById('reportContent').style.display = 'block';
        document.getElementById('reportTitle').textContent = this.getCategoryTitle(category);
        
        this.loadReportOptions(category);
        this.loadCategoryReport(category);
    }

    // Get category title
    getCategoryTitle(category) {
        const titles = {
            'members': 'Members Reports',
            'events': 'Events Reports',
            'finance': 'Financial Reports',
            'membership-fees': 'Membership Fees Reports',
            'id-cards': 'ID Cards Reports',
            'custom': 'Custom Reports'
        };
        return titles[category] || 'Report';
    }

    // Load report options for category
    loadReportOptions(category) {
        const optionsContainer = document.getElementById('reportOptions');
        
        const options = this.getReportOptions(category);
        optionsContainer.innerHTML = options;
    }

    // Get report options HTML
    getReportOptions(category) {
        const commonOptions = `
            <div class="col-md-3">
                <label class="form-label">Report Type</label>
                <select class="form-select" id="reportType" onchange="generateReport()">
                    ${this.getReportTypeOptions(category)}
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">Format</label>
                <select class="form-select" id="reportFormat">
                    <option value="table">Table</option>
                    <option value="chart">Chart</option>
                    <option value="summary">Summary</option>
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">Group By</label>
                <select class="form-select" id="reportGroupBy">
                    ${this.getGroupByOptions(category)}
                </select>
            </div>
            <div class="col-md-3 d-flex align-items-end">
                <button type="button" class="btn btn-primary" onclick="generateReport()">
                    <i class="bi bi-play-fill"></i> Generate Report
                </button>
            </div>
        `;
        
        return commonOptions;
    }

    // Get report type options for category
    getReportTypeOptions(category) {
        const types = {
            'members': `
                <option value="overview">Members Overview</option>
                <option value="demographics">Demographics</option>
                <option value="approvals">Approval Status</option>
                <option value="family">Family Memberships</option>
                <option value="renewals">Renewals & Expirations</option>
            `,
            'events': `
                <option value="overview">Events Overview</option>
                <option value="participation">Participation Analysis</option>
                <option value="registrations">Registration Trends</option>
                <option value="feedback">Event Feedback</option>
                <option value="revenue">Event Revenue</option>
            `,
            'finance': `
                <option value="overview">Financial Overview</option>
                <option value="income">Income Analysis</option>
                <option value="expenses">Expense Analysis</option>
                <option value="donations">Donations Report</option>
                <option value="balance">Balance Sheet</option>
            `,
            'membership-fees': `
                <option value="overview">Fees Overview</option>
                <option value="collection">Collection Status</option>
                <option value="pending">Pending Payments</option>
                <option value="renewals">Renewal Reminders</option>
                <option value="discounts">Discounts & Waivers</option>
            `,
            'id-cards': `
                <option value="overview">Cards Overview</option>
                <option value="issuance">Issuance Report</option>
                <option value="renewals">Renewal Status</option>
                <option value="expired">Expired Cards</option>
                <option value="printing">Print Queue</option>
            `,
            'custom': `
                <option value="template1">Member Activity Report</option>
                <option value="template2">Financial Summary</option>
                <option value="template3">Event Performance</option>
                <option value="template4">Compliance Report</option>
                <option value="custom">Custom Query</option>
            `
        };
        
        return types[category] || '<option value="overview">Overview</option>';
    }

    // Get group by options for category
    getGroupByOptions(category) {
        const groupOptions = {
            'members': `
                <option value="">No Grouping</option>
                <option value="status">Status</option>
                <option value="membership_type">Membership Type</option>
                <option value="month">Month</option>
                <option value="year">Year</option>
            `,
            'events': `
                <option value="">No Grouping</option>
                <option value="event_type">Event Type</option>
                <option value="month">Month</option>
                <option value="year">Year</option>
                <option value="status">Status</option>
            `,
            'finance': `
                <option value="">No Grouping</option>
                <option value="type">Transaction Type</option>
                <option value="month">Month</option>
                <option value="quarter">Quarter</option>
                <option value="year">Year</option>
            `,
            'membership-fees': `
                <option value="">No Grouping</option>
                <option value="status">Payment Status</option>
                <option value="month">Month</option>
                <option value="year">Year</option>
                <option value="fee_type">Fee Type</option>
            `,
            'id-cards': `
                <option value="">No Grouping</option>
                <option value="status">Card Status</option>
                <option value="type">Card Type</option>
                <option value="month">Month</option>
                <option value="year">Year</option>
            `,
            'custom': `
                <option value="">No Grouping</option>
                <option value="month">Month</option>
                <option value="quarter">Quarter</option>
                <option value="year">Year</option>
            `
        };
        
        return groupOptions[category] || '<option value="">No Grouping</option>';
    }

    // Load category report
    async loadCategoryReport(category) {
        try {
            showLoading();
            
            const params = new URLSearchParams({
                category: category,
                ...this.globalFilters
            });

            const response = await fetch(`../api/reports/category.php?${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.displayReportData(data.data);
            } else {
                this.displayNoData();
            }
        } catch (error) {
            console.error('Error loading category report:', error);
            this.displayError('Failed to load report data');
        } finally {
            hideLoading();
        }
    }

    // Generate specific report
    async generateReport() {
        if (!this.currentCategory) return;
        
        try {
            showLoading();
            
            const reportType = document.getElementById('reportType').value;
            const reportFormat = document.getElementById('reportFormat').value;
            const groupBy = document.getElementById('reportGroupBy').value;
            
            const params = new URLSearchParams({
                category: this.currentCategory,
                type: reportType,
                format: reportFormat,
                group_by: groupBy,
                ...this.globalFilters
            });

            const response = await fetch(`../api/reports/generate.php?${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.displayReportData(data.data, reportFormat);
                
                // Show chart if format is chart
                if (reportFormat === 'chart') {
                    this.showChartSection(data.data);
                } else {
                    this.hideChartSection();
                }
            } else {
                this.displayError('Error generating report: ' + data.message);
            }
        } catch (error) {
            console.error('Error generating report:', error);
            this.displayError('Failed to generate report');
        } finally {
            hideLoading();
        }
    }

    // Display report data
    displayReportData(data, format = 'table') {
        const container = document.getElementById('reportData');
        
        if (format === 'table') {
            container.innerHTML = this.generateTableHTML(data);
        } else if (format === 'summary') {
            container.innerHTML = this.generateSummaryHTML(data);
        } else {
            container.innerHTML = this.generateChartPlaceholder();
        }
    }

    // Generate table HTML
    generateTableHTML(data) {
        if (!data.rows || data.rows.length === 0) {
            return '<p class="text-center text-muted">No data available for the selected criteria.</p>';
        }

        const headers = data.headers || Object.keys(data.rows[0]);
        
        let html = `
            <div class="table-responsive">
                <table class="table table-striped table-bordered">
                    <thead class="table-dark">
                        <tr>
                            ${headers.map(header => `<th>${header}</th>`).join('')}
                        </tr>
                    </thead>
                    <tbody>
        `;
        
        data.rows.forEach(row => {
            html += '<tr>';
            headers.forEach(header => {
                html += `<td>${row[header] || ''}</td>`;
            });
            html += '</tr>';
        });
        
        html += `
                    </tbody>
                </table>
            </div>
        `;
        
        return html;
    }

    // Generate summary HTML
    generateSummaryHTML(data) {
        if (!data.summary) {
            return '<p class="text-center text-muted">No summary data available.</p>';
        }

        let html = '<div class="row">';
        
        Object.entries(data.summary).forEach(([key, value]) => {
            html += `
                <div class="col-md-3 col-sm-6">
                    <div class="card text-center">
                        <div class="card-body">
                            <h4 class="card-title">${value}</h4>
                            <p class="card-text">${key.replace(/_/g, ' ').toUpperCase()}</p>
                        </div>
                    </div>
                </div>
            `;
        });
        
        html += '</div>';
        return html;
    }

    // Generate chart placeholder
    generateChartPlaceholder() {
        return '<p class="text-center text-muted">Chart will be displayed above.</p>';
    }

    // Show chart section
    showChartSection(data) {
        document.getElementById('chartSection').style.display = 'block';
        this.renderReportChart(data);
    }

    // Hide chart section
    hideChartSection() {
        document.getElementById('chartSection').style.display = 'none';
    }

    // Render report chart
    renderReportChart(data) {
        const ctx = document.getElementById('reportChart').getContext('2d');
        
        if (this.charts.report) {
            this.charts.report.destroy();
        }

        // Determine chart type based on data
        const chartType = data.chart_type || 'bar';
        
        this.charts.report = new Chart(ctx, {
            type: chartType,
            data: data.chart_data,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true
                    }
                }
            }
        });
    }

    // Display no data message
    displayNoData() {
        document.getElementById('reportData').innerHTML = 
            '<p class="text-center text-muted">No data available for the selected period.</p>';
    }

    // Display error message
    displayError(message) {
        document.getElementById('reportData').innerHTML = 
            `<div class="alert alert-danger">${message}</div>`;
    }

    // Hide report content
    hideReportContent() {
        document.getElementById('reportContent').style.display = 'none';
        this.currentCategory = null;
    }

    // Set quick period
    setQuickPeriod() {
        const period = document.getElementById('reportPeriod').value;
        const today = new Date();
        let fromDate, toDate;

        switch (period) {
            case 'today':
                fromDate = toDate = today.toISOString().split('T')[0];
                break;
            case 'yesterday':
                const yesterday = new Date(today);
                yesterday.setDate(yesterday.getDate() - 1);
                fromDate = toDate = yesterday.toISOString().split('T')[0];
                break;
            case 'this_week':
                const startOfWeek = new Date(today);
                startOfWeek.setDate(today.getDate() - today.getDay());
                fromDate = startOfWeek.toISOString().split('T')[0];
                toDate = today.toISOString().split('T')[0];
                break;
            case 'last_week':
                const startOfLastWeek = new Date(today);
                startOfLastWeek.setDate(today.getDate() - today.getDay() - 7);
                const endOfLastWeek = new Date(startOfLastWeek);
                endOfLastWeek.setDate(startOfLastWeek.getDate() + 6);
                fromDate = startOfLastWeek.toISOString().split('T')[0];
                toDate = endOfLastWeek.toISOString().split('T')[0];
                break;
            case 'this_month':
                fromDate = new Date(today.getFullYear(), today.getMonth(), 1).toISOString().split('T')[0];
                toDate = today.toISOString().split('T')[0];
                break;
            case 'last_month':
                const lastMonth = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                const endOfLastMonth = new Date(today.getFullYear(), today.getMonth(), 0);
                fromDate = lastMonth.toISOString().split('T')[0];
                toDate = endOfLastMonth.toISOString().split('T')[0];
                break;
            case 'this_year':
                fromDate = new Date(today.getFullYear(), 0, 1).toISOString().split('T')[0];
                toDate = today.toISOString().split('T')[0];
                break;
            case 'last_year':
                fromDate = new Date(today.getFullYear() - 1, 0, 1).toISOString().split('T')[0];
                toDate = new Date(today.getFullYear() - 1, 11, 31).toISOString().split('T')[0];
                break;
            default:
                return; // Custom period, don't change dates
        }

        if (fromDate && toDate) {
            document.getElementById('reportDateFrom').value = fromDate;
            document.getElementById('reportDateTo').value = toDate;
        }
    }

    // Apply global filters
    applyGlobalFilters() {
        const fromDate = document.getElementById('reportDateFrom').value;
        const toDate = document.getElementById('reportDateTo').value;

        this.globalFilters = {};
        if (fromDate) this.globalFilters.from_date = fromDate;
        if (toDate) this.globalFilters.to_date = toDate;

        // Reload current report if one is open
        if (this.currentCategory) {
            this.loadCategoryReport(this.currentCategory);
        }

        // Reload quick analytics
        this.loadQuickAnalytics();
        
        showSuccess('Filters applied successfully');
    }

    // Clear global filters
    clearGlobalFilters() {
        document.getElementById('reportDateFrom').value = '';
        document.getElementById('reportDateTo').value = '';
        document.getElementById('reportPeriod').value = '';
        
        this.globalFilters = {};
        
        // Reload current report if one is open
        if (this.currentCategory) {
            this.loadCategoryReport(this.currentCategory);
        }

        // Reload quick analytics
        this.loadQuickAnalytics();
        
        showSuccess('Filters cleared successfully');
    }

    // Export report
    exportReport(format) {
        if (!this.currentCategory) {
            showWarning('Please generate a report first');
            return;
        }

        showInfo(`Export to ${format.toUpperCase()} functionality will be implemented`);
    }

    // Print report
    printReport() {
        if (!this.currentCategory) {
            showWarning('Please generate a report first');
            return;
        }

        window.print();
    }
}

// Global instance
const reportsManager = new ReportsManager();

// Global functions for HTML onclick events
function loadReportStats() {
    reportsManager.loadReportStats();
}

function loadQuickAnalytics() {
    reportsManager.loadQuickAnalytics();
}

function showReportCategory(category) {
    reportsManager.showReportCategory(category);
}

function generateReport() {
    reportsManager.generateReport();
}

function hideReportContent() {
    reportsManager.hideReportContent();
}

function setQuickPeriod() {
    reportsManager.setQuickPeriod();
}

function applyGlobalFilters() {
    reportsManager.applyGlobalFilters();
}

function clearGlobalFilters() {
    reportsManager.clearGlobalFilters();
}

function exportReport(format) {
    reportsManager.exportReport(format);
}

function printReport() {
    reportsManager.printReport();
}
