/**
 * KSO Chandigarh - Settings Management
 * Enhanced Settings functionality with DataTables
 */

class SettingsManager {
    constructor() {
        this.currentSection = 'general';
        this.settingsData = {};
        this.init();
    }

    init() {
        this.loadAllSettings();
        this.initializeEventListeners();
        this.checkEmailServiceStatus();
        this.loadSystemInfo();
        this.loadBackupHistory();
        this.loadEmailTemplates();
        this.loadLogs();
    }

    initializeEventListeners() {
        // Section navigation
        document.querySelectorAll('.list-group-item-action').forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const section = e.target.getAttribute('onclick').match(/'(\w+)'/)[1];
                this.showSettingsSection(section);
            });
        });

        // Auto-save on input changes
        document.addEventListener('change', (e) => {
            if (e.target.matches('input, select, textarea')) {
                const section = this.getCurrentSectionFromElement(e.target);
                if (section) {
                    this.autoSaveSettings(section, e.target);
                }
            }
        });

        // Email service toggle
        document.getElementById('emailEnabled')?.addEventListener('change', (e) => {
            this.toggleEmailService(e.target.checked);
        });
    }

    async loadAllSettings() {
        try {
            const response = await fetch('/access/admin/api/settings/get.php');
            const data = await response.json();
            
            if (data.success) {
                this.settingsData = data.settings;
                this.populateSettingsForms();
            } else {
                this.showError('Failed to load settings: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading settings:', error);
            this.showError('Failed to load settings');
        }
    }

    populateSettingsForms() {
        // General Settings
        this.setFieldValue('systemName', this.settingsData.organization_name || 'KSO Chandigarh Management System');
        this.setFieldValue('timezone', this.settingsData.timezone || 'Asia/Kolkata');
        this.setFieldValue('dateFormat', this.settingsData.date_format || 'd/m/Y');
        this.setFieldValue('currency', this.settingsData.currency || 'INR');
        this.setFieldValue('language', this.settingsData.language || 'en');
        this.setCheckbox('maintenanceMode', this.settingsData.maintenance_mode === '1');

        // Organization Settings
        this.setFieldValue('orgName', this.settingsData.organization_name || '');
        this.setFieldValue('orgShortName', this.settingsData.organization_short_name || 'KSO');
        this.setFieldValue('orgDescription', this.settingsData.organization_description || '');
        this.setFieldValue('orgAddress', this.settingsData.organization_address || '');
        this.setFieldValue('orgPhone', this.settingsData.organization_phone || '');
        this.setFieldValue('orgEmail', this.settingsData.organization_email || '');
        this.setFieldValue('orgWebsite', this.settingsData.organization_website || '');
        this.setFieldValue('orgEstablished', this.settingsData.organization_established || '');

        // Email Settings
        this.setFieldValue('smtpHost', this.settingsData.smtp_host || '');
        this.setFieldValue('smtpPort', this.settingsData.smtp_port || '587');
        this.setFieldValue('smtpUsername', this.settingsData.smtp_username || '');
        this.setFieldValue('smtpPassword', this.settingsData.smtp_password || '');
        this.setFieldValue('smtpEncryption', this.settingsData.smtp_encryption || 'tls');
        this.setFieldValue('emailFromAddress', this.settingsData.email_from_address || '');
        this.setFieldValue('emailFromName', this.settingsData.email_from_name || '');
        this.setCheckbox('emailEnabled', this.settingsData.email_enabled === '1');
        
        // Email automation settings
        this.setCheckbox('emailAutoWelcome', this.settingsData.email_auto_welcome === '1');
        this.setCheckbox('emailAutoApproval', this.settingsData.email_auto_approval === '1');
        this.setCheckbox('emailAutoEvents', this.settingsData.email_auto_events === '1');
        this.setCheckbox('emailAutoFees', this.settingsData.email_auto_fees === '1');
    }

    setFieldValue(fieldId, value) {
        const field = document.getElementById(fieldId);
        if (field) {
            field.value = value || '';
        }
    }

    setCheckbox(fieldId, checked) {
        const field = document.getElementById(fieldId);
        if (field) {
            field.checked = checked;
        }
    }

    showSettingsSection(section) {
        // Hide all sections
        document.querySelectorAll('.settings-section').forEach(section => {
            section.style.display = 'none';
        });

        // Remove active class from all navigation items
        document.querySelectorAll('.list-group-item-action').forEach(item => {
            item.classList.remove('active');
        });

        // Show selected section
        const sectionElement = document.getElementById(section + 'Settings');
        if (sectionElement) {
            sectionElement.style.display = 'block';
        }

        // Add active class to current navigation item
        const navItem = document.querySelector(`[onclick="showSettingsSection('${section}')"]`);
        if (navItem) {
            navItem.classList.add('active');
        }

        this.currentSection = section;

        // Load section-specific data
        switch (section) {
            case 'email':
                this.refreshEmailStats();
                break;
            case 'system':
                this.refreshSystemInfo();
                break;
            case 'backup':
                this.loadBackupHistory();
                break;
            case 'logs':
                this.loadLogs();
                break;
        }
    }

    async saveGeneralSettings() {
        const formData = {
            organization_name: document.getElementById('systemName').value,
            timezone: document.getElementById('timezone').value,
            date_format: document.getElementById('dateFormat').value,
            currency: document.getElementById('currency').value,
            language: document.getElementById('language').value,
            maintenance_mode: document.getElementById('maintenanceMode').checked ? '1' : '0'
        };

        await this.saveSettings(formData, 'General settings saved successfully!');
    }

    async saveOrganizationSettings() {
        const formData = {
            organization_name: document.getElementById('orgName').value,
            organization_short_name: document.getElementById('orgShortName').value,
            organization_description: document.getElementById('orgDescription').value,
            organization_address: document.getElementById('orgAddress').value,
            organization_phone: document.getElementById('orgPhone').value,
            organization_email: document.getElementById('orgEmail').value,
            organization_website: document.getElementById('orgWebsite').value,
            organization_established: document.getElementById('orgEstablished').value
        };

        await this.saveSettings(formData, 'Organization settings saved successfully!');
    }

    async saveEmailSettings() {
        const formData = {
            smtp_host: document.getElementById('smtpHost').value,
            smtp_port: document.getElementById('smtpPort').value,
            smtp_username: document.getElementById('smtpUsername').value,
            smtp_password: document.getElementById('smtpPassword').value,
            smtp_encryption: document.getElementById('smtpEncryption').value,
            email_from_address: document.getElementById('emailFromAddress').value,
            email_from_name: document.getElementById('emailFromName').value,
            email_enabled: document.getElementById('emailEnabled').checked ? '1' : '0',
            email_auto_welcome: document.getElementById('emailAutoWelcome').checked ? '1' : '0',
            email_auto_approval: document.getElementById('emailAutoApproval').checked ? '1' : '0',
            email_auto_events: document.getElementById('emailAutoEvents').checked ? '1' : '0',
            email_auto_fees: document.getElementById('emailAutoFees').checked ? '1' : '0'
        };

        await this.saveSettings(formData, 'Email settings saved successfully!');
        this.checkEmailServiceStatus();
    }

    async saveSettings(formData, successMessage) {
        try {
            const response = await fetch('/access/admin/api/settings/save.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(formData)
            });

            const data = await response.json();
            
            if (data.success) {
                this.showSuccess(successMessage);
                // Update local settings data
                Object.assign(this.settingsData, formData);
            } else {
                this.showError('Failed to save settings: ' + data.message);
            }
        } catch (error) {
            console.error('Error saving settings:', error);
            this.showError('Failed to save settings');
        }
    }

    async checkEmailServiceStatus() {
        try {
            const response = await fetch('/access/admin/api/settings/email-status.php');
            const data = await response.json();
            
            const statusElement = document.getElementById('emailServiceStatus');
            if (statusElement) {
                if (data.success && data.status === 'working') {
                    statusElement.textContent = 'Working';
                    statusElement.className = 'badge bg-success';
                } else {
                    statusElement.textContent = 'Not Configured';
                    statusElement.className = 'badge bg-warning';
                }
            }
        } catch (error) {
            console.error('Error checking email status:', error);
            const statusElement = document.getElementById('emailServiceStatus');
            if (statusElement) {
                statusElement.textContent = 'Error';
                statusElement.className = 'badge bg-danger';
            }
        }
    }

    async sendTestEmail() {
        const testEmail = document.getElementById('testEmailAddress').value;
        if (!testEmail) {
            alert('Please enter a test email address');
            return;
        }

        try {
            const response = await fetch('/access/admin/api/settings/test-email.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ email: testEmail })
            });

            const data = await response.json();
            
            if (data.success) {
                this.showSuccess('Test email sent successfully!');
            } else {
                this.showError('Failed to send test email: ' + data.message);
            }
        } catch (error) {
            console.error('Error sending test email:', error);
            this.showError('Failed to send test email');
        }
    }

    async refreshSystemInfo() {
        try {
            const response = await fetch('/access/admin/api/settings/system-info.php');
            const data = await response.json();
            
            if (data.success) {
                const info = data.info;
                document.getElementById('serverOS').textContent = info.server_os || 'Unknown';
                document.getElementById('phpVersion').textContent = info.php_version || 'Unknown';
                document.getElementById('mysqlVersion').textContent = info.mysql_version || 'Unknown';
                document.getElementById('webServer').textContent = info.web_server || 'Unknown';
                document.getElementById('dbSize').textContent = info.db_size || 'Unknown';
                document.getElementById('totalTables').textContent = info.total_tables || 'Unknown';
                document.getElementById('totalRecords').textContent = info.total_records || 'Unknown';
                document.getElementById('lastBackup').textContent = info.last_backup || 'Never';

                // Update progress bars
                if (info.disk_usage) {
                    const diskBar = document.getElementById('diskUsage');
                    diskBar.style.width = info.disk_usage + '%';
                    diskBar.textContent = info.disk_usage + '%';
                }

                if (info.memory_usage) {
                    const memoryBar = document.getElementById('memoryUsage');
                    memoryBar.style.width = info.memory_usage + '%';
                    memoryBar.textContent = info.memory_usage + '%';
                }
            }
        } catch (error) {
            console.error('Error loading system info:', error);
        }
    }

    async refreshEmailStats() {
        try {
            const response = await fetch('/access/admin/api/settings/email-stats.php');
            const data = await response.json();
            
            if (data.success) {
                const stats = data.stats;
                document.getElementById('emailsSentCount').textContent = stats.sent || 0;
                document.getElementById('emailsFailedCount').textContent = stats.failed || 0;
                document.getElementById('emailsThisWeekCount').textContent = stats.this_week || 0;
                document.getElementById('emailSuccessRate').textContent = (stats.success_rate || 0) + '%';
            }
        } catch (error) {
            console.error('Error loading email stats:', error);
        }
    }

    async loadEmailTemplates() {
        try {
            const response = await fetch('/access/admin/api/settings/email-templates.php');
            const data = await response.json();
            
            if (data.success && data.templates) {
                const tableBody = document.getElementById('emailTemplatesTable');
                if (tableBody) {
                    tableBody.innerHTML = data.templates.map(template => `
                        <tr>
                            <td>${template.name}</td>
                            <td>${template.purpose}</td>
                            <td><span class="badge ${template.status === 'active' ? 'bg-success' : 'bg-secondary'}">${template.status}</span></td>
                            <td>
                                <button class="btn btn-sm btn-outline-primary" onclick="editEmailTemplate('${template.id}')">
                                    <i class="bi bi-pencil"></i> Edit
                                </button>
                            </td>
                        </tr>
                    `).join('');
                }
            }
        } catch (error) {
            console.error('Error loading email templates:', error);
        }
    }

    async loadBackupHistory() {
        try {
            const response = await fetch('/access/admin/api/settings/backup-history.php');
            const data = await response.json();
            
            if (data.success && data.backups) {
                const tableBody = document.getElementById('backupHistory');
                if (tableBody) {
                    tableBody.innerHTML = data.backups.map(backup => `
                        <tr>
                            <td>${this.formatDateTime(backup.created_at)}</td>
                            <td>${backup.type}</td>
                            <td>${this.formatFileSize(backup.size)}</td>
                            <td><span class="badge ${backup.status === 'completed' ? 'bg-success' : 'bg-danger'}">${backup.status}</span></td>
                            <td>
                                <button class="btn btn-sm btn-outline-primary" onclick="downloadBackup('${backup.id}')">
                                    <i class="bi bi-download"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger" onclick="deleteBackup('${backup.id}')">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </td>
                        </tr>
                    `).join('');
                }
            }
        } catch (error) {
            console.error('Error loading backup history:', error);
        }
    }

    async loadLogs() {
        try {
            const response = await fetch('/access/admin/api/settings/logs.php');
            const data = await response.json();
            
            if (data.success && data.logs) {
                const container = document.getElementById('logsContainer');
                if (container) {
                    container.innerHTML = data.logs.map(log => `
                        <div class="log-entry ${log.level}">
                            <span class="log-timestamp">${this.formatDateTime(log.timestamp)}</span>
                            <span class="log-level badge ${this.getLogLevelClass(log.level)}">${log.level}</span>
                            <span class="log-message">${log.message}</span>
                        </div>
                    `).join('');
                }
            }
        } catch (error) {
            console.error('Error loading logs:', error);
        }
    }

    getLogLevelClass(level) {
        const classes = {
            'error': 'bg-danger',
            'warning': 'bg-warning',
            'info': 'bg-info',
            'debug': 'bg-secondary'
        };
        return classes[level] || 'bg-light';
    }

    formatDateTime(dateString) {
        if (!dateString) return '-';
        const date = new Date(dateString);
        return date.toLocaleString();
    }

    formatFileSize(bytes) {
        if (!bytes) return '0 B';
        const k = 1024;
        const sizes = ['B', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    showSuccess(message) {
        // Implement toast notification or alert
        console.log('Success:', message);
        // You can use a toast library here
        alert(message);
    }

    showError(message) {
        // Implement toast notification or alert
        console.error('Error:', message);
        alert(message);
    }

    getCurrentSectionFromElement(element) {
        const form = element.closest('form');
        if (form) {
            if (form.id.includes('general')) return 'general';
            if (form.id.includes('organization')) return 'organization';
            if (form.id.includes('email')) return 'email';
        }
        return null;
    }

    async autoSaveSettings(section, element) {
        // Implement auto-save functionality if needed
        console.log('Auto-saving', section, element.id, element.value);
    }

    toggleEmailService(enabled) {
        // Enable/disable email-related form fields
        const emailFields = document.querySelectorAll('#emailSettings input, #emailSettings select');
        emailFields.forEach(field => {
            if (field.id !== 'emailEnabled') {
                field.disabled = !enabled;
            }
        });
    }
}

// Global functions for onclick handlers
function showSettingsSection(section) {
    if (window.settingsManager) {
        window.settingsManager.showSettingsSection(section);
    }
}

function saveGeneralSettings() {
    if (window.settingsManager) {
        window.settingsManager.saveGeneralSettings();
    }
}

function saveOrganizationSettings() {
    if (window.settingsManager) {
        window.settingsManager.saveOrganizationSettings();
    }
}

function saveEmailSettings() {
    if (window.settingsManager) {
        window.settingsManager.saveEmailSettings();
    }
}

function resetEmailSettings() {
    if (confirm('Are you sure you want to reset email settings to defaults?')) {
        // Reset email form
        document.getElementById('emailSettingsForm').reset();
    }
}

function sendTestEmail() {
    if (window.settingsManager) {
        window.settingsManager.sendTestEmail();
    }
}

function refreshEmailStats() {
    if (window.settingsManager) {
        window.settingsManager.refreshEmailStats();
    }
}

function refreshSystemInfo() {
    if (window.settingsManager) {
        window.settingsManager.refreshSystemInfo();
    }
}

function downloadSystemReport() {
    window.open('/access/admin/api/settings/system-report.php', '_blank');
}

function createBackup() {
    const backupType = document.querySelector('input[name="backupType"]:checked').value;
    if (confirm(`Are you sure you want to create a ${backupType} backup?`)) {
        window.open(`/access/admin/api/settings/create-backup.php?type=${backupType}`, '_blank');
    }
}

function restoreBackup() {
    const fileInput = document.getElementById('backupFile');
    if (!fileInput.files[0]) {
        alert('Please select a backup file first.');
        return;
    }
    
    if (confirm('WARNING: Restoring a backup will overwrite current data. Are you sure?')) {
        // Implement restore functionality
        console.log('Restoring backup...');
    }
}

// Additional settings functions
function saveMembershipSettings() {
    if (window.settingsManager) {
        const formData = {
            member_approval_enabled: document.getElementById('memberApprovalEnabled').checked ? '1' : '0',
            member_auto_assignment: document.getElementById('memberAutoAssignment').checked ? '1' : '0',
            member_approval_timeout_days: document.getElementById('approvalTimeoutDays').value,
            max_pending_members: document.getElementById('maxPendingMembers').value
        };
        window.settingsManager.saveSettings(formData, 'Membership settings saved successfully!');
    }
}

function addMembershipType() {
    // Implementation for adding new membership type
    console.log('Adding new membership type...');
}

function saveFeeSettings() {
    if (window.settingsManager) {
        const formData = {
            fee_reminder_days: document.getElementById('feeReminderDays').value,
            late_fee_grace_days: document.getElementById('lateFeeGraceDays').value,
            auto_fee_reminders: document.getElementById('autoFeeReminders').checked ? '1' : '0',
            auto_late_fees: document.getElementById('autoLateFees').checked ? '1' : '0',
            payment_cash: document.getElementById('paymentCash').checked ? '1' : '0',
            payment_upi: document.getElementById('paymentUPI').checked ? '1' : '0',
            payment_bank: document.getElementById('paymentBank').checked ? '1' : '0',
            upi_id: document.getElementById('upiId').value,
            bank_account: document.getElementById('bankAccount').value
        };
        window.settingsManager.saveSettings(formData, 'Fee settings saved successfully!');
    }
}

function generateFeeReport() {
    window.open('/access/admin/api/settings/fee-report.php', '_blank');
}

function saveNotificationSettings() {
    if (window.settingsManager) {
        const formData = {
            notification_system_enabled: document.getElementById('notificationSystemEnabled').checked ? '1' : '0',
            notification_sound_enabled: document.getElementById('notificationSoundEnabled').checked ? '1' : '0',
            notification_cleanup_days: document.getElementById('notificationCleanupDays').value,
            notification_max_per_user: document.getElementById('notificationMaxPerUser').value,
            email_notify_member_approval: document.getElementById('emailNotifyMemberApproval').checked ? '1' : '0',
            email_notify_event_reminders: document.getElementById('emailNotifyEventReminders').checked ? '1' : '0',
            email_notify_fee_reminders: document.getElementById('emailNotifyFeeReminders').checked ? '1' : '0',
            email_notify_welcome: document.getElementById('emailNotifyWelcome').checked ? '1' : '0',
            email_notify_documents: document.getElementById('emailNotifyDocuments').checked ? '1' : '0',
            email_notify_system: document.getElementById('emailNotifySystem').checked ? '1' : '0',
            realtime_new_members: document.getElementById('realtimeNewMembers').checked ? '1' : '0',
            realtime_payments: document.getElementById('realtimePayments').checked ? '1' : '0',
            realtime_event_registrations: document.getElementById('realtimeEventRegistrations').checked ? '1' : '0',
            realtime_system_alerts: document.getElementById('realtimeSystemAlerts').checked ? '1' : '0'
        };
        window.settingsManager.saveSettings(formData, 'Notification settings saved successfully!');
    }
}

function testNotification() {
    if (window.settingsManager) {
        // Send a test notification
        fetch('/access/admin/api/settings/test-notification.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ message: 'This is a test notification from the settings panel.' })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Test notification sent successfully!');
            } else {
                alert('Failed to send test notification: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error sending test notification:', error);
            alert('Failed to send test notification');
        });
    }
}

function saveSecuritySettings() {
    if (window.settingsManager) {
        const formData = {
            min_password_length: document.getElementById('minPasswordLength').value,
            password_expire_days: document.getElementById('passwordExpireDays').value,
            require_uppercase: document.getElementById('requireUppercase').checked ? '1' : '0',
            require_lowercase: document.getElementById('requireLowercase').checked ? '1' : '0',
            require_numbers: document.getElementById('requireNumbers').checked ? '1' : '0',
            require_special_chars: document.getElementById('requireSpecialChars').checked ? '1' : '0',
            session_timeout: document.getElementById('sessionTimeout').value,
            max_login_attempts: document.getElementById('maxLoginAttempts').value,
            lockout_duration: document.getElementById('lockoutDuration').value,
            enable_two_factor: document.getElementById('enableTwoFactor').checked ? '1' : '0',
            log_login_attempts: document.getElementById('logLoginAttempts').checked ? '1' : '0',
            alert_suspicious_activity: document.getElementById('alertSuspiciousActivity').checked ? '1' : '0',
            allowed_ips: document.getElementById('allowedIPs').value
        };
        window.settingsManager.saveSettings(formData, 'Security settings saved successfully!');
    }
}

function viewSecurityLogs() {
    window.open('/access/admin/api/settings/security-logs.php', '_blank');
}

function filterLogs() {
    if (window.settingsManager) {
        window.settingsManager.loadLogs();
    }
}

function exportLogs() {
    const logType = document.getElementById('logType').value;
    const logLevel = document.getElementById('logLevel').value;
    const logDate = document.getElementById('logDate').value;
    
    const params = new URLSearchParams({
        type: logType,
        level: logLevel,
        date: logDate
    });
    
    window.open(`/access/admin/api/settings/export-logs.php?${params.toString()}`, '_blank');
}

function refreshLogs() {
    if (window.settingsManager) {
        window.settingsManager.loadLogs();
    }
}

function clearLogs() {
    if (confirm('Are you sure you want to clear all logs? This action cannot be undone.')) {
        fetch('/access/admin/api/settings/clear-logs.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Logs cleared successfully!');
                if (window.settingsManager) {
                    window.settingsManager.loadLogs();
                }
            } else {
                alert('Failed to clear logs: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error clearing logs:', error);
            alert('Failed to clear logs');
        });
    }
}

function editEmailTemplate(templateId) {
    // Implementation for editing email template
    console.log('Editing email template:', templateId);
    // You can implement a modal or redirect to edit page
}

function downloadBackup(backupId) {
    window.open(`/access/admin/api/settings/download-backup.php?id=${backupId}`, '_blank');
}

function deleteBackup(backupId) {
    if (confirm('Are you sure you want to delete this backup? This action cannot be undone.')) {
        fetch('/access/admin/api/settings/delete-backup.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ id: backupId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Backup deleted successfully!');
                if (window.settingsManager) {
                    window.settingsManager.loadBackupHistory();
                }
            } else {
                alert('Failed to delete backup: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error deleting backup:', error);
            alert('Failed to delete backup');
        });
    }
}

// Initialize settings manager when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    window.settingsManager = new SettingsManager();
});
