<?php
// Define API access constant
define('API_ACCESS', true);

require_once 'config.php';

// Require admin authentication for dashboard
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            getDashboardStats();
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    error_log("Error in dashboard.php: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

function getDashboardStats() {
    global $pdo;
    
    try {
        // Compile all statistics by calling modular functions
        $stats = [
            'members' => _getMemberStats($pdo),
            'events' => _getEventStats($pdo),
            'finance' => _getFinancialStats($pdo),
            'fees' => _getFeeStats($pdo),
            'recent_activities' => _getRecentActivities($pdo),
            'recent_members' => _getRecentMembers($pdo),
            'monthly_data' => _getMonthlyChartData($pdo)
        ];
        
        sendResponse(200, true, 'Dashboard statistics retrieved successfully', $stats);
        
    } catch (Exception $e) {
        error_log("Error getting dashboard stats: " . $e->getMessage());
        sendResponse(500, false, 'Failed to retrieve dashboard statistics: ' . $e->getMessage());
    }
}

/**
 * Helper function to get member statistics
 */
function _getMemberStats($pdo) {
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_members,
            SUM(CASE WHEN approval_status = 'approved' THEN 1 ELSE 0 END) as approved_members,
            SUM(CASE WHEN approval_status = 'pending' THEN 1 ELSE 0 END) as pending_approvals,
            SUM(CASE WHEN membership_type = 'Family' THEN 1 ELSE 0 END) as family_memberships,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_members,
            SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as approved_this_month
        FROM members
    ");
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return [
        'total' => (int)($result['total_members'] ?? 0),
        'approved' => (int)($result['approved_members'] ?? 0),
        'pending_approvals' => (int)($result['pending_approvals'] ?? 0),
        'family_memberships' => (int)($result['family_memberships'] ?? 0),
        'active' => (int)($result['active_members'] ?? 0),
        'approved_this_month' => (int)($result['approved_this_month'] ?? 0)
    ];
}

/**
 * Helper function to get event statistics
 */
function _getEventStats($pdo) {
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_events,
            SUM(CASE WHEN status IN ('upcoming', 'ongoing') AND event_date >= CURDATE() THEN 1 ELSE 0 END) as active_events,
            SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as events_this_month
        FROM events
    ");
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return [
        'total' => (int)($result['total_events'] ?? 0),
        'active' => (int)($result['active_events'] ?? 0),
        'this_month' => (int)($result['events_this_month'] ?? 0)
    ];
}

/**
 * Helper function to get financial statistics (income and expenses)
 */
function _getFinancialStats($pdo) {
    // Income
    $stmt = $pdo->prepare("
        SELECT 
            COALESCE(SUM(amount), 0) as total_income,
            COALESCE(SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN amount ELSE 0 END), 0) as income_this_month
        FROM finance
        WHERE status = 'completed'
    ");
    $stmt->execute();
    $incomeStats = $stmt->fetch(PDO::FETCH_ASSOC);

    // Expenses
    $stmt = $pdo->prepare("
        SELECT 
            COALESCE(SUM(amount), 0) as total_expenses,
            COALESCE(SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN amount ELSE 0 END), 0) as expenses_this_month
        FROM expenses
        WHERE status IN ('approved', 'paid')
    ");
    $stmt->execute();
    $expenseStats = $stmt->fetch(PDO::FETCH_ASSOC);

    $totalIncome = (float)($incomeStats['total_income'] ?? 0);
    $totalExpenses = (float)($expenseStats['total_expenses'] ?? 0);

    return [
        'total_income' => $totalIncome,
        'total_expenses' => $totalExpenses,
        'total_revenue' => $totalIncome - $totalExpenses,
        'income_this_month' => (float)($incomeStats['income_this_month'] ?? 0),
        'expenses_this_month' => (float)($expenseStats['expenses_this_month'] ?? 0)
    ];
}

/**
 * Helper function to get membership fee statistics
 */
function _getFeeStats($pdo) {
    $stmt = $pdo->prepare("
        SELECT 
            COALESCE(SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END), 0) as total_fees_collected,
            COALESCE(SUM(CASE WHEN status = 'pending' THEN amount ELSE 0 END), 0) as pending_fees,
            COALESCE(SUM(CASE WHEN status = 'failed' THEN amount ELSE 0 END), 0) as overdue_fees
        FROM finance
        WHERE transaction_type = 'membership_fee'
    ");
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return [
        'total_collected' => (float)($result['total_fees_collected'] ?? 0),
        'pending' => (float)($result['pending_fees'] ?? 0),
        'overdue' => (float)($result['overdue_fees'] ?? 0)
    ];
}

/**
 * Helper function to get recent activities
 */
function _getRecentActivities($pdo) {
    $stmt = $pdo->prepare("
        (SELECT 'member' as type, CONCAT(first_name, ' ', last_name) as title, 'New member registration' as description, created_at as date FROM members ORDER BY created_at DESC LIMIT 5)
        UNION ALL
        (SELECT 'event' as type, event_title as title, 'New event created' as description, created_at as date FROM events ORDER BY created_at DESC LIMIT 5)
        UNION ALL
        (SELECT 'finance' as type, description as title, CONCAT(transaction_type, ' transaction') as description, created_at as date FROM finance ORDER BY created_at DESC LIMIT 5)
        UNION ALL
        (SELECT 'expense' as type, expense_title as title, 'New expense recorded' as description, created_at as date FROM expenses ORDER BY created_at DESC LIMIT 5)
        ORDER BY date DESC 
        LIMIT 10
    ");
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Helper function to get recent members
 */
function _getRecentMembers($pdo) {
    $stmt = $pdo->prepare("
        SELECT id, kso_id, first_name, last_name, email, phone, approval_status, status, created_at, photo_url
        FROM members 
        ORDER BY created_at DESC 
        LIMIT 8
    ");
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Helper function to get monthly data for charts
 */
function _getMonthlyChartData($pdo) {
    $monthlyData = [];
    for ($i = 11; $i >= 0; $i--) {
        $month = date('Y-m', strtotime("-$i months"));
        $monthName = date('M Y', strtotime("-$i months"));
        
        // Members joined
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM members WHERE DATE_FORMAT(created_at, '%Y-%m') = ?");
        $stmt->execute([$month]);
        $memberCount = $stmt->fetchColumn();
        
        // Revenue
        $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount), 0) FROM finance WHERE status = 'completed' AND DATE_FORMAT(created_at, '%Y-%m') = ?");
        $stmt->execute([$month]);
        $revenue = $stmt->fetchColumn();
        
        $monthlyData[] = [
            'month' => $monthName,
            'members' => (int)$memberCount,
            'revenue' => (float)$revenue
        ];
    }
    return $monthlyData;
}
?>
