<?php
/**
 * DataTable API Endpoint
 * Handles all DataTable requests with filtering, search, and export
 */

// Define API access constant
define('API_ACCESS', true);

require_once '../config/database.php';
require_once '../services/DataTableService.php';

// Set JSON response headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Check authentication
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'admin') {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$pathInfo = isset($_SERVER['PATH_INFO']) ? trim($_SERVER['PATH_INFO'], '/') : '';
$pathParts = explode('/', $pathInfo);

try {
    switch ($method) {
        case 'GET':
            handleGetRequest($pathParts);
            break;
        case 'POST':
            handlePostRequest($pathParts);
            break;
        case 'PUT':
            handlePutRequest($pathParts);
            break;
        case 'DELETE':
            handleDeleteRequest($pathParts);
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    error_log("DataTable API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest($pathParts) {
    $tableName = $pathParts[0] ?? null;
    $action = $pathParts[1] ?? 'data';
    
    if (!$tableName) {
        http_response_code(400);
        echo json_encode(['error' => 'Table name required']);
        return;
    }
    
    $dataTable = new DataTableService($tableName);
    
    switch ($action) {
        case 'data':
            // Get DataTable data
            $params = $_GET;
            $response = $dataTable->getData($params);
            echo json_encode($response);
            break;
            
        case 'config':
            // Get table configuration
            $config = $dataTable->getTableConfig();
            echo json_encode($config);
            break;
            
        case 'filters':
            // Get filter options for a column
            $column = $_GET['column'] ?? null;
            if (!$column) {
                http_response_code(400);
                echo json_encode(['error' => 'Column parameter required']);
                return;
            }
            
            $options = $dataTable->getFilterOptions($column);
            echo json_encode($options);
            break;
            
        case 'export':
            // Export data
            $format = $_GET['format'] ?? 'csv';
            $filters = json_decode($_GET['filters'] ?? '[]', true);
            $columns = json_decode($_GET['columns'] ?? '[]', true);
            
            $dataTable->exportData($format, $filters, $columns);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Action not found']);
    }
}

function handlePostRequest($pathParts) {
    $tableName = $pathParts[0] ?? null;
    $action = $pathParts[1] ?? 'create';
    
    if (!$tableName) {
        http_response_code(400);
        echo json_encode(['error' => 'Table name required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    switch ($action) {
        case 'create':
            $result = createRecord($tableName, $input);
            echo json_encode($result);
            break;
            
        case 'bulk-action':
            $result = handleBulkAction($tableName, $input);
            echo json_encode($result);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Action not found']);
    }
}

function handlePutRequest($pathParts) {
    $tableName = $pathParts[0] ?? null;
    $recordId = $pathParts[1] ?? null;
    
    if (!$tableName || !$recordId) {
        http_response_code(400);
        echo json_encode(['error' => 'Table name and record ID required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    $result = updateRecord($tableName, $recordId, $input);
    echo json_encode($result);
}

function handleDeleteRequest($pathParts) {
    $tableName = $pathParts[0] ?? null;
    $recordId = $pathParts[1] ?? null;
    
    if (!$tableName || !$recordId) {
        http_response_code(400);
        echo json_encode(['error' => 'Table name and record ID required']);
        return;
    }
    
    $result = deleteRecord($tableName, $recordId);
    echo json_encode($result);
}

function createRecord($tableName, $data) {
    global $conn;
    
    try {
        // Validate table name
        $allowedTables = ['members', 'events', 'finance', 'expenses', 'admins', 'documents', 'notifications'];
        if (!in_array($tableName, $allowedTables)) {
            throw new Exception("Invalid table name");
        }
        
        // Prepare data for insertion
        $columns = [];
        $values = [];
        $types = '';
        
        foreach ($data as $column => $value) {
            $columns[] = $column;
            $values[] = $value;
            $types .= 's'; // Assuming string type for simplicity
        }
        
        // Build and execute query
        $sql = "INSERT INTO {$tableName} (" . implode(', ', $columns) . ") VALUES (" . str_repeat('?,', count($values) - 1) . "?)";
        $stmt = $conn->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }
        
        $stmt->bind_param($types, ...$values);
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        $insertId = $conn->insert_id;
        $stmt->close();
        
        // Log activity
        logActivity($_SESSION['user_id'], 'CREATE', "Created new {$tableName} record", $tableName, $insertId);
        
        return [
            'success' => true,
            'message' => 'Record created successfully',
            'id' => $insertId
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Failed to create record: ' . $e->getMessage()
        ];
    }
}

function updateRecord($tableName, $recordId, $data) {
    global $conn;
    
    try {
        // Validate table name
        $allowedTables = ['members', 'events', 'finance', 'expenses', 'admins', 'documents', 'notifications'];
        if (!in_array($tableName, $allowedTables)) {
            throw new Exception("Invalid table name");
        }
        
        // Prepare data for update
        $setParts = [];
        $values = [];
        $types = '';
        
        foreach ($data as $column => $value) {
            $setParts[] = "{$column} = ?";
            $values[] = $value;
            $types .= 's';
        }
        
        $values[] = $recordId;
        $types .= 'i';
        
        // Build and execute query
        $sql = "UPDATE {$tableName} SET " . implode(', ', $setParts) . " WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }
        
        $stmt->bind_param($types, ...$values);
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        $affectedRows = $stmt->affected_rows;
        $stmt->close();
        
        // Log activity
        logActivity($_SESSION['user_id'], 'UPDATE', "Updated {$tableName} record", $tableName, $recordId);
        
        return [
            'success' => true,
            'message' => 'Record updated successfully',
            'affected_rows' => $affectedRows
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Failed to update record: ' . $e->getMessage()
        ];
    }
}

function deleteRecord($tableName, $recordId) {
    global $conn;
    
    try {
        // Validate table name
        $allowedTables = ['members', 'events', 'finance', 'expenses', 'admins', 'documents', 'notifications'];
        if (!in_array($tableName, $allowedTables)) {
            throw new Exception("Invalid table name");
        }
        
        // Check if record exists
        $checkStmt = $conn->prepare("SELECT id FROM {$tableName} WHERE id = ?");
        $checkStmt->bind_param('i', $recordId);
        $checkStmt->execute();
        $checkResult = $checkStmt->get_result();
        
        if ($checkResult->num_rows === 0) {
            throw new Exception("Record not found");
        }
        
        $checkStmt->close();
        
        // Delete the record
        $deleteStmt = $conn->prepare("DELETE FROM {$tableName} WHERE id = ?");
        $deleteStmt->bind_param('i', $recordId);
        
        if (!$deleteStmt->execute()) {
            throw new Exception("Delete failed: " . $deleteStmt->error);
        }
        
        $affectedRows = $deleteStmt->affected_rows;
        $deleteStmt->close();
        
        // Log activity
        logActivity($_SESSION['user_id'], 'DELETE', "Deleted {$tableName} record", $tableName, $recordId);
        
        return [
            'success' => true,
            'message' => 'Record deleted successfully',
            'affected_rows' => $affectedRows
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Failed to delete record: ' . $e->getMessage()
        ];
    }
}

function handleBulkAction($tableName, $data) {
    $action = $data['action'] ?? null;
    $ids = $data['ids'] ?? [];
    
    if (!$action || empty($ids)) {
        return [
            'success' => false,
            'message' => 'Action and IDs required'
        ];
    }
    
    $results = [];
    $successCount = 0;
    
    foreach ($ids as $id) {
        switch ($action) {
            case 'delete':
                $result = deleteRecord($tableName, $id);
                break;
            case 'activate':
                $result = updateRecord($tableName, $id, ['status' => 'active']);
                break;
            case 'deactivate':
                $result = updateRecord($tableName, $id, ['status' => 'inactive']);
                break;
            default:
                $result = ['success' => false, 'message' => 'Unknown action'];
        }
        
        $results[] = $result;
        if ($result['success']) {
            $successCount++;
        }
    }
    
    return [
        'success' => $successCount > 0,
        'message' => "Bulk action completed: {$successCount} of " . count($ids) . " records processed successfully",
        'results' => $results
    ];
}

function getRecord($tableName, $recordId) {
    global $conn;
    
    try {
        $allowedTables = ['members', 'events', 'finance', 'expenses', 'admins', 'documents', 'notifications'];
        if (!in_array($tableName, $allowedTables)) {
            throw new Exception("Invalid table name");
        }
        
        $stmt = $conn->prepare("SELECT * FROM {$tableName} WHERE id = ?");
        $stmt->bind_param('i', $recordId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            throw new Exception("Record not found");
        }
        
        $record = $result->fetch_assoc();
        $stmt->close();
        
        return [
            'success' => true,
            'data' => $record
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Failed to get record: ' . $e->getMessage()
        ];
    }
}

function logActivity($userId, $action, $details, $entityType = null, $entityId = null) {
    global $conn;
    
    try {
        $stmt = $conn->prepare("INSERT INTO activity_logs (user_id, action, details, entity_type, entity_id, ip_address, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $stmt->bind_param('isssis', $userId, $action, $details, $entityType, $entityId, $ipAddress);
        $stmt->execute();
        $stmt->close();
    } catch (Exception $e) {
        error_log("Failed to log activity: " . $e->getMessage());
    }
}
?>
